% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summarize_PA.R
\name{summarize_PA}
\alias{summarize_PA}
\title{Compute physical activity summaries of minute level activity
data.}
\usage{
summarize_PA(
  acc,
  acc_ts,
  wear_flag,
  valid_day_flag,
  sedentary_thresh = 1853,
  subset_minutes = NULL,
  exclude_minutes = NULL,
  subset_weekdays = NULL,
  in_bed_time = NULL,
  out_bed_time = NULL,
  adjust_out_colnames = TRUE
)
}
\arguments{
\item{acc}{A numeric vector. A minute-level activity counts data vector.
It is assumed
to be in midnight-to-midnight format, meaning its vector length
is a multiple of number of minutes in a full day
(1440). See \code{arctools::midnight_to_midnight()}.}

\item{acc_ts}{A POSIXct vector. Time of activity data collection, corresponding to
\code{acc} in its original format (not: midnight-to-midnight).}

\item{wear_flag}{An integer vector. It has value \code{1} if a minute
belongs to a wear time-interval, value \code{0} if a minute
belongs to a non-wear time-interval, and value \code{NA} to denote minutes
before/after data collection started/finished.
See \code{arctools::get_wear_flag()}.}

\item{valid_day_flag}{An integer vector. It has value \code{1} if a minute
belongs to a valid day, and \code{0} otherwise. See
\code{arctools::get_valid_day_flag()}.}

\item{sedentary_thresh}{A numeric scalar. If an activity count value falls
below it then a corresponding minute is characterized as sedentary; otherwise,
a corresponding minute is characterized as active. Default is \code{1853}.}

\item{subset_minutes}{Integer vector. Contains subset of a day's minutes
within which activity summaries are to be computed. May take values from
\code{1} (day's minute from 00:00 to 00:01) to
\code{1440} (day's minute from 23:59 to 00:00). Default is \code{NULL}, i.e.
no subset used (all day's minutes are used).}

\item{exclude_minutes}{Integer vector. Contains subset of a day's minutes
to be excluded from activity summaries computation.
May take values from
\code{1} (day's minute from 00:00 to 00:01) to
\code{1440} (day's minute from 23:59 to 00:00). Default is \code{NULL}, i.e.
no minutes excluded (all day's minutes are used).}

\item{subset_weekdays}{Integer vector. Specfies days of a week within which
activity summaries are to be computed. Takes values between 1 (Sunday) to
7 (Saturday). Default is \code{NULL}, i.e.no subset used
(all days of a week are used).}

\item{in_bed_time}{A POSIXct vector. An estimated in-bed time start.
Together with a corresponding entry from \code{out_bed_time} vector,
it defines a
day-specific subset of "in bed time" minutes to be excluded from
activity summaries computation.  Default is \code{NULL}, i.e.
no minutes excluded.}

\item{out_bed_time}{A POSIXct vector. An estimated in-bed time end.
Together with a corresponding entry from \code{in_bed_time} vector,
it defines a
day-specific subset of "in bed time" minutes to be excluded from
activity summaries computation.  Default is \code{NULL}, i.e.
no minutes excluded.}

\item{adjust_out_colnames}{A logical scalar. Whether or not to
add an informative suffix to column names in the output data frame.
This may happen in case
any of the arguments:
\code{subset_minutes}, or
\code{exclude_minutes}, or
\code{in_bed_time} and \code{out_bed_time}
are set other than \code{NULL}.
Default is \code{TRUE}.}
}
\value{
A data frame with physical activity summaries of minute level activity
data. See README or vignette for summaries description.
}
\description{
Compute physical activity summaries of minute level activity
data.
}
\examples{
## Read exemplary data
fpath_i <- system.file("extdata", extdata_fnames[1], package = "arctools")
dat_i   <- as.data.frame(data.table::fread(fpath_i))
acc     <- dat_i$vectormagnitude
acc_ts  <- lubridate::ymd_hms(dat_i$timestamp)
## Get acc data vector in "midnight_to_midnight" format
acc <- midnight_to_midnight(acc, acc_ts)
## Get wear/non-wear flag
wear_flag <- get_wear_flag(acc)
## Get valid/non-valid day flag
valid_day_flag <- get_valid_day_flag(wear_flag)
## Impute missing data in acc data vector
acc_imputed <- impute_missing_data(acc, wear_flag, valid_day_flag)

## Example 1
## Summarize PA
summarize_PA(acc, acc_ts, wear_flag, valid_day_flag)

## Example 2
## Summarize PA within minutes range corresponding to 12:00 AM - 6:00 AM
subset_12am_6am <- 1 : (6 * 1440/24)
summarize_PA(acc, acc_ts, wear_flag, valid_day_flag, subset_minutes = subset_12am_6am)

## Example 3
## Summarize PA without (i.e., excluding) minutes range corresponding to 11:00 PM - 5:00 AM.
subset_11pm_5am <- c(
  (23 * 1440/24 + 1) : 1440,   ## 11:00 PM - midnight
  1 : (5 * 1440/24)            ## midnight - 5:00 AM
)
summarize_PA(acc, acc_ts, wear_flag, valid_day_flag, exclude_minutes = subset_11pm_5am)

}
