% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/equivalence_test.R
\name{equivalence_test}
\alias{equivalence_test}
\alias{equivalence_test.numeric}
\alias{equivalence_test.stanreg}
\alias{equivalence_test.brmsfit}
\title{Test for Practical Equivalence}
\usage{
equivalence_test(x, ...)

\method{equivalence_test}{numeric}(x, range = "default", ci = 0.95,
  verbose = TRUE, ...)

\method{equivalence_test}{stanreg}(x, range = "default", ci = 0.95,
  parameters = NULL, verbose = TRUE, ...)

\method{equivalence_test}{brmsfit}(x, range = "default", ci = 0.95,
  parameters = NULL, verbose = TRUE, ...)
}
\arguments{
\item{x}{Vector representing a posterior distribution. Can also be a \code{stanreg} or \code{brmsfit} model.}

\item{...}{Currently not used.}

\item{range}{ROPE's lower and higher bounds. Should be a vector of length two (e.g., \code{c(-0.1, 0.1)}) or \code{"default"}. If \code{"default"}, the range is set to \code{c(-0.1, 0.1)} if input is a vector, and \code{x +- 0.1*SD(response)} if a Bayesian model is provided.}

\item{ci}{The Credible Interval (CI) probability, corresponding to the proportion of HDI, to use.}

\item{verbose}{Toggle off warnings.}

\item{parameters}{Regular expression pattern that describes the parameters that
should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}
}
\value{
A data frame with following columns:
  \itemize{
    \item \code{Parameter} The model parameter(s), if \code{x} is a model-object. If \code{x} is a vector, this column is missing.
    \item \code{CI} The probability of the HDI.
    \item \code{ROPE_low}, \code{ROPE_high} The limits of the ROPE. These values are identical for all parameters.
    \item \code{ROPE_Percentage} The proportion of the HDI that lies inside the ROPE.
    \item \code{ROPE_Equivalence} The "test result", as character. Either "rejected", "accepted" or "undecided".
    \item \code{HDI_low} , \code{HDI_high} The lower and upper HDI limits for the parameters.
  }
}
\description{
Perform a \strong{Test for Practical Equivalence} based on the \emph{"HDI+ROPE decision rule"} (Kruschke, 2018) to check whether parameter values should be accepted or rejected against an explicitly formulated "null hypothesis" (\emph{i.e.}, a \link[=rope]{ROPE}).
}
\details{
Using the \link[=rope]{ROPE} and the \link[=hdi]{HDI}, Kruschke (2018)
  suggest using the percentage of the 95\% (or 90\%, considered more stable)
  \link[=hdi]{HDI} that falls within the ROPE as a decision rule. If the HDI
  is completely outside the ROPE, the "null hypothesis" for this parameter is
  "rejected". If the ROPE completely covers the HDI, i.e. all most credible
  values of a parameter are inside the region of practical equivalence, the
  null hypothesis is accepted. Else, it’s undecided whether to accept or
  reject the null hypothesis. If the full ROPE is used (\emph{i.e.}, 100\% of the
  HDI), then the null hypothesis is rejected or accepted if the percentage
  of the posterior within the ROPE is smaller than to 2.5\% or greater  than
  97.5\%. Desirable results are low proportions inside the ROPE  (the closer
  to zero the better) and the null hypothesis should be rejected.
  \cr \cr
  Some attention is required for finding suitable values for the ROPE limits
  (argument \code{range}). See 'Details' in \link{rope_range} for further
  information.
}
\note{
There is a \code{print()}-method with a \code{digits}-argument to control
  the amount of digits in the output, and there is a \code{plot()}-method
  to visualize the results from the equivalence-test (for models only).
}
\examples{
library(bayestestR)

equivalence_test(x = rnorm(1000, 0, 0.01), range = c(-0.1, 0.1))
equivalence_test(x = rnorm(1000, 0, 1), range = c(-0.1, 0.1))
equivalence_test(x = rnorm(1000, 1, 0.01), range = c(-0.1, 0.1))
equivalence_test(x = rnorm(1000, 1, 1), ci = c(.50, .99))

# print more digits
test <- equivalence_test(x = rnorm(1000, 1, 1), ci = c(.50, .99))
print(test, digits = 4)

\dontrun{
library(rstanarm)
model <- rstanarm::stan_glm(mpg ~ wt + cyl, data = mtcars)
equivalence_test(model)
equivalence_test(model, ci = c(.50, 1))

# plot result
test <- equivalence_test(model)
plot(test)

library(brms)
model <- brms::brm(mpg ~ wt + cyl, data = mtcars)
equivalence_test(model)
equivalence_test(model, ci = c(.50, .99))
}

}
\references{
Kruschke, J. K. (2018). Rejecting or accepting parameter values in Bayesian estimation. Advances in Methods and Practices in Psychological Science, 1(2), 270-280. \doi{10.1177/2515245918771304}.
}
