#' Extract index data
#'
#' Subsets area weights and extracts n values from JAGS samples
#'
#' @param jags_mod JAGS list generated by \code{run_model}
#'
#' @importFrom utils read.csv
#'
#' @return List of index values produced by JAGS samples
#'
#' @keywords internal
#' @export
#'

extract_index_data <- function(jags_mod = NULL,
                               alt_n = "n",
                               jags_data = NULL)
{
  # Read area weights
  stratify_by <- jags_mod$stratify_by
  all_area_weights <- utils::read.csv(system.file("area-weight", strata[[stratify_by]], package = "bbsBayes"))

  # Extract posterior data and other data from jags_mod
  n <- jags_mod$sims.list[[alt_n]]
  if (isTRUE(jags_mod$parallel))
  {
    bugs_data <- jags_mod$model$cluster1$data()
  }else
  {
    bugs_data <- jags_mod$model$data()
  }
  y_min <- bugs_data$ymin
  y_max <- bugs_data$ymax
  strat_list = unique(data.frame(strat_name = jags_mod$strat_name,
                          strat = bugs_data$strat,
                          stringsAsFactors = FALSE))
  strat_list = strat_list[order(strat_list$strat),]

  # Subset area weights based on strata used and ensure same order as JAGS
  strata_used <- strat_list$strat_name
  strata_num <- strat_list$strat
  area_weights <- all_area_weights[which(all_area_weights$region %in% strata_used), ]
  area_weights <- area_weights[ order(match(area_weights$region, strata_used)),]
  area_weights$num <- strata_num

  if(!is.null(jags_data)){
    original_data = get_prepared_data(jags_data = jags_data)
  }else{
    original_data = NULL
  }
  return(list(n = n,
              area_weights = area_weights,
              y_min = y_min,
              y_max = y_max,
              r_year = jags_mod$r_year,
              bugs_data = bugs_data,
              original_data = original_data))
}
