#' Obtain MCMC list from jagsUI object
#'
#' \code{get_mcmc_list} will return both the \code{mcmc.list} object
#'   and the \code{sims.list} object from jagsUI. \code{mcmc.list}
#'   is a list of the MCMC samples generated by the rjags library,
#'   and \code{sims.list} is a vectorized version of \code{mcmc.list}
#'   produced by the jagsUI library.
#'
#' @param jags_mod JAGS object returned by \code{run_model}
#'
#' @return List containing:
#' \item{mcmc_list}{MCMC samples produced by rjags}
#' \item{sims_list}{Vectorized posterior samples produced by jagsUI}
#'
#' @export
#'
#' @examples
#'
#' # Toy example with Pacific Wren sample data
#' # First, stratify the sample data
#'
#' strat_data <- stratify(by = "bbs_cws", sample_data = TRUE)
#'
#' # Prepare the stratified data for use in a JAGS model.
#' jags_data <- prepare_jags_data(strat_data = strat_data,
#'                                species_to_run = "Pacific Wren",
#'                                model = "firstdiff",
#'                                min_year = 2009,
#'                                max_year = 2018)
#'
#' # Now run a JAGS model. For the sake of speed, we've adjusted
#' #   some arguments so that the JAGS model will not run any
#' #   adaptation steps (n_adapt = 0), no burnin steps (n_burnin = 0),
#' #   only 50 iterations per chain (n_iter = 50), and will not
#' #   thin the chain (n_thin = 1). This will produce several convergence
#' #   warnings, but we can ignore them for the sake of this toy example.
#'
#' jags_mod <- run_model(jags_data = jags_data,
#'                       n_adapt = 0,
#'                       n_burnin = 0,
#'                       n_iter = 10,
#'                       n_thin = 1)
#'
#' # Now, obtain the MCMC list
#' mcmc_list <- get_mcmc_list(jags_mod = jags_mod)
#'

get_mcmc_list <- function(jags_mod = NULL)
{
  return(list(mcmc_list = jags_mod$samples,
              sims_list = jags_mod$sims.list))
}
