% edited for Unicode correction
\name{climateGraph}
\alias{climateGraph}
\title{climate graph after Walter and Lieth}
\usage{
climateGraph(temp, rain, main = "StatName\\n52\U{00B0}24' N / 12\U{00B0}58' E\\n42 m aSL",
  units = c("\U{00B0}C", "mm"), labs = substr(month.abb, 1, 1), textprop = 0.25,
  ylim = range(temp, rain/2), compress = FALSE, ticklab = -8:30 * 10,
  ticklin = -15:60 * 5, box = TRUE, mar = c(1.5, 2.3, 4.5, 0.2),
  keeppar = TRUE, colrain = "blue", coltemp = "red", lwd = 2,
  arghumi = NULL, argarid = NULL, argcomp = NULL, arggrid = NULL,
  argtext = NULL, ...)
}
\arguments{
\item{temp}{monthly temperature mean in degrees C}

\item{rain}{monthly rain sum in mm (12 values)}

\item{main}{location info as character string. can have \\n. 
DEFAULT: "StatName\\n52d 24' N / 12d 58' E\\n42 m aSL"}

\item{units}{units used for labelling. DEFAULT: c("d C", "mm")}

\item{labs}{labels for x axis. DEFAULT: J,F,M,A,M,J,J,A,S,O,N,D}

\item{textprop}{proportion of graphic that is used for writing the values 
in a table to the right. DEFAULT: 0.25}

\item{ylim}{limit for y axis in temp units. DEFAULT: range(temp, rain/2)}

\item{compress}{should rain>100 mm be compressed with adjusted labelling? 
(not recommended for casual visualization!). DEFAULT: FALSE}

\item{ticklab}{positions for vertical labelling. DEFAULT: -8:30*10}

\item{ticklin}{positions for horizontal line drawing. DEFAULT: -15:60*5}

\item{box}{draw box along outer margins of graph? DEFAULT: TRUE}

\item{mar}{plot margins. DEFAULT: c(1.5,2.3,4.5,0.2)}

\item{keeppar}{Keep the changed graphical parameters? DEFAULT: TRUE}

\item{colrain}{Color for rain line and axis labels. DEFAULT: "blue"}

\item{coltemp}{color for temperature line and axis labels. DEFAULT: "red"}

\item{lwd}{line width of actual temp and rain lines. DEFAULT: 2}

\item{arghumi}{List of arguments for humid \code{\link{polygon}}, 
like density, angle. DEFAULT: NULL (internal x,y, col, border)}

\item{argarid}{List of arguments for arid area. DEFAULT: NULL}

\item{argcomp}{List of arguments for compressed rainfall polygon. DEFAULT: NULL}

\item{arggrid}{List of arguments for bachground grid lines. DEFAULT: NULL}

\item{argtext}{List of arguments for text at right hand if textprop>0. DEFAULT: NULL}

\item{\dots}{further arguments passed to plot, like col.main}
}
\value{
None. Plots data and table.
}
\description{
Draw a climate diagramm by the standards of Walter and Lieth.
}
\examples{

temp <- c(-9.3,-8.2,-2.8,6.3,13.4,16.8,18.4,17,11.7,5.6,-1,-5.9)#
rain <- c(46,46,36,30,31,21,26,57,76,85,59,46)

climateGraph(temp, rain) 
climateGraph(temp, rain, textprop=0.6) 
climateGraph(temp, rain, mar=c(2,3,4,3), textprop=0) # no table written to the right
# vertical lines instead of filled polygon:
climateGraph(temp, rain, arghumi=list(density=15, angle=90)) 
# fill color for arid without transparency:
climateGraph(temp, rain, argarid=list(col="gold")) 
# for the Americans - axes should be different, though!:
climateGraph(temp, rain, units=c("\\U{00B0}F","in")) 

rain2 <- c(23, 11, 4, 2, 10, 53, 40, 15, 21, 25, 29, 22)
# fix ylim if you want to compare diagrams of different stations:
climateGraph(temp, rain2, ylim=c(-15, 50)) # works with two arid phases as well

op <- par(mfrow=c(2,1)) # mulipanel plot
climateGraph(temp, rain, argtext=list(cex=0.7))
climateGraph(temp, rain2, argtext=list(cex=0.7))
par(op)

rain <- c(54, 23, 5, 2, 5, 70, 181, 345, 265, 145, 105, 80) # with extrema
climateGraph(temp, rain) # August can be visually compared to June
climateGraph(temp, rain, compress=TRUE) 
# compressing extrema enables a better view of the temperature, 
# but heigths of rain cannot be visually compared anymore
climateGraph(temp, rain, compress=TRUE, ylim=c(-10, 90))
# needs ylim in linearly continued temp units
climateGraph(temp, rain, compress=TRUE, argcomp=list(density=30, col=6))

\dontrun{
pdf("ClimateGraph.pdf")
climateGraph(temp, rain, main="Another Station\\nlocated somewhere else")
dev.off()
system2("open", "ClimateGraph.pdf")
unlink("ClimateGraph.pdf")

# further German reading:
browseURL("http://www.klimadiagramme.de/all.html")


# Climate Graphs for the USA:
NOOAlink <- "http://www1.ncdc.noaa.gov/pub/data/normals/1981-2010/"
browseURL(NOOAlink)
# Find your Station here:
browseURL(paste0(NOOAlink,"/station-inventories/allstations.txt"))

# Data from Roseburg, Oregon:
download.file(destfile="Roseburg.txt", url=paste0("http://www1.ncdc.noaa.gov/",
          "pub/data/normals/1981-2010/products/station/USC00357331.normals.txt"))
RT <- read.table(file="Roseburg.txt", skip=11, nrows=1, as.is=TRUE)[1,-1]
RT <- ( as.numeric(substr(RT,1,3))/10   - 32) * 5/9     # converted to degrees C
RP <- read.table(file="Roseburg.txt", skip=580, nrows=1, as.is=TRUE)[1,-1]
RP <-  as.numeric(substr(RP,1,nchar(RP)-1))/100*25.4
meta <- read.table(file="Roseburg.txt", nrows=5, as.is=TRUE, sep=":")
meta <- paste(meta[1,2], paste(meta[3:4 ,2], collapse=" /"), meta[5,2], sep="\\n")
unlink("Roseburg.txt")

climateGraph(RT, RP, main=meta)
climateGraph(RT, RP, main=meta, compress=TRUE)


# Climate Graphs for Germany:
browseURL("https://github.com/brry/rdwd#rdwd") 
instGit("brry/rdwd")
link <- rdwd::selectDWD("Potsdam", res="monthly", var="kl", per="h")
clim <- rdwd::dataDWD(link, dir=tempdir())
dates <- strptime(clim$MESS_DATUM_BEGINN, "\%Y\%m\%d")
temp <- tapply(clim$LUFTTEMPERATUR, INDEX=format(dates, "\%m"), FUN=mean, na.rm=FALSE)
precsums <- tapply(clim$NIEDERSCHLAGSHOEHE, INDEX=format(dates, "\%Y-\%m"), FUN=sum)
eachmonth <- format(strptime(paste(names(precsums),"01"), "\%Y-\%m \%d"),"\%m")
prec <- tapply(precsums, eachmonth, FUN=mean)
meta <- paste("Potsdam\\n", paste(range(dates, na.rm=TRUE), collapse=" to "), "\\n", sep="")

climateGraph(temp, prec, main=meta, ylim=c(-2, 45))
# Add Quartiles (as in boxplots): numerically sorted, 50\% of the data lie inbetween
TQ <- tapply(clim$LUFTTEMPERATUR, INDEX=format(dates, "\%m"), FUN=quantile)
TQ <- sapply(TQ, I)
arrows(x0=1:12, y0=TQ["25\%",], y1=TQ["75\%",], angle=90, code=3, col=2, len=0.1)
#
PQ <- tapply(precsums, eachmonth, FUN=quantile)
PQ <- sapply(PQ, I)
arrows(x0=1:12, y0=PQ["25\%",]/2, y1=PQ["75\%",]/2, angle=90, code=3, col=4, len=0, lwd=3, lend=1)
mtext("IQR shown als lines", col=8, at=6.5, line=0.7, cex=1.2, font=2)


# Comparison to diagram in climatol
install.packages("climatol")
help(package="climatol")
library(climatol)
data(datcli)
diagwl(datcli,est="Example station",alt=100,per="1961-90",mlab="en")

}

}
\author{
Berry Boessenkool, \email{berry-b@gmx.de}, June 2013
}
\references{
Heinrich Walter, Helmut Lieth: Klimadiagramm-Weltatlas. Gustav Fischer Verlag, Jena 1967\cr 
Examples:\cr
\url{http://www.hoelzel.at/_verlag/geojournal/archiv/klima/2006_01/lieth.gif}\cr
\url{http://www.hoelzel.at/_verlag/geojournal/archiv/klima/istanbul/istanbul200.gif}\cr
\url{http://www.ipb.uni-tuebingen.de/kurs/comp/1_excel2007/1_pic/2007diagramm_verbund02.jpg}\cr
\url{http://www.zivatar.hu/felhotar/albums/userpics/wldp.png}
}
\seealso{
\code{diagwl} in package \code{climatol}
}
\keyword{hplot}

