% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/boxplotcluster.R
\name{boxplotcluster}
\alias{boxplotcluster}
\title{Boxplot Cluster Function}
\usage{
boxplotcluster(
  x,
  aggl.meth = "ward.D2",
  part = NULL,
  cex.dndr.lab = 0.75,
  cex.sil.lab = 0.75,
  oneplot = TRUE
)
}
\arguments{
\item{x}{A dataframe or matrix where each column represents a data series to be clustered.}

\item{aggl.meth}{A string specifying the agglomeration method to be used in hierarchical
clustering. Defaults to "ward.D2". For other methods see \code{\link[stats]{hclust}}.}

\item{part}{An optional integer specifying the desired number of clusters. If not provided,
the function will select the number of clusters that maximizes the average silhouette width.}

\item{cex.dndr.lab}{A numeric specifying the character expansion factor for the labels in the
dendrogram plot. Defaults to 0.75.}

\item{cex.sil.lab}{A numeric specifying the character expansion factor for the labels in the
silhouette plot. Defaults to 0.75.}

\item{oneplot}{TRUE (default) or FALSE if the user wants or does not want the plots to be visualized
in a single window.}
}
\value{
A dist object representing the distance matrix used for clustering.
}
\description{
The function \code{boxplotcluster} implements a special clustering method
based on boxplot statistics. Following Arroyo-Maté-Roque (2006), the function calculates
the distance between columns of the dataset using the generalized Minkowski metric as described
by Ichino and Yaguchi (1994). The distance measure gives more weight to differences between
quartiles than to differences between extremes, making it less sensitive to outliers. Further,
the function calculates the silhouette width for different numbers of clusters (Rousseeuw 1987)
and selects the number of clusters that maximizes the average silhouette width
(unless a specific number of clusters is provided by the user).
}
\details{
The function first calculates the pairwise distance between each column of the input
dataset using the Ichino-Yaguchi dissimilarity measure (equations 7 and 8 in Arroyo-Maté-Roque (2006)).
The distance between A and B is defined as:\cr

\eqn{(0.5 * (abs(m1 - m2) + 2 * abs(q1 - q2) + 2 * abs(Me1 - Me2) + 2 * abs(Q1 - Q2) + abs(M1 - M2))) /4}\cr

  where\cr

  m1 <- min(A)\cr
  m2 <- min(B)\cr
  q1 <- quantile(A, probs = 0.25)\cr
  q2 <- quantile(B, probs = 0.25)\cr
  Q1 <- quantile(A, probs = 0.75)\cr
  Q2 <- quantile(B, probs = 0.75)\cr
  M1 <- max(A)\cr
  M2 <- max(B)\cr
  Me1 <- median(A)\cr
  Me2 <- median(B)\cr

The distance matrix is then used to perform hierarchical clustering. Also, the function calculates the
silhouette width for different numbers of clusters and selects the number of clusters
that maximizes the average silhouette width (unless a specific number of clusters is provided by the user).\cr

The silhouette method allows to measure how 'good' is the selected cluster solution. If the parameter \code{part}
is left empty (default), an optimal cluster solution is obtained. The optimal partition is selected via an iterative
procedure which identifies at which cluster solution the highest average silhouette width is achieved.
If a user-defined partition is needed, the user can input the desired number of clusters using the parameter \code{part}.
In either case, an additional plot is returned besides the cluster dendrogram and the silhouette plot; it displays a
scatterplot in which the cluster solution (x-axis) is plotted against the average silhouette width (y-axis).
A black dot represents the partition selected either by the iterative procedure or by the user.\cr

In summary, the function generates a series of plots to visualize the results:\cr

(a) boxplots colored by cluster membership,\cr
(b) a dendrogram (where clusters are indicated by rectangles whose color is consistent with the color assigned to the
boxplot in the previous plot),\cr
(c) a silhouette plot, and\cr
(d) a plot of the average silhouette width vs. the number of clusters.\cr

The silhouette plot is obtained from the \code{silhouette()} function out from the \code{cluster} package.
For a detailed description of the silhouette plot, its rationale, and its interpretation, see Rousseeuw 1987.
}
\examples{
  # Create a toy dataset
  df <- data.frame(
a = rnorm(30, mean = 30, sd = 5),
b = rnorm(30, mean = 40, sd = 5),
c = rnorm(30, mean = 20, sd = 5),
d = rnorm(30, mean = 25, sd = 5),
e = rnorm(30, mean = 50, sd = 5),
f = rnorm(30, mean = 10, sd = 5),
g = rnorm(30, mean = 100, sd = 5),
h = rnorm(30, mean = 20, sd = 5),
i = rnorm(30, mean = 40, sd = 5),
l = rnorm(30, mean = 35, sd = 5),
m = rnorm(30, mean = 35, sd = 5),
n = rnorm(30, mean = 70, sd = 5),
o = rnorm(30, mean = 20, sd = 5),
p = rnorm(30, mean = 70, sd = 5),
q = rnorm(30, mean = 90, sd = 5)
)

# Run the function
result <- boxplotcluster(df)

# Same as above, but selecting a 4-cluster solution
result <- boxplotcluster(df, part=4)

}
\references{
Rousseeuw, P J. (1987). Silhouettes: A graphical aid to the interpretation and validation of cluster analysis,
Journal of Computational and Applied Mathematics 20, 53-65.

Ichino, M., & Yaguchi, H. (1994). Generalized Minkowski Metrics for Mixed
Feature-Type Data Analysis. IEEE Transactions on Systems, Man, and Cybernetics, 24(4), 698-708.

Arroyo, J., Maté, C., & Roque, A. M-S. (2006). Hierarchical Clustering for Boxplot Variables.
In Studies in Classification, Data Analysis, and Knowledge Organization (pp. 59–66).
Springer Berlin Heidelberg.
}
\seealso{
\code{\link[cluster]{silhouette}}, \code{\link[stats]{hclust}}
}
