% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stan_fit.R
\name{stan_fit}
\alias{stan_fit}
\title{Bayesian Stan fit to 13C Breath Data}
\usage{
stan_fit(
  data,
  dose = 100,
  sample_minutes = 15,
  student_t_df = 10,
  chains = 2,
  iter = 1000,
  model = "breath_test_1",
  seed = 4711
)
}
\arguments{
\item{data}{Data frame or tibble as created by \code{\link[breathtestcore]{cleanup_data}},
with mandatory columns \code{patient_id, group, minute} and \code{pdr}.
It is recommended to run all data through \code{\link[breathtestcore]{cleanup_data}} which
will insert dummy columns for \code{patient_id} and \code{minute} if the
data are distinct, and report an error if not. Since the Bayesian method
is stabilized by priors, it is possible to fit single curves.}

\item{dose}{Dose of acetate or octanoate. Currently, only one common dose
for all records is supported.}

\item{sample_minutes}{If mean sampling interval is < sampleMinutes, data are subsampled
using a spline algorithm}

\item{student_t_df}{When student_t_df < 10, the student distribution is used to
model the residuals. Recommended values to model typical outliers are from 3 to 6.
When student_t_df >= 10, the normal distribution is used.}

\item{chains}{Number of chains for Stan}

\item{iter}{Number of iterations for each Stan chain}

\item{model}{Name of model; use \code{names(stanmodels)} for other models.}

\item{seed}{Optional seed for rstan}
}
\value{
A list of classes "breathteststanfit" and "breathtestfit" with elements
\itemize{
  \item {\code{coef} Estimated parameters as data frame in a key-value format with
   columns \code{patient_id, group, parameter, method} and \code{value}.
   Has an attribute AIC.}
   \item {\code{data}  The effectively analyzed data. If density of points
   is too high, e.g. with BreathId devices, data are subsampled before fitting.}
   \item {\code{stan_fit} The Stan fit for use with \code{shinystan::launch_shiny}
   or extraction of chains. }
}
}
\description{
Fits exponential beta curves to 13C breath test series data using
Bayesian Stan methods. See
\url{https://menne-biomed.de/blog/breath-test-stan} for a comparision between
single curve, mixed-model population and Bayesian methods.
}
\examples{
library(breathtestcore)
suppressPackageStartupMessages(library(dplyr))
d = breathtestcore::simulate_breathtest_data(n_records = 3) # default 3 records
data = breathtestcore::cleanup_data(d$data)
# Use more than 80 iterations and 4 chains for serious fits
fit = stan_fit(data, chains = 1, iter = 80)
plot(fit) # calls plot.breathtestfit
# Extract coefficients and compare these with those
# used to generate the data
options(digits = 2)
cf = coef(fit)
cf \%>\%
  filter(grepl("m|k|beta", parameter )) \%>\%
  select(-method, -group) \%>\%
  tidyr::spread(parameter, value) \%>\%
  inner_join(d$record, by = "patient_id") \%>\%
  select(patient_id, m_in = m.y, m_out = m.x,
         beta_in = beta.y, beta_out = beta.x,
         k_in = k.y, k_out = k.x)
# For a detailed analysis of the fit, use the shinystan library
\donttest{
library(shinystan)
# launch_shinystan(fit$stan_fit)
}
# The following plots are somewhat degenerate because
# of the few iterations in stan_fit
suppressPackageStartupMessages(library(rstan))
stan_plot(fit$stan_fit, pars = c("beta[1]","beta[2]","beta[3]"))
stan_plot(fit$stan_fit, pars = c("k[1]","k[2]","k[3]"))
stan_plot(fit$stan_fit, pars = c("m[1]","m[2]","m[3]"))

}
\seealso{
Base methods \code{coef, plot, print}; methods from package
 \code{broom: tidy, augment}.
}
