\name{train}
\alias{train}
\alias{train.default}
\alias{train.formula}

\newcommand{\cpkg}{\href{http://CRAN.R-project.org/package=#1}{\pkg{#1}}}
\newcommand{\bpkg}{\href{http://www.bioconductor.org/packages/release/bioc/html/#1.html}{\pkg{#1}}}

\title{Fit Predictive Models over Different Tuning Parameters}
\description{
  This function sets up a grid of tuning parameters for a number of classification and regression routines,
  fits each model and calculates a resampling based performance measure. 
}
\usage{
train(x, ...)

\method{train}{default}(x, y, 
      method = "rf",  
      preProcess = NULL,
      ..., 
      weights = NULL,
      metric = ifelse(is.factor(y), "Accuracy", "RMSE"),   
      maximize = ifelse(metric == "RMSE", FALSE, TRUE),
      trControl = trainControl(), 
      tuneGrid = NULL, 
      tuneLength = 3)

\method{train}{formula}(form, data, ..., weights, subset, na.action, contrasts = NULL)
}
\arguments{
  \item{x}{a data frame containing training data where samples are in rows and features are in columns.}
  \item{y}{a numeric or factor vector containing the outcome for each sample.}
\item{form}{A formula of the form \code{y ~ x1 + x2 + ...}}
\item{data}{Data frame from which variables specified in \code{formula} are preferentially to be taken.}
  \item{weights}{a numeric vector of case weights. This argument will only affect models that allow case weights.}
\item{subset}{An index vector specifying the cases to be used in the training sample. (NOTE: If given, this argument must be named.)}
\item{na.action}{A function to specify the action to be taken if NAs are found. The default action is for the procedure to fail. An alternative is na.omit, which leads to rejection of cases with missing values on any required variable. (NOTE: If given, this argument must be named.)}
\item{contrasts}{a list of contrasts to be used for some or all of the factors appearing as variables in the model formula.}
  \item{method}{a string specifying which classification or regression model to use. Possible values are found using \code{names(getModelInfo())}. See \url{http://caret.r-forge.r-project.org/bytag.html}. A list of funciotns can also be passed for a custom model function. See \url{http://caret.r-forge.r-project.org/custom_models.html} for details.}  
  \item{\ldots}{ arguments passed to the classification or regression routine (such as \code{\link[randomForest]{randomForest}}). Errors will occur if values
         for tuning parameters are passed here.} 
\item{preProcess}{a string vector that defines an pre-processing of the predictor data. Current possibilities are "BoxCox", "YeoJohnson", "expoTrans", "center", "scale", "range", "knnImpute", "bagImpute", "medianImpute", "pca", "ica" and "spatialSign". The default is no pre-processing. See \code{\link{preProcess}} and \code{\link{trainControl}} on the procedures and how to adjust them.} 
  \item{metric}{a string that specifies what summary metric will be used to select the optimal model. By default, possible values are "RMSE" and "Rsquared" for regression and "Accuracy" and "Kappa" for classification. If custom performance metrics are used (via the \code{summaryFunction} argument in \code{\link{trainControl}}, the value of \code{metric} should match one of the arguments. If it does not, a warning is issued and the first metric given by the \code{summaryFunction} is used. (NOTE: If given, this argument must be named.)}         
  \item{maximize}{a logical: should the metric be maximized or minimized?}
  \item{trControl}{a list of values that define how this function acts. See \code{\link{trainControl}} and  \url{http://caret.r-forge.r-project.org/training.html#custom}. (NOTE: If given, this argument must be named.)}
  \item{tuneGrid}{a data frame with possible tuning values. The columns are named the same as the tuning parameters. Use \code{\link{getModelInfo}} to get a list of tuning parameters for each model or see \url{http://caret.r-forge.r-project.org/modelList.html}. (NOTE: If given, this argument must be named.)}  
  \item{tuneLength}{an integer denoting the number of levels for each tuning parameters that should be
         generated by \code{\link{train}}. (NOTE: If given, this argument must be named.)}
}
\details{
\code{train} can be used to tune models by picking the complexity parameters that are associated with the optimal resampling statistics. For particular model, a grid of parameters (if any) is created and the model is trained on slightly different data for each candidate combination of tuning parameters. Across each data set, the performance of held-out samples is calculated and the mean and standard deviation is summarized for each combination. The combination with the optimal resampling statistic is chosen as the final model and the entire training set is used to fit a final model.

More details on this function can be found at \url{http://caret.r-forge.r-project.org/training.html}.

A variety of models are currently available and are enumerated by tag (i.e. their model characteristics) at  \url{http://caret.r-forge.r-project.org/bytag.html}.

}



\value{
  A list is returned of class \code{train} containing:
  \item{method }{the chosen model.}  
  \item{modelType }{an identifier of the model type.}  
  \item{results }{a data frame the training error rate and values of the tuning parameters.}
  \item{bestTune }{a data frame with the final parameters.}  

  \item{call }{the (matched) function call with dots expanded}
  \item{dots}{a list containing any ... values passed to the original call}
  \item{metric}{a string that specifies what summary metric will be used to select the optimal model.}  
  \item{control}{the list of control parameters.}
  \item{preProcess }{either \code{NULL} or an object of class \code{\link{preProcess}}}
   \item{finalModel}{an fit object using the best parameters}
   \item{trainingData}{a data frame}
   \item{resample}{A data frame with columns for each performance
     metric. Each row corresponds to each resample. If leave-one-out
     cross-validation or out-of-bag estimation methods are requested,
     this will be \code{NULL}. The \code{returnResamp} argument of \code{\link{trainControl}}
     controls how much of the resampled results are saved.}
  \item{perfNames}{a character vector of performance metrics that are produced by the summary function}
  \item{maximize}{a logical recycled  from the function arguments.}
  \item{yLimits}{the range of the training set outcomes.}
  \item{times}{a list of execution times: \code{everything} is for the entire call to \code{train}, \code{final} for the final model fit and, optionally, \code{prediction} for the time to predict new samples (see \code{\link{trainControl}})}
}


\author{Max Kuhn (the guts of \code{train.formula} were based on Ripley's \code{nnet.formula})}

\references{ \url{http://caret.r-forge.r-project.org/training.html}

Kuhn (2008), ``Building Predictive Models in R Using the caret'' (\url{http://www.jstatsoft.org/v28/i05/})}

\seealso{\code{\link{trainControl}}, \code{\link{update.train}},
\code{\link{modelLookup}},
\code{\link{createFolds}}}

\examples{
\dontrun{

#######################################
## Classification Example

data(iris)
TrainData <- iris[,1:4]
TrainClasses <- iris[,5]

knnFit1 <- train(TrainData, TrainClasses,
                 method = "knn",
                 preProcess = c("center", "scale"),
                 tuneLength = 10,
                 trControl = trainControl(method = "cv"))

knnFit2 <- train(TrainData, TrainClasses,
                 method = "knn",
                 preProcess = c("center", "scale"),
                 tuneLength = 10, 
                 trControl = trainControl(method = "boot"))


library(MASS)
nnetFit <- train(TrainData, TrainClasses,
                 method = "nnet",
                 preProcess = "range", 
                 tuneLength = 2,
                 trace = FALSE,
                 maxit = 100)

#######################################
## Regression Example

library(mlbench)
data(BostonHousing)

lmFit <- train(medv ~ . + rm:lstat,
               data = BostonHousing, 
               "lm")

library(rpart)
rpartFit <- train(medv ~ .,
                  data = BostonHousing,
                  "rpart",
                  tuneLength = 9)

#######################################
## Example with a custom metric

madSummary <- function (data,
                        lev = NULL,
                        model = NULL) 
{
  out <- mad(data$obs - data$pred, 
             na.rm = TRUE)  
  names(out) <- "MAD"
  out
}

robustControl <- trainControl(summaryFunction = madSummary)
marsGrid <- expand.grid(degree = 1, nprune = (1:10) * 2)

earthFit <- train(medv ~ .,
                  data = BostonHousing, 
                  "earth",
                  tuneGrid = marsGrid,
                  metric = "MAD",
                  maximize = FALSE,
                  trControl = robustControl)

#######################################
## Parallel Processing Example via multicore package

## library(doMC)
## registerDoMC(2)

## NOTE: don't run models form RWeka when using
### multicore. The session will crash.

## The code for train() does not change:
set.seed(1)
usingMC <-  train(medv ~ .,
                  data = BostonHousing, 
                  "glmboost")

## or use:
## library(doMPI) or 
## library(doSMP) and so on

}

}
\keyword{models}
