% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estQ.R
\name{estQ}
\alias{estQ}
\title{Empirical Q-matrix estimation}
\usage{
estQ(
  r,
  K,
  n.obs = NULL,
  criterion = "row",
  boot = FALSE,
  efa.args = list(cor = "tet", rotation = "oblimin", fm = "uls"),
  boot.args = list(N = 0.8, R = 100, verbose = TRUE, seed = NULL)
)
}
\arguments{
\item{r}{A correlation matrix or raw data (\code{matrix} or \code{data.frame}). If a correlation matrix is used, it must have dimensions \emph{J} items × \emph{J} items. Please note that tetrachoric or polychoric correlations should be used when working with dichotomous or polytomous items, respectively. If raw data is used, it must have dimensions \emph{N} individuals × \emph{J} items. Missing values need to be coded as \code{NA}.}

\item{K}{Number of attributes to use.}

\item{n.obs}{Number of individuals if \code{r} is a correlation matrix. If \code{n.obs} is provided, \code{r} will be treated as a correlation matrix. Use \code{NULL} if \code{r} is raw data. The default is \code{NULL}.}

\item{criterion}{Dichotomization criterion to transform the factor loading matrix into the Q-matrix. The possible options include \code{"row"} (for row means), \code{"col"} (for column means), \code{"loaddiff"} (for the procedure based on loading differences), or a value between 0 and 1 (for a specific threshold). The default is \code{"row"}.}

\item{boot}{Apply the bagging bootstrap implementation? Only available if \code{r} is raw data. If \code{FALSE}, the EFA will be applied once using the whole sample size. If \code{TRUE}, several EFAs will be applied with different subsamples; the estimated Q-matrix will be dichotomized from the bootstrapped Q-matrix, but the EFA fit indices, factor loadings, and communalities will be computed from the EFA with the whole sample size. The default is \code{FALSE}.}

\item{efa.args}{A list of arguments for the EFA estimation:
\describe{
\item{\code{cor}}{Type of correlations to use. It includes \code{"cor"} (for Pearson correlations) and \code{"tet"} (for tetrachoric/polychoric correlations), among others. See \code{fa} function from the \code{psych} R package for additional details. The default is \code{"tet"}.}
\item{\code{rotation}}{Rotation procedure to use. It includes \code{"oblimin"}, \code{"varimax"}, and \code{"promax"}, among others. An oblique rotation procedure is usually recommended. See \code{fa} function from the \code{psych} R package for additional details. The default is \code{"oblimin"}.}
\item{\code{fm}}{Factoring method to use. It includes \code{"uls"} (for unweighted least squares), \code{"ml"} (for maximum likelihood), and \code{"wls"} (for weighted least squares), among others. See \code{fa} function from the \code{psych} R package for additional details. The default is \code{"uls"}.}
}}

\item{boot.args}{A list of arguments for the bagging bootstrap implementation (ignored if \code{boot = FALSE}):
\describe{
\item{\code{N}}{Sample size (or proportion of the total sample size, if lower than 1) to use in each bootstrap replication. The default is .8.}
\item{\code{R}}{Number of bootstrap replications. The default is 100.}
\item{\code{verbose}}{Show progress? The default is \code{TRUE}.}
\item{\code{seed}}{A seed for obtaining consistent results. If \code{NULL}, no seed is used. The default is \code{NULL}.}
}}
}
\value{
\code{estQ} returns an object of class \code{estQ}.
\describe{
\item{\code{est.Q}}{Estimated Q-matrix (\code{matrix}).}
\item{\code{efa.loads}}{Factor loading matrix (\code{matrix}).}
\item{\code{efa.comm}}{EFA communalities (\code{vector}).}
\item{\code{efa.fit}}{EFA model fit indices (\code{vector}).}
\item{\code{boot.Q}}{Bagging bootstrap Q-matrix before dichotomization. Only if \code{boot = TRUE} (\code{matrix}).}
\item{\code{is.Qid}}{Q-matrix identifiability information (\code{list}).}
\item{\code{specifications}}{Function call specifications (\code{list}).}
}
}
\description{
Empirical Q-matrix estimation based on the \emph{discrete factor loading} method (Wang, Song, & Ding, 2018) as used in Nájera, Abad, and Sorrel (2021).
Apart from the conventional dichotomization criteria, the procedure based on loading differences described in Garcia-Garzon, Abad, and Garrido (2018) is also available.
Furthermore, the bagging bootstrap implementation (Xu & Shang, 2018) can be applied; it is recommended when working with small sample sizes.
The \code{psych} package (Revelle, 2020) is used for estimating the required exploratory factor analysis (EFA).
}
\examples{
library(GDINA)
dat <- sim30GDINA$simdat
Q <- sim30GDINA$simQ

#------------------------------
# Using default specifications
#------------------------------
sugQ1 <- estQ(r = dat, K = 5) # Estimate Q-matrix
sugQ1$est.Q <- orderQ(sugQ1$est.Q, Q)$order.Q # Reorder Q-matrix attributes
mean(sugQ1$est.Q == Q) # Check similarity with the generating Q-matrix

#------------------------------------
# Using the bagging bootstrap method
#------------------------------------
# In boot.args argument, R >= 100 is recommended (R = 20 is here used for illustration purposes)
sugQ2 <- estQ(r = dat, K = 5, boot = TRUE, boot.args = list(R = 20, seed = 123)) # Estimate Q-matrix
sugQ2$est.Q <- orderQ(sugQ2$est.Q, Q)$order.Q # Reorder Q-matrix attributes
sugQ2$boot.Q # Proportion of replicas a q-entry was specified in the estimated Q-matrix
mean(sugQ2$est.Q == Q) # Check similarity with the generating Q-matrix
}
\references{
Garcia-Garzon, E., Abad, F. J., & Garrido, L. E. (2018). Improving bi-factor exploratory modelling: Empirical target rotation based on loading differences. \emph{Methodology}, \emph{15}, 45–55. https://doi.org/10.1027/1614-2241/a000163

Nájera, P., Abad, F. J., & Sorrel, M. A. (2021). Determining the number of attributes in cognitive diagnosis modeling. \emph{Frontiers in Psychology}, \emph{12}:614470. https://doi.org/10.3389/fpsyg.2021.614470

Revelle, W. (2019). \emph{psych: Procedures for Psychological, Psychometric, and Personality Research}. R package version 1.9.12. https://CRAN.R-project.org/package=psych.

Wang, W., Song, L., & Ding, S. (2018). An exploratory discrete factor loading method for Q-matrix specification in cognitive diagnosis models. In: M. Wilberg, S. Culpepper, R. Janssen, J. Gonzalez, & D. Molenaar (Eds.), \emph{Quantitative Psychology. IMPS 2017. Springer Proceedings in Mathematics & Statistics} (Vol. 233, pp. 351–362). Springer.

Xu, G., & Shang, Z. (2018). Identifying latent structures in restricted latent class models. \emph{Journal of the American Statistical Association}, \emph{113}, 1284–1295. https://doi.org/10.1080/01621459.2017.1340889
}
\author{
{Pablo Nájera, Universidad Autónoma de Madrid}
}
