#' Create Hex Stickers for R Packages
#'
#' @title Create Hex Stickers for R Packages
#' @description Generates hexadecimal stickers for R packages using a provided
#'    description and package name. The stickers are generated by the OpenAI API.
#'    This is an experimental function.
#' @param description A string. This is the description of the R package for which the stickers will be generated.
#' @param package_name A string. This is the name of the R package for which the stickers will be generated.
#' @param n An integer. This is the number of stickers to generate. The default value is 3.
#' @importFrom assertthat assert_that
#' @importFrom httr POST add_headers content
#' @importFrom jsonlite toJSON
#' @importFrom purrr map
#' @importFrom hexSticker sticker
#' @return A list of EBImage objects. These are the hexadecimal stickers generated for the R package.
#' @export createHEX4R
#' @author Satoshi Kume
#'
#' @examples
#' \dontrun{
#' # Define a description for the R package
#' description = "hatAI4R provides a seamless interface for integrating OpenAI and other APIs into R."
#' package_name = "R package"
#'
#' # Generate hex stickers
#' res <- createHEX4R(description, package_name)
#'
#' # Display the stickers
#' Display(res)
#' }

createHEX4R <- function(description, package_name, n = 3){
  # Verify the types of the input parameters
  assertthat::assert_that(assertthat::is.string(description))
  assertthat::assert_that(assertthat::is.string(package_name))
  assertthat::assert_that(assertthat::is.count(n))

  # Create the prompt
  pr <- paste0('Generate one hexadecimal sticker as called "hex sticker" for the R package at the center of the output image using the following description: ',
               description)

  # Generate the image
  res <-  generateImage4R(content = pr,
                          n = n,
                          size = "256x256",
                          response_format = "url",
                          Output_image = FALSE,
                          SaveImg = FALSE,
                          api_key = Sys.getenv("OPENAI_API_KEY"))

  # Create the hex stickers
  img <- list()
  for(k in seq_len(n)){
    suppressMessages(
    img[[k]] <-  hexSticker::sticker(res[k],
                                     package = package_name,
                                     p_y = 1.5, p_size=12,
                                     s_x=1, s_y=.8, s_width=.5)
    )
  }

  # Return the result
  return(img)
}
