## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----echo=FALSE---------------------------------------------------------------
options(digits=2)

## ----setup,message=FALSE, warning=FALSE---------------------------------------
library(clusterMI)

## ----message=FALSE------------------------------------------------------------
require(stargazer)
set.seed(123456)
data(wine)
stargazer(wine, type = "text")
table(wine$cult)

## -----------------------------------------------------------------------------
ref <- wine$cult # "True" partition
nb.clust <- 3 # Number of clusters
wine.na <- wine
wine.na$cult <- NULL # Remove the reference partition
wine.na <- as.matrix(wine.na)
wine.na[sample(seq(length(wine.na)), size = ceiling(length(wine.na)/3))] <- NA

## -----------------------------------------------------------------------------
# proportion of missing values
colMeans(is.na(wine.na))

# proportion of incomplete individuals
mean(apply(is.na(wine.na), 1, any))

## ----warning=FALSE------------------------------------------------------------
m <- 20 # Number of imputed data sets
res.imp.JM <- imputedata(data.na = wine.na,
                         nb.clust = nb.clust,
                         m = m)

## ----warning=FALSE, eval=FALSE------------------------------------------------
#  res.imp <- imputedata(data.na = wine.na,
#                           method = "JM-DP",
#                           nb.clust = nb.clust,
#                           m = m)

## ----warning=FALSE------------------------------------------------------------
maxit <- 20 # Number of iterations for FCS imputation, should be larger in practice
res.imp.FCS <- imputedata(data.na = wine.na,
                      method = "FCS-homo",
                      nb.clust = nb.clust,
                      maxit = maxit,
                      m = m)

## ----eval=FALSE---------------------------------------------------------------
#  imputedata(data.na = wine.na,
#             method = "FCS-homo",
#             nb.clust = nb.clust,
#             maxit = maxit,
#             m = m,
#             method.mice = "norm")

## ----conv, fig.height = 7, fig.width = 7, fig.align = "center", results='hide'----
choosemaxit(res.imp.FCS)

## ----eval = FALSE-------------------------------------------------------------
#  res.imp <- imputedata(data.na = wine.na,
#                        method = "FCS-homo",
#                        nb.clust = nb.clust,
#                        maxit = 100,
#                        m = m,
#                        verbose = TRUE)
#  choosemaxit(res.imp, layout= c(2,2))

## ----varselecho, eval=FALSE---------------------------------------------------
#  nnodes <- 2
#  # Number of CPU cores used for parallel computation.
#  # Use parallel::detectCores() to choose an appropriate number
#  
#  # variable selection to impute the "alco" variable
#  B <- 50 # number of bootstrap subsets, should be increased in practice
#  res.varsel <- varselbest(res.imputedata = res.imp.FCS, B = B, listvar = "alco",
#                           nnodes = nnodes, graph = FALSE)
#  
#  res.varsel$predictormatrix["alco", ]

## ----varsel, eval=TRUE, echo=FALSE--------------------------------------------
nnodes <- 2
# Number of CPU cores used for parallel computation.
# Use parallel::detectCores() to choose an appropriate number

# variable selection to impute the "alco" variable
B <- 50 # number of bootstrap subsets, should be increased in practice

# res.chooseB<-chooseB(res.varsel)
# sink(file = "C:/Users/vince/OneDrive - LECNAM/Recherche/MI_clustering/Rpackage/vignettes/sink/chooseB.txt");dput(res.chooseB);sink()
# 
# res.varsel.light<-res.varsel
# res.varsel.light$res.varsel$alco$res.detail<- rep(NA,length(res.varsel.light$res.varsel$alco$res.detail))
# str(res.varsel.light$res.varsel$alco$res.detail)
# res.varsel.light$call$res.imputedata<-NULL
# res.varsel.light$res.varsel$alco$res$listvarblock<-NULL
# res.varsel.light$res.varsel$alco$call$knockoff.arg<-NULL
# res.varsel.light$res.varsel$alco$call$glmnet.arg<-NULL
# res.varsel.light$res.varsel$alco$call$stepwise.arg<-NULL
# sink(file = "C:/Users/vince/OneDrive - LECNAM/Recherche/MI_clustering/Rpackage/vignettes/sink/resvarsel.txt");dput(res.varsel.light);sink()

res.varsel <- list(predictormatrix = structure(c(0, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 0, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 0, 1, 
1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0), dim = c(13L, 13L), dimnames = list(
    c("alco", "malic", "ash", "alca", "mg", "phe", "fla", "nfla", 
    "pro", "col", "hue", "ratio", "prol"), c("alco", "malic", 
    "ash", "alca", "mg", "phe", "fla", "nfla", "pro", "col", 
    "hue", "ratio", "prol"))), res.varsel = list(alco = list(
    res = list(garde = c(malic = 23, ash = 22, alca = 20, mg = 21, 
    phe = 20, fla = 21, nfla = 21, pro = 20, col = 19, hue = 21, 
    ratio = 20, prol = 22), effectif = c(malic = 6, ash = 8, 
    alca = 9, mg = 7, phe = 2, fla = 7, nfla = 3, pro = 9, col = 18, 
    hue = 15, ratio = 12, prol = 8), proportion = c(malic = 0.260869565217391, 
    ash = 0.363636363636364, alca = 0.45, mg = 0.333333333333333, 
    phe = 0.1, fla = 0.333333333333333, nfla = 0.142857142857143, 
    pro = 0.45, col = 0.947368421052632, hue = 0.714285714285714, 
    ratio = 0.6, prol = 0.363636363636364), selection = c("ash", 
    "alca", "mg", "fla", "pro", "col", "hue", "ratio", "prol"
    ), failure = c(malic = 0, ash = 0, alca = 0, mg = 0, phe = 0, 
    fla = 0, nfla = 0, pro = 0, col = 0, hue = 0, ratio = 0, 
    prol = 0)), res.detail = c(NA, NA, NA, NA, NA, NA, NA, NA, 
    NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, 
    NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, 
    NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA, NA), call = list(
        nnodes = 2, X = structure(c(1.71, NA, NA, NA, 1.87, 2.15, 
        1.35, NA, 1.73, 1.87, NA, 1.92, NA, 3.1, 3.8, NA, 1.6, 
        1.81, NA, NA, 1.9, 1.5, NA, 1.83, 1.53, 1.65, NA, 1.71, 
        NA, NA, 3.98, NA, 4.04, 1.68, NA, NA, 1.67, 1.7, 1.97, 
        0.94, NA, 1.36, NA, NA, 1.21, NA, NA, 1.51, 1.67, 1.09, 
        1.88, NA, 3.87, NA, NA, 1.13, NA, NA, 1.61, NA, NA, 1.51, 
        NA, NA, NA, 1.41, NA, 2.08, NA, NA, NA, 1.29, NA, 2.68, 
        1.39, NA, NA, 2.4, 4.43, NA, 4.31, 2.16, 2.13, 4.3, 1.35, 
        2.99, 3.55, 1.24, NA, 5.51, NA, 2.81, 2.56, NA, NA, 3.88, 
        4.61, 3.24, 2.67, NA, 5.19, 4.12, 3.03, 1.68, 1.67, NA, 
        NA, 3.45, 2.76, NA, 2.58, 4.6, 2.39, NA, 3.91, NA, 2.59, 
        4.1, 2.43, 2.14, 2.67, NA, 2.45, 2.61, NA, 2.41, 2.39, 
        2.38, 2.7, 2.72, 2.62, 2.56, 2.65, NA, 2.52, 2.61, NA, 
        NA, NA, NA, 2.36, NA, 2.7, 2.55, 2.51, NA, 2.12, 2.59, 
        2.29, 2.1, 2.44, 2.12, 2.04, NA, NA, 2.3, 2.68, 1.36, 
        NA, NA, 2.16, 2.53, 2.56, NA, 1.75, 2.67, 2.6, 2.3, NA, 
        NA, 2.4, 2, NA, 2.51, 2.32, 2.58, NA, 2.3, 2.32, NA, 
        2.26, 2.28, 2.74, 1.98, NA, 1.7, NA, NA, 2.28, 1.94, 
        2.7, 2.92, 2.5, 2.2, 1.99, 2.42, NA, 2.13, 2.39, 2.17, 
        NA, 2.38, NA, 2.4, 2.36, 2.25, 2.54, 2.64, 2.61, 2.7, 
        2.35, 2.72, 2.35, 2.2, 2.48, NA, 2.48, 2.28, 2.32, 2.38, 
        NA, NA, 2.64, 2.54, NA, 2.35, 2.3, NA, 2.69, NA, 2.28, 
        NA, 2.48, 2.26, 2.37, 2.74, 15.6, NA, NA, 21, NA, 17.6, 
        16, 16, 11.4, NA, NA, NA, 20, 15.2, 18.6, NA, 17.8, 20, 
        16.1, 17, NA, NA, 19.1, NA, NA, NA, 13.2, 16.2, 18.8, 
        NA, NA, 17, NA, NA, 12.4, NA, NA, 16.3, 16.8, 10.6, NA, 
        16.8, 19, 19, 18.1, NA, 16.8, NA, 30, 21, NA, 18, NA, 
        19, NA, 24, 22.5, 18, 22.8, NA, NA, 22, NA, 18, 21.5, 
        16, 18, 17.5, 18.5, 20.5, 22.5, 19, 20, NA, NA, 21.5, 
        20.8, NA, NA, 21.5, 21, 21, 28.5, 22, NA, 20, 21.5, NA, 
        NA, 25, NA, 21, NA, 23.5, 20, NA, NA, NA, 22, 18.5, 22, 
        19.5, NA, 25, NA, NA, NA, 18.5, NA, 22.5, 24.5, 25, 19.5, 
        NA, 23, 20, NA, 24.5, 127, 100, 101, NA, 96, 121, NA, 
        89, 91, 102, 112, 120, NA, 116, 102, NA, 95, NA, 93, 
        NA, 107, 101, 106, 104, NA, NA, NA, 117, NA, 101, NA, 
        107, 111, 101, 92, 111, NA, 118, 102, 88, 101, 100, 87, 
        104, 98, NA, 151, 86, 139, 101, NA, 112, NA, 86, NA, 
        78, 85, NA, 90, 70, 81, NA, NA, NA, 134, 85, NA, 97, 
        88, 85, 84, 92, 94, 103, 84, 85, 86, 96, 102, 86, NA, 
        85, 92, 80, 122, NA, 106, NA, 89, NA, 101, NA, 89, 97, 
        NA, 112, NA, 92, NA, 98, NA, 89, 97, 98, 89, NA, 106, 
        106, 90, 88, 105, 112, 86, 91, 102, NA, 120, NA, 2.8, 
        2.65, 2.8, 2.8, 2.5, 2.6, 2.98, 2.6, 3.1, NA, 2.85, 2.8, 
        NA, NA, 2.41, 2.61, 2.48, 2.53, NA, 2.4, 2.95, 3, NA, 
        2.42, 2.95, 2.45, NA, 3.15, 2.45, 3.25, 2.64, 3, 2.85, 
        3.1, 2.72, 3.88, NA, 3.2, 3, 1.98, 2.05, 2.02, 3.5, 1.89, 
        NA, 2.53, NA, 2.95, NA, NA, NA, NA, 2.83, NA, 2.2, NA, 
        1.65, NA, NA, 2.2, 1.6, 1.45, 1.38, 3.02, NA, 2.55, NA, 
        NA, NA, 2.2, NA, 2.36, 2.74, 1.75, 2.56, 2.46, NA, 2.9, 
        NA, NA, 2.86, NA, 2.13, 2.1, 1.51, NA, 1.7, NA, 1.38, 
        NA, NA, NA, 1.4, NA, 2, 1.38, 1.7, 1.93, 1.48, NA, NA, 
        1.8, 1.9, NA, NA, 1.83, NA, 1.39, 1.35, NA, 1.55, NA, 
        1.39, 1.68, NA, NA, 1.65, 2.05, NA, 2.76, NA, NA, 2.52, 
        2.51, NA, 2.76, NA, 3.64, 2.91, 3.14, 3.4, NA, 2.41, 
        2.88, 2.37, 2.61, 2.94, NA, 2.97, 3.25, 3.19, 2.69, NA, 
        2.43, 3.04, 3.29, 2.68, 3.56, 2.63, 3, 2.65, NA, NA, 
        3.74, 2.9, NA, 3.23, 0.57, NA, 1.41, NA, 1.75, 2.65, 
        1.3, NA, 2.86, 2.89, 2.14, 1.57, NA, NA, 2.26, 2.53, 
        1.58, 1.59, 2.21, 1.69, NA, NA, 1.25, 1.46, NA, 0.99, 
        NA, 2.99, 2.17, 1.36, 1.92, 1.76, NA, 2.92, 2.03, 2.29, 
        2.17, 1.6, NA, NA, NA, 3.03, 2.65, 2.24, 1.75, NA, NA, 
        1.2, 0.58, 0.47, NA, 0.6, 0.5, NA, 0.52, 0.8, NA, 0.65, 
        0.76, 1.36, 0.83, 0.63, NA, 0.58, 1.31, 1.1, NA, 0.6, 
        NA, 0.68, 0.47, 0.84, 0.96, NA, 0.7, NA, 0.69, 0.68, 
        0.76, 0.28, NA, 0.3, 0.39, NA, 0.31, 0.22, 0.29, 0.43, 
        0.29, 0.3, NA, NA, 0.17, 0.25, 0.27, 0.26, NA, 0.34, 
        0.27, NA, 0.29, NA, 0.42, NA, NA, 0.2, 0.34, NA, NA, 
        0.32, 0.28, 0.3, 0.21, 0.17, 0.32, 0.21, 0.26, NA, 0.28, 
        NA, 0.53, NA, 0.45, 0.37, NA, NA, NA, 0.21, 0.13, 0.34, 
        0.43, 0.43, 0.3, NA, 0.4, 0.61, 0.22, NA, NA, NA, 0.5, 
        NA, 0.17, NA, NA, NA, NA, 0.29, NA, 0.48, 0.39, 0.29, 
        NA, NA, 0.52, 0.3, 0.32, 0.43, 0.3, NA, NA, NA, NA, NA, 
        NA, 0.17, 0.6, 0.53, 0.63, 0.53, 0.53, NA, NA, NA, NA, 
        0.47, 0.45, NA, 0.61, NA, NA, 0.63, 0.53, 0.52, 0.5, 
        0.6, 0.4, 0.41, NA, 0.39, NA, 0.48, NA, 0.43, 0.43, 0.53, 
        NA, 2.29, 1.28, 2.81, 1.82, 1.98, 1.25, 1.85, 1.81, NA, 
        2.96, 1.46, NA, NA, 1.66, NA, 1.69, 1.46, 1.66, 1.45, 
        1.35, 1.76, 2.38, 1.95, NA, NA, 1.44, NA, NA, NA, NA, 
        NA, 2.03, NA, 2.14, NA, 1.87, 1.62, NA, 1.66, 0.42, NA, 
        NA, 1.87, 1.03, NA, NA, 2.5, 1.87, NA, NA, 1.15, NA, 
        1.95, 1.43, NA, NA, 1.62, NA, 1.56, 1.38, 1.64, 1.63, 
        NA, 1.35, 1.56, 1.77, 2.81, 1.4, 1.35, NA, 1.63, 2.08, 
        2.49, NA, 1.04, NA, NA, 1.83, 1.71, NA, 2.91, 1.35, NA, 
        NA, 0.94, NA, 0.84, 1.25, 0.8, 1.1, NA, 0.75, NA, 0.55, 
        NA, 1.14, 0.86, 1.25, 1.26, NA, 1.55, 1.56, 1.14, 2.7, 
        2.29, NA, NA, 0.94, NA, 1.15, 1.54, 1.11, 0.64, 1.24, 
        1.41, 1.35, 1.46, 1.35, 5.64, 4.38, 5.68, NA, NA, 5.05, 
        7.22, 5.6, NA, 7.5, NA, NA, NA, 5.1, 4.5, NA, 3.93, 3.52, 
        4.8, 3.95, 4.5, 5.7, NA, NA, 5.4, 4.25, 5.1, 6.13, 4.28, 
        5.43, 4.36, 5.04, 5.24, 6.1, 7.2, NA, 5.85, NA, NA, NA, 
        3.27, 5.75, 4.45, 2.95, 4.6, 3.17, 2.85, 3.38, NA, 3.21, 
        3.8, NA, NA, 2.5, 3.9, NA, 4.8, 3.05, 2.45, 1.74, 2.4, 
        3.6, 3.05, 3.25, NA, 2.9, NA, NA, NA, 2.94, 3.3, 2.7, 
        NA, NA, 2.9, 1.9, NA, 3.25, NA, NA, 2.8, NA, NA, NA, 
        NA, NA, 5, 5.45, NA, 5, 4.92, NA, NA, 4.35, 4.4, 8.21, 
        NA, 8.42, NA, 10.52, 7.9, NA, 7.5, 13, NA, NA, 5.58, 
        NA, NA, 6.62, NA, 8.5, NA, 9.7, 7.3, NA, 9.3, 9.2, 1.04, 
        1.05, NA, NA, 1.02, NA, NA, 1.15, 1.25, 1.2, 1.28, NA, 
        1.13, NA, NA, 1.11, 1.09, 1.12, NA, 1.02, NA, 1.19, 1.09, 
        NA, 1.25, NA, NA, 0.95, 0.91, 0.88, 0.82, NA, 0.87, NA, 
        1.12, NA, NA, 0.94, 1.07, NA, 1.25, 0.98, 1.22, NA, NA, 
        1.02, 1.28, 1.36, 1.31, NA, 1.23, 0.96, 1.19, 1.38, NA, 
        1.31, 0.84, NA, NA, 1.07, 1.08, 1.05, NA, NA, NA, NA, 
        1.42, 1.27, NA, 1.04, NA, 0.86, NA, NA, 0.93, 1.71, 0.95, 
        0.8, 0.92, 0.73, NA, 0.86, 0.97, 0.79, NA, 0.74, 0.78, 
        0.75, 0.75, 0.82, NA, NA, NA, 0.89, NA, 0.65, 0.54, 0.55, 
        0.48, 0.56, 0.6, 0.57, 0.67, 0.57, NA, 0.96, 0.87, 0.68, 
        0.7, 0.78, 0.74, 0.67, NA, NA, 0.7, 0.59, 0.6, 0.61, 
        3.92, 3.4, 3.17, 2.93, 3.58, 3.58, 3.55, 2.9, NA, 3, 
        NA, 2.65, NA, 3.36, 3.52, 4, 3.63, NA, NA, 2.77, NA, 
        2.71, 2.88, 2.87, 3, NA, NA, 3.38, 3, NA, 3, 3.35, 3.33, 
        3.33, NA, 3.26, 3.2, NA, 2.84, 1.82, NA, 1.59, 2.87, 
        NA, 2.3, NA, NA, NA, 3.5, 3.13, 2.14, 2.52, NA, NA, 3.14, 
        2.72, 2.01, NA, NA, 3.21, NA, 2.65, NA, NA, NA, 2.74, 
        2.83, 2.96, 2.77, 3.57, 2.42, 3.02, 3.26, 2.5, 3.19, 
        2.87, 3.33, 3.39, 3.12, NA, 3.64, NA, NA, NA, NA, 1.42, 
        1.29, 1.51, 1.27, 1.69, 2.15, NA, NA, NA, 2.05, 2, NA, 
        1.62, 1.47, 1.51, 1.48, 1.64, 1.73, 1.96, 1.78, NA, NA, 
        1.75, 1.68, 1.75, NA, NA, NA, NA, NA, 1.56, 1.62, 1.6, 
        1065, 1050, NA, 735, 1290, 1295, 1045, 1320, NA, 1547, 
        1310, 1280, NA, 845, 770, 1035, 1015, 845, 1195, 1285, 
        915, 1285, NA, NA, 1235, NA, 760, 795, 1035, 1095, 680, 
        885, 1080, 985, 1150, 1190, 1060, 970, 1270, NA, NA, 
        NA, NA, NA, NA, NA, 718, 410, NA, 886, NA, NA, 463, 278, 
        714, NA, 515, NA, NA, 625, 480, 450, 495, 345, 625, 428, 
        406, NA, 562, 672, NA, 312, 680, NA, 385, NA, NA, NA, 
        365, 380, NA, 378, 466, 580, NA, NA, 600, NA, 720, 515, 
        590, 600, NA, 520, 550, NA, NA, NA, 480, 675, NA, 480, 
        880, 660, NA, 680, 570, NA, 615, NA, NA, NA, 470, NA, 
        NA, 835, 840, NA), dim = c(118L, 12L), dimnames = list(
            NULL, c("V1", "V2", "V3", "V4", "V5", "V6", "V7", 
            "V8", "V9", "V10", "V11", "V12"))), Y = c(14.23, 
        13.2, 13.16, 13.24, 14.39, 14.06, 13.86, 13.75, 14.75, 
        14.38, 13.63, 14.3, 13.83, 13.64, 12.93, 13.71, 12.85, 
        13.5, 13.39, 13.3, 13.87, 13.73, 13.58, 13.68, 13.76, 
        13.05, 14.22, 13.56, 13.41, 13.88, 13.24, 13.05, 14.21, 
        13.9, 13.05, 13.82, 13.74, 14.22, 13.29, 12.37, 12.33, 
        12.64, 12.37, 12.17, 12.37, 13.34, 12.21, 13.86, 12.99, 
        11.96, 11.66, 11.84, 12.7, 12, 12.72, 12.08, 13.05, 11.84, 
        12.16, 12.08, 12.08, 12, 12.69, 11.62, 11.81, 12.29, 
        12.29, 12.08, 12.6, 12.51, 12.72, 12.22, 11.61, 11.76, 
        12.08, 11.03, 11.82, 11.45, 12.42, 13.05, 11.87, 12.07, 
        11.79, 12.04, 12.86, 12.88, 12.7, 12.51, 12.25, 12.53, 
        12.84, 12.93, 13.36, 13.52, 13.62, 12.25, 12.87, 13.32, 
        12.79, 13.23, 13.17, 13.84, 12.45, 14.34, 13.48, 13.69, 
        12.85, 12.96, 13.78, 13.73, 13.58, 13.4, 12.77, 14.16, 
        13.4, 13.27, 13.17, 14.13), B = 50, path.outfile = NULL, 
        methods = "knockoff", sizeblock = 5, printflag = FALSE, 
        r = c(alco = 0.3), seed = 1234567, nb.clust = 3, modelNames = NULL))), 
    proportion = structure(c(0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
    1, 1, 0.260869565217391, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
    1, 0.363636363636364, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
    0.45, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0.333333333333333, 
    1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0.1, 1, 1, 1, 1, 0, 1, 
    1, 1, 1, 1, 1, 1, 0.333333333333333, 1, 1, 1, 1, 1, 0, 1, 
    1, 1, 1, 1, 1, 0.142857142857143, 1, 1, 1, 1, 1, 1, 0, 1, 
    1, 1, 1, 1, 0.45, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 0.947368421052632, 
    1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0.714285714285714, 1, 
    1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0.6, 1, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 0, 1, 0.363636363636364, 1, 1, 1, 1, 1, 1, 1, 1, 
    1, 1, 1, 0), dim = c(13L, 13L), dimnames = list(c("alco", 
    "malic", "ash", "alca", "mg", "phe", "fla", "nfla", "pro", 
    "col", "hue", "ratio", "prol"), c("alco", "malic", "ash", 
    "alca", "mg", "phe", "fla", "nfla", "pro", "col", "hue", 
    "ratio", "prol"))), call = list(data.na = NULL, listvar = "alco", 
        nb.clust = NULL, nnodes = 2, sizeblock = 5, method.select = "knockoff", 
        B = 50, r = 0.3, graph = FALSE, printflag = TRUE, path.outfile = NULL, 
        mar = c(2.1, 4.1, 2.1, 0.6), cex.names = 0.7, modelNames = NULL))


res.varsel$predictormatrix["alco", ]

## ----eval=FALSE---------------------------------------------------------------
#  # multiple imputation with the new model
#  res.imp.select <- imputedata(data.na = wine.na,
#                               method = "FCS-homo",
#                               nb.clust = nb.clust,
#                               maxit = maxit,
#                               m = m,
#                               predictmat = res.varsel$predictormatrix)

## ----eval=FALSE---------------------------------------------------------------
#  varselbest(res.imputedata = res.imp.FCS, B = B, nnodes = nnodes) # (time consuming)

## ----eval=FALSE---------------------------------------------------------------
#  res.B <- chooseB(res.varsel)

## ----convb,fig.height = 4, fig.width = 4, fig.align = "center",echo=FALSE-----
res.chooseB <- list(alco = structure(c(1, 1, 0.5, 0.5, 0.333333333333333, 0.333333333333333, 
0.25, 0.25, 0.25, 0.2, 0.166666666666667, 0.166666666666667, 
0.166666666666667, 0.142857142857143, 0.125, 0.125, 0.125, 0.125, 
0.125, 0.111111111111111, 0.1, 0.1, 0.1, 0.0909090909090909, 
0.0833333333333333, 0.0833333333333333, 0.0833333333333333, 0.0833333333333333, 
0.0833333333333333, 0.153846153846154, 0.214285714285714, 0.214285714285714, 
0.266666666666667, 0.266666666666667, 0.266666666666667, 0.25, 
0.235294117647059, 0.235294117647059, 0.235294117647059, 0.277777777777778, 
0.277777777777778, 0.263157894736842, 0.263157894736842, 0.25, 
0.25, 0.238095238095238, 0.227272727272727, 0.227272727272727, 
0.260869565217391, 0.260869565217391, 0, 0, 0, 0, 0, 0.5, 0.333333333333333, 
0.333333333333333, 0.25, 0.25, 0.25, 0.2, 0.2, 0.333333333333333, 
0.285714285714286, 0.285714285714286, 0.285714285714286, 0.25, 
0.25, 0.25, 0.25, 0.333333333333333, 0.3, 0.3, 0.363636363636364, 
0.363636363636364, 0.363636363636364, 0.333333333333333, 0.333333333333333, 
0.384615384615385, 0.384615384615385, 0.428571428571429, 0.4, 
0.4, 0.375, 0.375, 0.352941176470588, 0.352941176470588, 0.352941176470588, 
0.352941176470588, 0.388888888888889, 0.388888888888889, 0.388888888888889, 
0.368421052631579, 0.35, 0.35, 0.380952380952381, 0.380952380952381, 
0.380952380952381, 0.363636363636364, 1, 1, 0.5, 0.5, 0.5, 0.5, 
0.5, 0.333333333333333, 0.333333333333333, 0.5, 0.4, 0.4, 0.4, 
0.5, 0.5, 0.428571428571429, 0.428571428571429, 0.428571428571429, 
0.428571428571429, 0.428571428571429, 0.428571428571429, 0.5, 
0.444444444444444, 0.444444444444444, 0.444444444444444, 0.444444444444444, 
0.444444444444444, 0.4, 0.363636363636364, 0.363636363636364, 
0.363636363636364, 0.416666666666667, 0.416666666666667, 0.384615384615385, 
0.357142857142857, 0.357142857142857, 0.333333333333333, 0.333333333333333, 
0.3125, 0.3125, 0.352941176470588, 0.352941176470588, 0.352941176470588, 
0.352941176470588, 0.352941176470588, 0.388888888888889, 0.421052631578947, 
0.421052631578947, 0.421052631578947, 0.45, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0.25, 0.2, 0.2, 0.166666666666667, 0.166666666666667, 
0.166666666666667, 0.285714285714286, 0.375, 0.375, 0.375, 0.333333333333333, 
0.4, 0.4, 0.4, 0.363636363636364, 0.363636363636364, 0.333333333333333, 
0.333333333333333, 0.333333333333333, 0.333333333333333, 0.384615384615385, 
0.384615384615385, 0.357142857142857, 0.357142857142857, 0.357142857142857, 
0.357142857142857, 0.333333333333333, 0.3125, 0.3125, 0.294117647058824, 
0.294117647058824, 0.333333333333333, 0.333333333333333, 0.315789473684211, 
0.315789473684211, 0.3, 0.3, 0.3, 0.333333333333333, 0.333333333333333, 
0.333333333333333, 0, 0, 0, 0, 0, 0, 0, 0, 0.2, 0.2, 0.2, 0.2, 
0.166666666666667, 0.166666666666667, 0.166666666666667, 0.166666666666667, 
0.166666666666667, 0.142857142857143, 0.125, 0.125, 0.125, 0.111111111111111, 
0.111111111111111, 0.1, 0.1, 0.1, 0.1, 0.0909090909090909, 0.0909090909090909, 
0.0909090909090909, 0.0833333333333333, 0.0833333333333333, 0.0833333333333333, 
0.0769230769230769, 0.0769230769230769, 0.0769230769230769, 0.0769230769230769, 
0.0714285714285714, 0.0714285714285714, 0.0666666666666667, 0.125, 
0.125, 0.117647058823529, 0.117647058823529, 0.111111111111111, 
0.111111111111111, 0.111111111111111, 0.105263157894737, 0.105263157894737, 
0.1, 0, 0, 0, 0, 0, 0, 0, 0, 0.25, 0.25, 0.25, 0.4, 0.333333333333333, 
0.333333333333333, 0.428571428571429, 0.428571428571429, 0.428571428571429, 
0.375, 0.333333333333333, 0.333333333333333, 0.4, 0.4, 0.363636363636364, 
0.363636363636364, 0.363636363636364, 0.333333333333333, 0.307692307692308, 
0.307692307692308, 0.285714285714286, 0.285714285714286, 0.285714285714286, 
0.285714285714286, 0.333333333333333, 0.333333333333333, 0.333333333333333, 
0.3125, 0.3125, 0.294117647058824, 0.294117647058824, 0.294117647058824, 
0.294117647058824, 0.294117647058824, 0.277777777777778, 0.277777777777778, 
0.315789473684211, 0.315789473684211, 0.35, 0.35, 0.333333333333333, 
0.333333333333333, 0, 1, 0.5, 0.5, 0.333333333333333, 0.333333333333333, 
0.25, 0.25, 0.25, 0.25, 0.2, 0.2, 0.166666666666667, 0.166666666666667, 
0.142857142857143, 0.142857142857143, 0.125, 0.125, 0.125, 0.111111111111111, 
0.1, 0.1, 0.1, 0.0909090909090909, 0.0909090909090909, 0.0833333333333333, 
0.0769230769230769, 0.0769230769230769, 0.0769230769230769, 0.0769230769230769, 
0.0714285714285714, 0.0714285714285714, 0.0714285714285714, 0.0714285714285714, 
0.0666666666666667, 0.0666666666666667, 0.0666666666666667, 0.0666666666666667, 
0.0666666666666667, 0.0625, 0.0625, 0.0588235294117647, 0.0588235294117647, 
0.111111111111111, 0.157894736842105, 0.157894736842105, 0.15, 
0.15, 0.15, 0.142857142857143, 0, 1, 0.5, 0.5, 0.5, 0.5, 0.5, 
0.5, 0.5, 0.666666666666667, 0.666666666666667, 0.75, 0.6, 0.6, 
0.6, 0.5, 0.5, 0.428571428571429, 0.375, 0.375, 0.375, 0.375, 
0.333333333333333, 0.333333333333333, 0.333333333333333, 0.3, 
0.363636363636364, 0.363636363636364, 0.333333333333333, 0.333333333333333, 
0.307692307692308, 0.307692307692308, 0.357142857142857, 0.357142857142857, 
0.4, 0.4, 0.4, 0.4, 0.4, 0.4375, 0.4375, 0.470588235294118, 0.470588235294118, 
0.470588235294118, 0.470588235294118, 0.444444444444444, 0.444444444444444, 
0.473684210526316, 0.45, 0.45, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 0.875, 0.875, 0.888888888888889, 0.888888888888889, 
0.888888888888889, 0.888888888888889, 0.9, 0.9, 0.9, 0.909090909090909, 
0.916666666666667, 0.916666666666667, 0.916666666666667, 0.923076923076923, 
0.923076923076923, 0.928571428571429, 0.928571428571429, 0.928571428571429, 
0.928571428571429, 0.933333333333333, 0.9375, 0.9375, 0.9375, 
0.9375, 0.941176470588235, 0.941176470588235, 0.941176470588235, 
0.944444444444444, 0.944444444444444, 0.944444444444444, 0.947368421052632, 
0.947368421052632, 0, 0, 0, 0, 0, 0, 0.333333333333333, 0.333333333333333, 
0.5, 0.5, 0.5, 0.6, 0.6, 0.6, 0.666666666666667, 0.666666666666667, 
0.714285714285714, 0.714285714285714, 0.714285714285714, 0.625, 
0.625, 0.666666666666667, 0.666666666666667, 0.7, 0.7, 0.727272727272727, 
0.75, 0.75, 0.769230769230769, 0.769230769230769, 0.785714285714286, 
0.785714285714286, 0.785714285714286, 0.733333333333333, 0.75, 
0.75, 0.75, 0.705882352941177, 0.666666666666667, 0.666666666666667, 
0.666666666666667, 0.684210526315789, 0.684210526315789, 0.7, 
0.7, 0.7, 0.7, 0.7, 0.7, 0.714285714285714, 1, 1, 1, 0.5, 0.666666666666667, 
0.666666666666667, 0.666666666666667, 0.5, 0.5, 0.5, 0.6, 0.6, 
0.6, 0.6, 0.6, 0.6, 0.5, 0.5, 0.5, 0.428571428571429, 0.428571428571429, 
0.428571428571429, 0.5, 0.5, 0.555555555555556, 0.555555555555556, 
0.6, 0.6, 0.6, 0.636363636363636, 0.636363636363636, 0.636363636363636, 
0.666666666666667, 0.666666666666667, 0.666666666666667, 0.615384615384615, 
0.642857142857143, 0.642857142857143, 0.642857142857143, 0.666666666666667, 
0.666666666666667, 0.6875, 0.647058823529412, 0.647058823529412, 
0.647058823529412, 0.611111111111111, 0.611111111111111, 0.631578947368421, 
0.6, 0.6, 1, 1, 1, 0.5, 0.5, 0.333333333333333, 0.333333333333333, 
0.25, 0.2, 0.2, 0.2, 0.333333333333333, 0.333333333333333, 0.285714285714286, 
0.285714285714286, 0.25, 0.25, 0.222222222222222, 0.2, 0.2, 0.2, 
0.272727272727273, 0.272727272727273, 0.272727272727273, 0.25, 
0.25, 0.25, 0.230769230769231, 0.230769230769231, 0.285714285714286, 
0.285714285714286, 0.333333333333333, 0.333333333333333, 0.3125, 
0.3125, 0.294117647058824, 0.294117647058824, 0.277777777777778, 
0.263157894736842, 0.263157894736842, 0.3, 0.3, 0.3, 0.333333333333333, 
0.333333333333333, 0.333333333333333, 0.333333333333333, 0.363636363636364, 
0.363636363636364, 0.363636363636364), dim = c(50L, 12L), dimnames = list(
    NULL, c("V1", "V2", "V3", "V4", "V5", "V6", "V7", "V8", "V9", 
    "V10", "V11", "V12"))))
gridB.intern<-seq(1,length(res.varsel$res.varsel$alco$res.detail),ceiling(length(res.varsel$res.varsel$alco$res.detail)/100))
plotvar.intern<-seq(length(res.varsel$res.varsel$alco$res$proportion))
matplot(gridB.intern,res.chooseB$alco[gridB.intern,plotvar.intern],
            cex=.2,type="b",xlab="B",ylab="proportion", pch=16)

## ----fig.height = 4, fig.width = 4, fig.align = "center"----------------------
# check the variable importance
round(res.varsel$proportion["alco",], 2)
barplot(sort(res.varsel$proportion["alco",], decreasing=TRUE),
        ylab = "proportion",
        main = "alco",
        ylim = c(0, 1),
        las = 2,
        cex.names = .5)

r <- 0.2 # a new threshold value (r = 0.3 by default)
abline(h = r, col = 2, lty = 2)

## ----fig.height = 4, fig.width = 4, fig.align = "center"----------------------
predictormatrix <- res.varsel$predictormatrix
predictormatrix[res.varsel$proportion>r] <- 1
predictormatrix[res.varsel$proportion<=r] <- 0
predictormatrix["alco", ]

## ----eval=FALSE---------------------------------------------------------------
#  chooser(res.varsel = res.varsel)

## ----eval=FALSE---------------------------------------------------------------
#  # kmeans clustering
#  res.pool.kmeans <- clusterMI(res.imp.JM, nnodes = nnodes)

## ----echo=FALSE---------------------------------------------------------------
res.pool.kmeans <- clusterMI(res.imp.JM, nnodes = nnodes, instability = FALSE,verbose = FALSE)
res.pool.kmeans$instability<-list(U = c(0.0151786390607247, 0.0391692968059588, 0.0345537179649034, 
0.0317384168665573, 0.0237596263098094, 0.0328544375710138, 0.036463830324454, 
0.0227812144931196, 0.0292362075495518, 0.0219606110339604, 0.0146938517863906, 
0.0179661658881454, 0.0146585027143038, 0.0315503093043808, 0.0232798889029163, 
0.0169864916045954, 0.0174157303370787, 0.019757606362833, 0.0154488069688171, 
0.0194558767832344), Ubar = 0.0239454614316374, B = 0.0818651084004049, 
    Tot = 0.105810569832042)

## -----------------------------------------------------------------------------
part <- res.pool.kmeans$part
table(part) #compute cluster sizes
table(part, ref) #compare the partition with the reference partition
res.pool.kmeans$instability # look at instabilitiy measures

## ----eval=FALSE---------------------------------------------------------------
#  res.pool.all <- lapply(c("kmeans", "pam", "clara","agnes", "mixture", "cmeans"),
#                     FUN = clusterMI,
#                     nnodes = nnodes,
#                     output = res.imp.JM)

## ----results='hide', message=FALSE, eval=FALSE--------------------------------
#  library(clustrd)
#  res.ana.rkm <- lapply(res.imp.JM$res.imp,
#                        FUN = cluspca,
#                        nclus = nb.clust,
#                        ndim = 2,
#                        method= "RKM")
#  
#  # extract the set of partitions (as list)
#  res.ana.rkm <- lapply(res.ana.rkm, "[[", "cluster")
#  
#  # pooling by NMF
#  res.pool.rkm <- fastnmf(res.ana.rkm, nb.clust = nb.clust)$clust

## ----overimpecho, eval=FALSE--------------------------------------------------
#  # Multiple imputation is rerun with more imputed data sets (m = 100)
#  res.imp.over <- imputedata(data.na = wine.na,
#                        nb.clust = nb.clust,
#                        maxit = maxit,
#                        m = 100,
#                        verbose = FALSE)
#  
#  
#  ## selection of 20 complete individuals on variable "alco"
#  plotinds <- sample(which(!is.na(wine.na[, "alco"])),
#                      size = 20)
#  
#  res.over <- overimpute(res.imp.over,
#                         nnodes = nnodes,
#                         plotvars = "alco",
#                         plotinds = plotinds)

## ----overimp, fig.height = 4, fig.width = 4, fig.align = "center", warning=FALSE, echo=FALSE, results='hide'----
# Multiple imputation is rerun with more imputed data sets (m = 100) 
# sink(file = "C:/Users/vince/OneDrive - LECNAM/Recherche/MI_clustering/Rpackage/vignettes/sink/overimpute.txt");dput(res.over);sink()

res.over <- list(res.plot = structure(list(var = c("alco", "alco", "alco", 
"alco", "alco", "alco", "alco", "alco", "alco", "alco", "alco", 
"alco", "alco", "alco", "alco", "alco", "alco", "alco", "alco", 
"alco"), trueval = c(14.06, 11.87, 13.86, 12.37, 13.87, 14.75, 
12.25, 12.22, 12.96, 13.76, 13.69, 12.93, 14.3, 13.58, 13.05, 
13.32, 13.73, 13.74, 12.84, 12.07), xbar = c(12.955000834408, 
11.8072411644835, 13.1491426380545, 11.9699476990474, 12.8621999853874, 
13.6168240832601, 11.9172852292351, 11.2941261378442, 12.2388808316143, 
12.9602436033018, 12.6237155939025, 12.2329613963989, 13.2114134615845, 
12.7959087755641, 11.6261477901354, 12.6141543103721, 12.6632901368168, 
12.8615579964161, 12.1585498154185, 11.4104891472368), binf = c(12.2063583071149, 
11.1319287483459, 12.5614497106717, 11.2315211853932, 12.2515115930677, 
12.8558814405274, 11.5762881299303, 10.7054121597584, 11.6480817997647, 
12.3392791509311, 11.7345686325608, 11.6650666965623, 12.4301220851931, 
12.1668547455557, 10.8780067837937, 12.0408484840616, 11.8643795368643, 
12.1740237087407, 11.6057229909004, 10.8878816525925), bsup = c(13.9018592426154, 
12.7025467896808, 13.7805760392031, 13.0822543969863, 13.6304370814934, 
14.4840221169834, 12.2833158516362, 12.0602549743186, 12.8861246342017, 
13.6180948502503, 13.4428617741057, 12.8217448319469, 14.0232220558536, 
13.39882492484, 12.4521833513675, 13.1744475988583, 13.5418065720333, 
13.5794156258357, 12.6949375101213, 12.0047588304373), pct = c(15, 
15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 
15, 15, 15), col = c("blue", "green", "green", "green", "green", 
"green", "green", "blue", "green", "green", "#FFFF00", "green", 
"green", "green", "blue", "green", "green", "green", "green", 
"green")), row.names = c(NA, -20L), class = "data.frame"), res.values = structure(c(13.0323895753234, 
11.5174202956721, 12.9075966201992, 12.6916434175101, 13.2556018330181, 
13.2392635723372, 12.010083919185, 11.7346695017141, 12.9301227359729, 
12.7509391102086, 12.9978495818517, 12.6301980101234, 13.4532307671844, 
12.5610016377195, 11.1900131701732, 12.94978551615, 13.4156301991453, 
13.1623789371858, 11.6496044049324, 11.4331907931104, 12.5048702658825, 
11.4621838499407, 12.5940305490481, 12.6813180610063, 12.8224118354398, 
13.2950891300802, 12.0992438711208, 10.7597553464333, 12.5392808406689, 
12.4333728964793, 11.3985939809995, 12.8064389166305, 13.0409513168706, 
12.0863239371598, 10.9314082137912, 12.6457046971045, 11.9582708748953, 
12.0938767117048, 11.7985123068921, 10.9695741710072, 12.850792322408, 
12.4537499736337, 12.8680569969254, 11.5600131376278, 13.4921098785876, 
13.5302228682291, 12.0260289560373, 11.8801086867982, 12.3592726101199, 
13.1090153094042, 12.2541725490936, 12.6522825405585, 13.0551710400065, 
13.734679707969, 11.258875572352, 12.6894143376251, 12.5780931003487, 
13.500160723937, 12.2239526663629, 11.0768437115802, 12.6064347204207, 
11.7630866586133, 12.5634226462229, 11.0195918475262, 12.7689342868519, 
13.0006972771072, 12.2607999789725, 11.0787044392856, 12.2282275006733, 
12.7582842158441, 12.3964861022862, 12.4471404265788, 12.9415687044793, 
12.3134687850845, 11.3425410134319, 12.6455636589996, 11.8894898589627, 
12.6260257048802, 11.9785860408999, 11.0363627161901, 12.7003271054231, 
11.7231863515567, 12.7144389690504, 11.6046017771623, 12.8077651556455, 
13.0021219339132, 12.1007904010016, 11.1069980079144, 12.2529094421551, 
12.640369054478, 11.7469752279562, 11.6395198262, 12.7941158143717, 
12.3799171905067, 11.2347171876996, 12.7835686833855, 12.1541423493306, 
12.7776977429459, 12.4972555606697, 10.9053072457327, 13.9012123718502, 
11.8701532033603, 13.9333119920529, 12.3245394341299, 13.7215090678245, 
14.6411905219257, 12.0190537619657, 12.0432694301794, 12.968545129858, 
13.7158576499837, 13.5682891103474, 12.9322495376633, 14.1826298789958, 
13.3916891488297, 12.7241948937472, 13.3864336204964, 13.4073141165742, 
13.6913686871982, 12.8910458473152, 12.000050981097, 12.8756954471915, 
13.3482138940502, 12.9128098385301, 12.6890388299717, 12.9677735236583, 
13.6056862495686, 12.1343613001186, 11.5085989643236, 12.438989219752, 
12.6304920878139, 12.4037004107626, 12.6354426166349, 13.0338400846924, 
13.0363210355914, 11.2227690076377, 13.0719692752808, 12.0697094760292, 
12.9064886203174, 12.333790808805, 11.5179196439864, 13.9141497871555, 
13.157255204151, 13.8184798486876, 12.232798792786, 13.574788192527, 
14.6233311613052, 12.24840126372, 11.9574376872096, 12.880969585366, 
13.6589895459656, 13.4718164106881, 12.7584769296094, 13.9987875686018, 
13.4395923992929, 11.3589836258866, 13.1418969037932, 13.5149761964248, 
13.3109613196998, 12.4590974270596, 11.8300330207748, 13.3412194365312, 
12.475464862808, 13.3275348860563, 13.1508168482125, 13.2589652348243, 
14.2415299127113, 11.694170169819, 11.6376809372488, 12.6168984934682, 
13.1135702904972, 13.0907821781695, 12.3812202057238, 13.489147115088, 
13.0381011437771, 12.0381855868731, 12.8689598654568, 13.2292793589866, 
13.149097415003, 12.2433562073325, 11.5477441640812, 13.7475127459235, 
12.6222143599858, 13.581356067406, 11.7705771527471, 13.2396307437503, 
13.9097654959105, 11.8286023444541, 11.569499571635, 12.4059911991262, 
13.1630038712893, 13.2237727799471, 12.3782825053874, 13.5924341868697, 
12.922371624928, 11.9531410755499, 12.6306370267662, 13.0572663902209, 
12.4126924169792, 12.7020790510775, 11.7717433980956, 13.576530132971, 
12.6892747362008, 13.4468766228987, 11.8549325618467, 12.6798059240269, 
14.2062201824224, 11.9745443242608, 11.7142770371091, 12.4797385932418, 
13.2087285622614, 13.3232275071838, 12.2776673193937, 13.6954033560727, 
12.2069996930949, 11.9989701040996, 12.7431573319037, 13.1795292475897, 
13.2048015636, 12.4599316240735, 11.6379827292583, 13.5521593864864, 
11.3469609375444, 13.4198920083496, 13.08142324468, 13.3309487235238, 
14.1912006009106, 11.9186106070332, 11.7642327336307, 12.5063942354271, 
12.9527557903783, 13.2389005512185, 12.2110022467737, 13.1293503750163, 
12.5041464391817, 12.1306378464793, 12.7848450983698, 13.0501473073588, 
13.2305938185743, 12.3976060047631, 11.7197496920139, 13.9474768703584, 
11.2358430045107, 13.0185014326956, 12.2048771320321, 13.6445528544199, 
13.615590473645, 12.1315546491615, 12.0727234493687, 12.8752369377558, 
13.6283747307525, 12.3974935878533, 12.6585005009276, 14.0688424204738, 
13.3987215299597, 12.4647787793746, 13.1739426079174, 13.055933975006, 
12.5550373152807, 12.2023751472606, 11.1997866548234, 13.5598318900374, 
12.5510226626598, 13.1828933443464, 12.0187269679724, 13.4093917623499, 
14.2117334031518, 12.1027907220149, 11.7378062382572, 12.3394673745972, 
13.2143229367221, 12.9751476879812, 12.5462634771743, 13.6866494409512, 
13.1804900819034, 12.1521019142741, 12.7493209071937, 13.0509279118932, 
13.1865405194916, 12.4830396027582, 11.5874478973618, 13.914837658719, 
11.8367245340363, 13.5513755460826, 12.6889420155076, 13.667990650928, 
14.4400214995031, 12.1794861598714, 11.7953773683508, 12.9757753680486, 
13.6634422616436, 13.33107193368, 12.9384388419911, 14.0750137204289, 
13.4010301487262, 12.4266258337615, 13.184042426735, 13.6134006047763, 
13.6186542182028, 12.7832426024268, 11.8832958371702, 13.5983627477099, 
12.5973804393299, 13.3881450361976, 11.9007476899074, 13.4422101850244, 
14.1360441108742, 11.7239414703912, 11.7456321218774, 12.4196424457489, 
13.2827280391937, 13.2300687360128, 12.307409617768, 13.7130756570856, 
13.2042170726596, 12.1306907972309, 12.8225169169118, 12.9027629832347, 
13.2247906732674, 12.3906833720205, 11.6412723771674, 13.0722752320271, 
12.6748986502992, 13.2542988725658, 11.9477205689345, 13.2197009075667, 
14.1513544686971, 11.7872235539765, 11.728765365575, 12.3039827807045, 
13.2364528681332, 13.1774406195077, 12.452230026193, 13.7392298896394, 
12.9832211085973, 11.4469147641087, 12.8276003141901, 13.0702212668852, 
13.1513729602296, 12.4335595530685, 11.5593210880067, 13.8157411172285, 
12.1314068903515, 13.5775856455978, 12.4450764534251, 13.6296941460762, 
14.5145042497181, 12.2257540445382, 12.1153880161735, 12.9576309346198, 
13.6178378831308, 12.6653176892366, 12.6940606732256, 13.9626717838769, 
13.3681994372995, 12.451520434104, 12.7588666627822, 13.5563807839206, 
13.3554751665383, 12.7339308091712, 11.8918814544565, 12.6960639615274, 
12.5167027013226, 13.3011895700271, 11.980464320931, 13.2114330220947, 
14.2705491818155, 11.7496149535072, 11.6015349682029, 12.3211187254235, 
13.1790708239374, 13.0703156024046, 12.3964154771971, 13.539433981415, 
13.0170862581867, 12.0581868591257, 12.7145535143394, 12.8006951279724, 
12.4091004468497, 12.3767698315785, 11.3819567378016, 13.5003924067207, 
11.2476804167279, 13.3490439466184, 11.8650740408142, 13.3074114080439, 
14.2487534431159, 12.0640131977525, 11.5915827571238, 12.482399653356, 
13.2251811304532, 12.9916177615128, 12.1674522005006, 13.644529744478, 
13.0734054140282, 11.3722446229283, 12.7932980785797, 13.0387538306988, 
12.9038432098125, 12.2958369568884, 11.5632816721282, 13.9206219226196, 
11.6895328205293, 13.7275957485278, 13.078805411964, 12.2831313279244, 
14.5265781259271, 12.2074565873106, 12.0543683212421, 12.8071649628339, 
13.6018337766641, 13.4413378458646, 12.777965473179, 14.0208209840314, 
13.2523236487849, 12.4296439589359, 13.0567747720892, 13.6212353563359, 
13.6154933862233, 12.6934496973301, 11.9861714363226, 13.9006536497105, 
10.9018045727728, 13.7285514657377, 11.7673196701185, 12.9762240983353, 
14.4478481154984, 12.2822391259516, 12.0595987387896, 12.3124554451553, 
13.6229772255212, 13.3408516661994, 12.5545447678341, 14.1152432097915, 
13.4007894275651, 12.5731451222457, 11.7262498057878, 12.5151621959258, 
13.5775167963416, 12.6449207645902, 11.9478208838239, 13.7938030335949, 
11.8861014535168, 13.57505227681, 12.4728951359744, 13.6251349639982, 
14.3627846633792, 12.2315937499785, 12.1407546096706, 12.8155925869346, 
13.4828739515504, 13.325144775363, 12.7507308409615, 13.9615147941317, 
13.2457231459404, 11.5350312372811, 13.110350492622, 13.5410395082498, 
13.4963522352437, 12.6651785301719, 11.8282478431989, 13.377575080652, 
12.4578342346477, 13.1695552403589, 13.0980462908061, 13.2720863424083, 
14.1469641434033, 11.5595803377655, 11.6176625022756, 12.3154796603158, 
13.2085209278329, 12.8853716375421, 12.2412314939006, 13.5558367420002, 
13.0560950612571, 12.131200939904, 12.4526161984102, 12.6203992776322, 
13.100701528884, 12.2389312271319, 11.4181728936889, 13.5018537485157, 
11.3860348183976, 13.3847720103219, 13.2334073912717, 13.2882212593706, 
13.9951279985171, 11.2246902253645, 11.5302324873018, 12.455647908347, 
13.4896064713476, 12.1041770579187, 12.3344105423151, 13.6591830415937, 
13.1436293672455, 11.8472144613637, 12.758696969773, 13.023410919601, 
13.0908606162234, 12.38663346977, 11.3927766413284, 13.16038253586, 
11.8848949265359, 13.744951649002, 11.9422627842907, 13.6849830398575, 
13.9372714304164, 12.2061041075575, 12.0759734702621, 12.1581209377371, 
13.0195893355471, 13.4352732090869, 12.8801482151722, 14.110209052903, 
13.292437448402, 12.425742407498, 12.625839254978, 12.904090347629, 
13.5221490552219, 12.6945616395447, 11.9618664654885, 13.5756600918593, 
11.2760578782101, 12.9840994828424, 13.2098416974619, 12.4999166068476, 
14.1331667854498, 11.843860266795, 11.0202206566745, 12.471477819517, 
12.6821340971127, 12.6677307998269, 12.422405559731, 13.6409982022853, 
13.0012145577104, 12.0744747555911, 12.8872373224106, 13.2093293785672, 
12.426242310831, 11.7851775972913, 11.5957153633217, 13.1463436537171, 
11.4780276169388, 13.4159304838534, 13.0804271152805, 13.3091789320922, 
14.308554436182, 11.8243585111376, 11.4206570194363, 12.2457047411029, 
13.2793755207994, 12.9189579696863, 12.2762515023262, 13.6765932702933, 
13.0029084007437, 12.1225957660735, 12.8799342533777, 13.3183324151978, 
12.9246691001442, 12.4796743928459, 11.5615968266071, 13.5156671776311, 
12.6366861572443, 13.3765127996176, 11.9204063751861, 13.3363762487689, 
13.6906073019783, 11.8055571620752, 11.4889623784218, 12.5591310925786, 
13.3235262112011, 13.4025061629626, 12.4110696029019, 13.4844426269754, 
13.1272500165566, 12.1008057577402, 12.8415595352504, 13.0231530518374, 
13.4436947017325, 12.303676769876, 11.3715137375089, 12.9140009474233, 
12.9547158058001, 13.7343008898705, 12.3554154320452, 12.5385874211733, 
14.4831021818285, 12.1916621056551, 11.2113723371782, 11.8122722726681, 
13.5803163339831, 13.5163166186906, 12.6196805034157, 12.7814532688368, 
13.4132022296608, 12.5641647971197, 13.1730181750503, 12.5350302720644, 
13.5745026721834, 12.7538141155887, 11.8985809684957, 13.5038414884731, 
11.4755151333842, 13.3344959221438, 12.9299850616842, 13.3244650822442, 
14.3015942014655, 12.5787240157804, 11.7055568773545, 13.3120467694682, 
13.2737308173554, 12.9098515023182, 12.3791023942965, 13.6620714267778, 
13.2617197073329, 12.1803479868898, 12.9371599833352, 13.1236725489675, 
13.6283436917169, 12.6476629201234, 11.598394385407, 13.4579166235033, 
12.31995466778, 13.3525576883754, 11.8005541489519, 13.185502124206, 
14.2567518270861, 11.7609741510236, 11.8699898078381, 12.4320861778816, 
13.0840609876022, 12.7740137751989, 12.3380421564933, 13.6423728182635, 
13.0187007678494, 11.4800844957989, 12.9401761206759, 12.7962257753398, 
13.2342261004331, 12.2317820748891, 11.3494758854618, 12.8890855351347, 
11.2871873198307, 13.2465130798601, 12.9204009703492, 13.0530441580773, 
14.1087568751991, 11.6651372459158, 11.6694563611806, 12.5308954184799, 
13.1767538434165, 13.1053497351552, 12.3079311276817, 13.5312243585347, 
12.8749664626571, 11.8907467377021, 12.0867615770468, 13.2827386921318, 
13.2153358070594, 12.44033787768, 11.0697197526172, 13.6166019475459, 
11.5984157230685, 12.8291298891264, 12.2415075944922, 13.6593738955839, 
13.9147096842286, 11.1497696160797, 12.1771747490956, 12.8838089446348, 
13.580679778527, 13.5136512365971, 12.8270247478795, 13.962402451727, 
12.5298010168455, 12.3482981168437, 13.2433750888685, 13.6725572049132, 
13.6467100369502, 12.6299138447144, 12.0942079679029, 13.447700148679, 
12.5032288933672, 13.5121262753032, 11.8911590415566, 12.3334144204468, 
13.2692109473557, 11.8118625402498, 11.7179510680138, 11.8933750921863, 
13.2888755991109, 13.0820616997052, 12.4871450649364, 13.6400112577109, 
12.3969762636476, 11.4193561903376, 12.8576278607828, 13.0525505900002, 
13.1957077084977, 12.0361593431453, 11.537763348018, 13.5243724269322, 
12.6389134305873, 13.2991937717485, 13.0174029300371, 13.357509017482, 
14.224421797028, 11.8564697555716, 10.9915889358253, 12.5144442356046, 
13.3208231086053, 13.2466897749381, 12.4023481690922, 13.5829454782734, 
12.7355265200299, 11.9975943824308, 12.8579705135145, 13.5046838676739, 
13.2112113433086, 12.4296593915856, 11.5263552736372, 13.5586947830407, 
12.4953145608636, 13.3683391375106, 12.1740832898499, 13.2561126052388, 
14.2161586293652, 11.799526496425, 11.7277481127239, 12.4798346939786, 
13.5259334693914, 13.1920569195249, 12.3573562398856, 13.6699315003591, 
12.9794608515818, 10.742572175365, 12.0139120026061, 13.2538116859586, 
13.1777630523886, 12.3865962909947, 11.4953615875227, 13.5250873407825, 
12.6180672514986, 13.3098895540277, 11.7418807562489, 13.3478370219387, 
13.695398668326, 12.6270778926599, 11.1217363589285, 12.1074339695165, 
13.2308457781802, 12.0029518557583, 12.3617569868897, 13.5976621098826, 
12.9135899365281, 12.1130732330487, 12.793960676579, 13.2284785857006, 
13.1407211909155, 12.5967568798225, 11.5880492488422, 13.5615616903573, 
12.5404386089849, 13.0080726860296, 13.0668791558753, 11.6881431750503, 
14.1918889999893, 11.762867484012, 11.6443623459521, 12.5068710837047, 
13.2245411960427, 13.3319437688276, 12.4279561389062, 13.6375151467803, 
12.9563696969436, 12.1173538578446, 12.8179537454776, 13.076214574147, 
13.1696531917258, 12.4238201785081, 11.4149033726131, 13.5063161536743, 
12.569610868116, 13.305338196447, 11.8092106356971, 13.359249511568, 
14.5015008849264, 11.7530086475936, 10.6738129516334, 12.5008391222597, 
13.2347091814887, 12.539335945737, 12.3460283571067, 13.6421509913565, 
12.9930166512613, 12.0779580852305, 12.7658412581808, 12.6151771865904, 
13.2416157622577, 12.4004787598268, 11.5774665168304, 13.2995700869945, 
11.6767300642874, 13.2116798816895, 12.0894063056158, 12.8569662695042, 
13.0832270096833, 11.752175482588, 11.4692033238152, 12.0632792862225, 
12.5830884023961, 12.8165633233942, 12.1639179348238, 13.4122923840442, 
12.6985726746073, 11.8688787552982, 12.4283803575252, 13.4872682476101, 
13.1168219612042, 11.9697156582709, 11.0977116505823, 13.4350383937051, 
12.071975429659, 13.0137841189866, 11.8035936131142, 12.5761737650453, 
13.6732443955513, 12.3037736396439, 11.6935876265875, 12.6262258403357, 
13.0910524726356, 12.9480420071442, 12.4249217852399, 13.8054838659026, 
12.8973632775111, 11.8339896697791, 12.5538625069768, 13.6469899332407, 
13.1572799600934, 12.1696733480337, 11.205697792085, 13.2169186390352, 
11.6910921036823, 13.3958790063482, 13.2491272610698, 13.1013790757684, 
13.7578359897307, 11.8439190321031, 11.2168555096149, 12.4760934520526, 
13.320913272626, 12.4188999802264, 12.330994396908, 13.5454921195001, 
12.6309559750455, 11.6429285876993, 12.7713855333587, 12.7922426981034, 
12.9722274281631, 12.2343654401181, 12.3013224001111, 13.5539853432437, 
13.066823663711, 13.4474588318702, 11.9126463744572, 13.272572635263, 
14.3434044168773, 11.8127631528334, 11.623004229162, 12.4531927336774, 
13.178531654275, 13.1272975706075, 12.4529759655639, 13.4164120718723, 
13.1025407407923, 11.9818082338662, 12.7070162582502, 13.3684377456436, 
13.1794650142795, 12.3715721453417, 11.6110405997693, 13.5808903660984, 
11.4782394966763, 13.4338224341271, 11.7532139107432, 13.4228914529028, 
13.3112230653873, 11.997284754936, 11.6492997288061, 12.5161570852668, 
13.092189115053, 12.9263724181677, 11.7898893154467, 13.6311570380716, 
13.0494731003281, 12.0944009627064, 12.7969513423296, 13.1419345457682, 
13.1334181622176, 12.4918004395912, 11.5230315007507, 13.6123469707965, 
11.5460569525888, 13.4683772643986, 12.0324988078879, 13.4488025404808, 
14.3241275903879, 11.8213698150251, 11.6724222599459, 12.5429649565366, 
13.1993148918897, 13.3194418767795, 12.4227197638256, 13.6399258480287, 
13.0417958766937, 12.2487646409937, 12.8184738070251, 13.3117015669215, 
13.1990736044075, 12.4677739495089, 11.7617258170924, 13.5836370821408, 
13.0955852051751, 13.6535753329929, 12.0558077145317, 13.3952629385729, 
14.2316786889283, 11.8585225886135, 11.7638933611248, 12.6028070634878, 
13.2786706760923, 13.3443637544796, 12.4984955621402, 13.7271853594055, 
13.2045618663271, 12.24267305543, 12.8790260686332, 13.1949299714896, 
13.2918604146475, 12.4872848665174, 11.789306552271, 13.5715956580025, 
11.4547850160545, 13.4605820219177, 12.0853096250912, 13.4345425537487, 
14.2958956451042, 11.5766503780938, 11.7543270669905, 12.5938211029434, 
13.2858192901152, 13.349995061222, 11.840688898591, 13.2693503387947, 
13.1366972518679, 12.2431016399515, 12.3322812491924, 12.756774129395, 
12.275895913933, 12.4476102001223, 11.5741430157004, 13.6196891997056, 
12.5392082587962, 13.4612008879213, 11.8416346438954, 13.2944403728378, 
14.2386145997873, 11.9137536126928, 11.6993012795266, 12.5835006906129, 
13.426107281122, 14.3112682042015, 12.8132115227675, 13.7302083786745, 
13.0098692458319, 12.0965235929882, 12.9506551014332, 13.2421780637561, 
13.2300229451673, 12.4152374002539, 11.6646264237919, 13.4423517528513, 
11.1348924886479, 13.7790520688914, 11.8457802419506, 13.1901968825048, 
13.865734505796, 12.1177787693783, 11.3726207859877, 12.4227839522995, 
13.2847259296344, 12.8875055793167, 12.4650280292933, 13.6991283756781, 
12.6653268271602, 12.4855912791423, 12.5845416863827, 12.6844188267331, 
12.9627955888267, 12.5919810756708, 11.5676716925839, 13.0527477107732, 
11.4827501457124, 13.2413887169568, 11.955178913909, 12.95192088928, 
13.3380006506283, 12.2664900322926, 11.0980760700032, 12.5275010850565, 
12.8695529058025, 12.7048518454582, 11.9026224386629, 13.530365169048, 
12.73874606713, 11.1724155372515, 12.895564291758, 13.1258403377039, 
13.2585347125688, 12.3634877739737, 11.4791777480514, 12.9209837057887, 
12.5498533534987, 13.3705000903456, 11.7826865947665, 13.0304685078811, 
14.1309419014853, 11.6388733166965, 10.9616504532837, 12.252367733002, 
13.2564226227787, 12.6634271427478, 11.8508335673011, 13.2847545353769, 
12.537470526188, 11.8566286829436, 12.8413903848635, 13.009411943932, 
13.0964895232342, 12.1765646632165, 11.5774012066545, 12.7175042329397, 
10.7523244951065, 13.2498056407573, 12.0421729820963, 12.8835634040219, 
12.835208812736, 11.6098014200456, 10.5434125490701, 11.9144931815316, 
12.8767252210466, 11.9425152898453, 12.3305724496606, 13.1143016139567, 
12.5946666843658, 12.1294863748191, 12.5711790735616, 12.9452495730963, 
12.8006865837065, 12.1686401630443, 11.0305672590601, 12.8670041534274, 
12.3708570277418, 13.0140502229309, 11.4352935453833, 12.534320511191, 
13.0844755439882, 11.7760716228595, 10.9081419525931, 11.7971228709234, 
12.5330925576884, 12.2879614873915, 11.5388045458191, 12.8381575694064, 
12.3651759365212, 11.1442820049748, 13.2281523536958, 11.9948660413589, 
12.5575340545676, 11.8174989236145, 10.8874874126713, 12.3408438770546, 
11.3096740749935, 12.9918838031495, 11.2839167435646, 12.2514462341277, 
13.2872100594914, 12.0206768409408, 11.0644548894927, 11.8299942355998, 
12.2057408934183, 12.7681186703539, 11.9237389155518, 12.3590865828061, 
12.8367503575758, 11.0451192829609, 12.3250167011541, 12.1722034193301, 
12.3368271488204, 11.9099294720535, 11.2060631046462, 12.3328612315431, 
11.3361823055834, 13.8615562344087, 11.0615465574654, 12.5018212093476, 
12.7345135996965, 12.0591808096143, 11.18570834214, 11.788744262212, 
12.8141584382597, 12.1465454543684, 12.1766229262241, 13.4870646358236, 
12.0542146767168, 11.3332727123101, 12.5647278771483, 12.1378300207973, 
13.4855294106205, 11.608222728423, 11.0290325475075, 12.6250290505984, 
11.7274460074863, 12.9856970693053, 11.3986616922536, 12.6406265029032, 
13.3827505188934, 11.902206078805, 10.8597741979766, 12.1585283202313, 
13.054949950859, 12.2941084865761, 12.0333456115565, 12.4359286682886, 
12.498653888916, 11.054446051969, 12.4607147391411, 12.3097331255192, 
12.5853202039005, 11.8660698817915, 10.887902402062, 12.4482833672501, 
11.2106468611975, 13.0973149681652, 11.6635727200391, 12.7170387686356, 
13.8002165111732, 11.569405414824, 10.9743294035397, 12.2874089006334, 
12.9902776591139, 12.5603199615093, 12.2941383331135, 13.0258854194146, 
12.6659147455772, 11.1323036215498, 12.6587670815312, 12.3925880460979, 
12.3192704290958, 12.0657624396052, 11.2632578014143, 12.3022219912384, 
11.4469522691003, 12.8405857939758, 12.4700526998562, 12.7391703588246, 
13.4429643293726, 11.8973751285798, 10.7614742671105, 12.0574054070354, 
12.7070339820089, 12.7353459622176, 11.1835720601925, 13.2166124816391, 
12.5019419720313, 11.4764499186419, 12.4194409779327, 12.1868127922711, 
12.6045782386896, 11.7913818741816, 11.0951040586365, 12.4447916668894, 
12.4607471618131, 12.6517751120251, 12.3626936041548, 12.472977307367, 
13.0490867966598, 11.912056425071, 11.0476690399679, 11.640035733818, 
12.5601188002468, 12.1904302700582, 11.7574152941246, 12.3756541148467, 
13.2472304203874, 11.3839053406043, 12.3014283451742, 12.3663709434367, 
12.5163864358513, 11.6641054613405, 10.8439880067289, 12.1474322864384, 
11.7807494539501, 12.8930153122974, 12.1347659640236, 12.4755613626597, 
12.718928545112, 11.9009803368932, 11.0881691761525, 11.5899596602582, 
12.800556530256, 11.3375255307198, 12.0396702982912, 12.4737089974595, 
12.1178723570269, 11.0511311509861, 11.9642959993216, 11.742615289473, 
12.1471160636589, 12.0108200499929, 11.0072044992169, 12.4130920530953, 
11.1355626505959, 14.303002635148, 11.686686090862, 12.5301979816076, 
13.4579667271033, 12.1725783258033, 11.0822826488367, 11.8605785925745, 
12.7183811554716, 12.8986008701209, 12.8499819533213, 12.4589808431762, 
12.9839450303045, 11.2779627978127, 13.252378982619, 11.8646198216346, 
12.1556319103577, 12.0383290073011, 12.3221129236736, 12.4917933708712, 
11.2830091222967, 12.7950359441039, 11.4233946980311, 12.856025514392, 
13.0007308111077, 11.8676609391194, 10.4099277919656, 11.9016854694749, 
12.8376170742469, 12.341621080391, 11.9814162617515, 13.0594621026442, 
12.6386900394856, 11.3758390794184, 12.5331536411637, 12.3901972720588, 
12.8826778810415, 11.9986298490687, 10.8348621149908, 12.1192975184584, 
11.7752380736192, 12.6681092212942, 11.7925949618914, 12.251515033012, 
13.5291530530509, 11.8896739859772, 10.9397081243911, 12.0541696916114, 
12.679652677531, 11.9307843457063, 12.2567316591836, 13.3426341944172, 
12.6770234363508, 11.2083135363223, 12.3015169486888, 12.2060791881114, 
12.5940869334766, 11.7959487097387, 11.008257645195, 12.2368799299229, 
11.0756176826084, 12.494988852956, 11.627413087499, 12.4109887745578, 
12.8899799602428, 11.8585660784428, 10.7999423923863, 11.6485052769198, 
12.1828942087991, 12.134699626099, 11.6679906687922, 12.1526229518276, 
12.4699408961253, 11.1209475809709, 12.3076189327216, 12.1022590517069, 
12.1010438178649, 12.2733944731565, 11.2468156179783, 12.3627345210546, 
11.3917659685596, 13.1251210116814, 11.7129581712308, 12.7707538332227, 
12.4486048357179, 11.9877733966184, 11.1327327012154, 12.5584918688611, 
12.4174173742949, 12.1732190056701, 11.9096840598909, 12.5114443580209, 
12.8974057304085, 11.2994315519479, 12.3578094903745, 12.1520192408741, 
12.6788265917331, 11.7809661299417, 11.0132524953091, 12.5889037585934, 
10.9485919844584, 13.2256748833491, 12.0514542680684, 12.3796864278759, 
13.7954907960262, 11.6289663424064, 10.7056649164582, 11.8128638731711, 
12.6845756126846, 12.538527541978, 12.3987439493481, 13.2040368817819, 
12.4648519108467, 11.5802677787619, 12.1508839918688, 12.3854635447946, 
12.6804404631412, 11.7663058389862, 11.4867051015278, 12.3545360295973, 
11.548820426387, 13.0555047823878, 12.2447928867869, 12.3443781002889, 
14.0452064593422, 12.3683692144877, 10.948767093199, 12.2810814936144, 
12.845909022378, 12.3351418486255, 11.6859216616078, 12.5490940273404, 
12.6417334737453, 11.6528972981237, 12.3136499179613, 12.3137203704308, 
13.1318348278129, 11.7463544679667, 11.4443202182517, 12.4199013387365, 
11.6735672900898, 12.5773209533643, 11.8355896169725, 12.571945927104, 
13.1018214583631, 12.0524099287801, 10.9161144441268, 11.8907653863686, 
12.6115843230949, 12.3364284144565, 12.0603917206979, 12.6431492064624, 
12.6132112055934, 11.4065463821295, 12.7068995392701, 12.3895787059372, 
12.4589553654429, 11.8192742095511, 11.4363866253895, 12.4642917538245, 
11.3349461257694, 12.8674489360625, 11.5151259976816, 12.349217598264, 
13.2771761594414, 12.1382600758026, 11.5639853568131, 11.7679146119775, 
12.7689337449642, 12.6343220469464, 12.2730889473507, 12.8216780947389, 
12.7692239047469, 11.4412914168517, 12.0154305084866, 12.3753486116017, 
12.2802723456588, 11.634599259593, 11.3479551915935, 12.7879784802306, 
11.6293746466646, 12.9295423507741, 11.6610840618867, 12.4521814014594, 
12.8903834902239, 11.9860489588993, 11.1386925727562, 11.9100953704636, 
12.899662455628, 12.1333334924552, 11.9053695033782, 12.8674821951975, 
12.4215315831738, 10.8342611832501, 12.2172868332517, 12.3045072944773, 
12.7663329188865, 11.9284920501575, 11.0401255684698, 12.1442774803339, 
11.4327895953551, 12.5040631892057, 11.4315861849807, 12.7931448381974, 
13.0112740347838, 12.022731361216, 10.9452737520523, 12.1162015513931, 
12.6048991869885, 12.1520038659475, 11.8821615123507, 13.3899848428484, 
12.2632598349592, 10.7338174524995, 12.1969992762107, 11.9459917574411, 
12.6435691395924, 12.0835991102195, 12.3154629923795, 12.2835341071197, 
11.7314589596126, 12.8444755837678, 11.6413308526513, 12.461408079647, 
13.1573138016137, 11.9671276478432, 11.0561378253668, 12.1649905671483, 
12.6637013554446, 12.4914714529888, 12.0376059192177, 12.8417154850934, 
12.3292272674609, 11.1955049869072, 12.4153714546682, 12.0750069692787, 
12.0578291849011, 11.7152748890105, 11.0770982105075, 12.5135036790379, 
11.2828386743285, 12.5239639352004, 11.7264601041176, 12.4057081192197, 
13.1655209881836, 12.2249618866234, 11.1614838571002, 11.9339429877258, 
12.4910498015334, 11.2944150364751, 11.763888769558, 12.8639087483316, 
12.5218064685124, 11.1749282408871, 12.4607467702212, 12.4798622779472, 
12.4491591247008, 11.9715836962201, 11.1575881321614, 12.4246795740061, 
11.4095022700105, 13.2963722006639, 11.2097329324811, 12.7127959793556, 
13.1415587160067, 11.612963636906, 10.990069075621, 11.8183795826606, 
12.3279704473493, 12.7519608842513, 12.1109490902596, 13.0936429419304, 
12.5969504714023, 11.4129846536178, 12.4392936914141, 12.4428683526149, 
12.8212349852949, 12.4644538202309, 11.0684933308062, 12.6641778589201, 
11.3064364526321, 13.1948050458431, 11.4232679152837, 12.6892851795429, 
13.4382770830343, 11.700765201675, 10.9190142101192, 12.2780995680349, 
12.8129421205486, 12.3374970294045, 12.6958369637693, 12.634651230383, 
12.7990647637687, 11.6835377677891, 12.5680628414462, 11.850257869361, 
12.8031934337272, 12.0607487987665, 11.434447207086, 12.2032703499009, 
12.1318958457604, 12.8946187332784, 11.9419019070567, 12.6727338424166, 
13.1940630859913, 11.9525666379759, 10.8618765547123, 11.8319368666734, 
12.4815338977992, 12.4672288054013, 12.141233705452, 13.1017573926683, 
12.4820668699977, 11.66378384326, 12.2387667623748, 12.2864999038933, 
12.4808690337732, 11.6778045855691, 11.0723744052588, 12.3030306999737, 
11.5264845021911, 12.5979785119389, 11.2245786667154, 12.2805996682781, 
12.8569694735691, 12.0049512221983, 10.828909482399, 11.7450049102884, 
12.3393348489606, 12.4649011237382, 12.0368326527343, 12.4443365505282, 
12.3850522545072, 11.3846836615526, 11.9882933147521, 12.5452341590631, 
12.5394506892568, 11.4728094905045, 11.0551628720627, 12.3061783458504, 
11.6972547671188, 12.9346237959608, 12.4342647153249, 12.4854188447928, 
13.581673021853, 11.8854219793565, 11.1152284507352, 11.5998395316246, 
12.5486380881493, 12.147164270123, 11.6018423571102, 12.4329888204745, 
11.979273663472, 11.1455115059857, 12.20770474641, 11.8125459304443, 
12.3505141203127, 11.7536474353295, 11.2121445681553, 12.4599772836527, 
11.8142269653312, 13.2591466733788, 11.5972847330811, 12.466509862515, 
13.3189234673879, 11.8749430245585, 10.9402720352083, 11.7629536589818, 
12.7106491561829, 12.3633675700392, 12.308048287484, 12.8739369712049, 
12.5480935585808, 11.4356594494551, 12.7243089326596, 12.4884362841094, 
12.595494116486, 12.1303790491057, 11.4037850345443, 13.9586083108787, 
11.5055298357099, 13.809531475125, 11.8946015632423, 12.2132501736969, 
13.5356673543264, 11.7921105424283, 10.9391080273141, 12.3447919993595, 
12.8049770407108, 12.4465870966755, 12.0021660185769, 12.7044819535884, 
12.7337509907949, 11.8181894673042, 12.732976392003, 12.4236308311686, 
12.190828274857, 12.2491127688489, 11.6210517895981, 12.5370702600141, 
11.79640040542, 12.6678480787744, 11.3633387650712, 12.335054741836, 
13.0654888648995, 11.4873850880155, 10.7006097824622, 11.8319836821327, 
12.3506451066823, 11.8293892784962, 12.0226760800783, 12.5152047543893, 
12.8872216429393, 11.1134361919403, 12.1862465873696, 12.4767206517501, 
12.4126609962069, 11.4288574080565, 11.4397095307725, 12.4310208025066, 
11.1520948934916, 13.5527741130383, 11.658562714405, 12.3639307319647, 
13.2137454252411, 12.4657482453265, 11.7574208079814, 11.1160071978147, 
12.6140259349743, 11.0251072302649, 12.8214669416347, 12.3193515387843, 
13.2724328015338, 11.6002327498864, 12.2547128822804, 11.8598141262283, 
12.6938084957558, 12.1291792788758, 11.4980600022133, 12.4770124790556, 
11.649898500984, 12.9058113310135, 11.5912321451179, 12.6073104941276, 
13.2086100317036, 11.7688529472277, 10.836424420456, 11.9113360667273, 
12.7755589848132, 11.9157340494063, 11.8213988715974, 13.0768764700326, 
12.6264380756592, 11.9483557784898, 12.3688658707595, 12.4688999612456, 
12.4780672995779, 11.5582279779702, 11.2647530103967, 12.3006717599566, 
12.2807604726844, 12.9507762707856, 13.0534204865996, 12.2461202635808, 
12.8961598029426, 11.8416369644198, 10.535137756563, 11.2151533693208, 
12.784118905659, 12.7964472463571, 12.2943793629546, 12.9389465648423, 
13.2218682424969, 10.9654228564663, 12.4657140957631, 12.1154502957961, 
12.4227952309822, 12.334358326078, 11.1868743734757, 12.2065208311788, 
11.3077793252781, 12.4132274827191, 11.4276847104954, 12.242682169793, 
13.1226500646959, 12.0177402972845, 10.8436715876252, 11.8814734203608, 
12.338220888371, 11.8430266657182, 11.7483560465278, 12.5934679067598, 
12.7700187328676, 11.0080536870484, 12.0421862722497, 12.4478065489344, 
12.2875341903586, 11.7606476327186, 11.1855911964116, 11.8500290155708, 
11.8274578743884, 12.6200980389983, 11.5127177140704, 12.6012018691919, 
12.9073980649133, 11.7913928781367, 11.0039923734732, 11.8163557138231, 
12.8956441052171, 12.054575948304, 11.7839986211551, 13.0432374981078, 
12.6171775632135, 11.4820953235113, 12.3428270056232, 12.3184234472879, 
12.5504162089182, 11.6445059727774, 11.2743640092044, 12.3476719469949, 
11.5283521600597, 12.31874214624, 11.5125581961529, 12.4871429285749, 
13.0238035803818, 11.6811017125311, 10.7639342272051, 12.6253579248495, 
12.7437630674881, 11.9125341251222, 12.0507598830734, 12.4457572949074, 
12.4680868348452, 11.1850446427037, 12.2523427019583, 12.1739673015009, 
12.5731741732884, 11.2624024758039, 10.5768464224621, 12.8035187332142, 
11.2470706848288, 12.9859834807989, 11.5159062755699, 12.8270389991698, 
13.3611174529454, 11.7128360429671, 11.0500808445277, 12.0969332675025, 
12.8765851566468, 13.0145202755603, 12.0420397471965, 12.9740791825966, 
13.099895514282, 11.3478122357476, 12.8863839049099, 12.6209153732575, 
12.7599168273367, 12.41380547733, 11.2862210540864, 12.541330718912, 
11.4925061247971, 12.9406135217386, 11.5073225883015, 12.7109959681455, 
13.2709724528625, 12.0236376281564, 11.1188857124861, 12.0526626215614, 
12.9704723221147, 12.3319412386557, 12.0646668694882, 12.6879018070979, 
12.4715505181008, 11.2941431364173, 12.3941246168649, 12.282328330264, 
12.1749916981293, 12.0361167027765, 11.0101015304054, 12.5531890268836, 
11.2623169817898, 13.1491303130417, 11.561110840693, 12.7278229967523, 
13.1098104983623, 11.8257114419959, 11.0020422759025, 12.1238943068285, 
12.6691899727797, 12.5279081421958, 11.8936960650561, 12.7084319485236, 
12.5243256900075, 11.1354682180934, 12.5285660844784, 12.4338705702336, 
12.8438649582488, 12.0290537607834, 12.4204959809126, 12.6895710358542, 
11.2910930234308, 12.9705540358235, 11.0060910601918, 12.674695383497, 
13.2487911027269, 11.9835694667938, 10.7348583601676, 12.261108161504, 
13.1477857377529, 12.3215594904322, 12.2351905139601, 12.8549526619608, 
12.1690094339012, 10.8803091838223, 12.6328287207738, 12.3016450531771, 
12.7388912367304, 11.9875130029518, 10.8416610956758, 12.6318896105998, 
11.5430743324406, 12.7642013872458, 11.5725460398048, 12.6141334946116, 
13.4877090166199, 11.631852301686, 11.1138970815885, 12.2218935788295, 
12.716968783902, 12.1243868115876, 11.9932865503997, 13.0441045895269, 
12.8772746969164, 12.1692710800748, 12.9196240294692, 12.5479583000461, 
12.7300271380473, 12.4852879800762, 10.9478191989662, 12.8049550284541, 
11.5587986050883, 12.8434858670175, 11.8546762149223, 12.750452061796, 
13.410057718632, 11.7605458112578, 10.9945763818868, 11.9990959581804, 
12.9583289672084, 12.7159265983585, 12.1187839819474, 13.2202928262203, 
12.9824634149221, 11.7651859420945, 12.4587047448963, 12.5161707090047, 
12.7569327010912, 12.145426635969, 11.1174647515362, 12.512901651319, 
11.9770069661746, 12.9387889554444, 11.4751849162527, 12.5911367368185, 
13.411252442142, 11.841455869055, 11.0486402544577, 12.4460790118229, 
13.2025984244092, 12.3474561441448, 11.8817817136027, 13.1677106717351, 
12.7882001768796, 11.2651629604944, 12.5685768113964, 12.2533709146048, 
12.7741028793225, 11.3395148438525, 11.4315491956598, 12.3479091589836, 
11.4895045504892, 12.8425789041151, 11.3004566706366, 12.5590301271419, 
13.03018379653, 11.6842005162185, 10.8618179760352, 12.0129553641387, 
12.9073099685961, 12.5451832009942, 11.8764745625946, 12.8380050859621, 
12.5758355269808, 11.2555808246271, 12.4423996234361, 12.4344164776064, 
12.6842316272796, 12.1118629296578, 11.5575774084364, 12.4027076694619, 
10.8646093028878, 12.8237959260422, 11.2987849299673, 12.4214295416683, 
12.803738335153, 11.8279099528145, 11.2109412752973, 11.8083601060831, 
12.267959783779, 12.0074543246254, 11.6264834580532, 12.0487537958571, 
12.494369485396, 10.6921447480614, 12.354745062241, 11.3534463627763, 
12.5393433221333, 11.9862411576322, 10.9943216284375, 12.2840187285584, 
11.7854137904915, 13.0495590249223, 11.2318865811131, 12.5825755534528, 
13.3384392767751, 11.7082546989094, 11.0376336555272, 12.3147719551037, 
12.4747773947176, 12.61137509759, 11.6664112686866, 12.6972405106717, 
12.1259156669924, 10.7589408830842, 12.1239680873863, 11.8713721895418, 
12.4863347873458, 12.021083930749, 11.0445348312836, 12.2744730611128, 
11.3400242569576, 12.8650052098443, 11.3539822949361, 12.3785156031973, 
13.2838567346754, 11.9812870123294, 10.8478700702981, 11.9727197829677, 
12.4289008359945, 11.4988433200491, 11.7223272516639, 12.6071277575023, 
12.3715245156524, 11.2746429928446, 12.4619766942024, 12.4510252726041, 
12.5719531419958, 11.6883571764051, 11.2537300035148, 12.7812335499978, 
11.6923101532807, 12.769157501828, 11.3968276448525, 12.3777195623096, 
12.9343182754766, 11.7318553349358, 11.0843893966019, 11.7759845182213, 
12.6456431791435, 11.8054009525151, 12.4140747921317, 12.9390983531654, 
12.5986862269444, 11.0501090785422, 12.3551394854017, 12.0466687024757, 
12.6829453129085, 11.8378916866666, 11.3464675790235), dim = c(20L, 
100L)))


par(mar=c(5, 4, 4, 2) - 1.9)
by(res.over$res.plot, INDICES = res.over$res.plot$var, FUN = function(xx) {

    plot(x = xx[, "trueval"], y = xx[, "xbar"], 
         col = as.character(xx[, "col"]), xlab = "observed values", 
         ylab = "imputed values", main = paste(xx[1, "var"], " (cov =", xx[1, "pct"],"%)"), 
         ylim = c(min(xx[, "binf"], na.rm = T), max(xx[, 
                                                       "bsup"], na.rm = T)))
    abline(0, 1)
    segments(x0 = xx[, "trueval"], x1 = xx[, "trueval"], 
             y0 = xx[, "binf"], y1 = xx[, "bsup"], 
             col = as.character(xx[, "col"]))
    legend("bottomright", legend = c("0-0.2", "0.2-0.4", 
                                     "0.4-0.6", "0.6-0.8", "0.8-1"), 
           col = c("blue", "green", heat.colors(3)[c(3, 
                                                     2, 1)]),
           bty = "n", lty = 1, horiz =TRUE, 
           cex = .8, lwd = 0.4)
  })

## ----fig.height = 4, fig.width = 4, fig.align = "center"----------------------
res.m <- choosem(res.pool.kmeans)

## ----eval=FALSE---------------------------------------------------------------
#  res.nbclust <- choosenbclust(res.pool.kmeans)

## ----fig.height = 4, fig.width = 4, fig.align = "center", echo=FALSE----------
res.nbclust <- list(nb.clust = 3L, crit = c(`2` = 0.195884403759201, `3` = 0.10576316420144, 
`4` = 0.145365200448241, `5` = 0.176785207030488))


plot(as.numeric(names(res.nbclust$crit)),res.nbclust$crit,xlab="nb clust",ylab="Total instability",type="b",xaxt = "n")
    axis(1, as.numeric(names(res.nbclust$crit)), as.numeric(names(res.nbclust$crit)))

## ----fig.width=7,fig.height=5,fig.align="center",results=FALSE, warning=FALSE, message=FALSE----
require(reshape2)
require(ggplot2)
dat.m = melt(data.frame(wine.na, part = as.factor(part)), id.var=c("part"))
ggplot(dat.m, aes(part, value, col = part)) +
  facet_wrap(variable~., scales="free_y") +
  geom_boxplot(width=0.7)

## ----fig.width=7,fig.height=5,fig.align="center", message=FALSE---------------
library(VIM)
pairsVIM(wine.na,
         pch = 21,
         bg = c("red", "green3", "blue")[part],
         cex = .2,
         gap = 0)

## ----fig.width=7, fig.height=5, fig.align="center"----------------------------
library(FactoMineR)
library(missMDA)
# merge the partition variable with the incomplete data set
data.pca <- cbind.data.frame(class = factor(part, levels = seq(nb.clust)),
                             wine.na)
# perform PCA with missing values by specifying where is the partition variable 
res.imputepca <- imputePCA(data.pca, quali.sup = 1)
res.pca <- PCA(res.imputepca$completeObs, quali.sup = 1, graph = FALSE)
plot(res.pca, habillage = 1)

## -----------------------------------------------------------------------------
library(clusterCrit)
res.crit <- extCriteria(part, ref, crit = "all")
round(unlist(res.crit), 2)

