#' @title Balance Statistics for `twang` Objects
#' 
#' @description
#' Generates balance statistics for `ps`, `mnps`, and `iptw` objects from \pkg{twang} and for `ps.cont` objects from \pkg{twangContinuous}.
#' 
#' @inheritParams bal.tab
#' @param x a `ps`, `mnps`, `iptw`, or `ps.cont` object; the output of a call to \pkgfun{twang}{ps}, \pkgfun{twang}{mnps}, \pkgfun{twang}{iptw} or \pkgfun{twangContinuous}{ps.cont}.
#' @param stop.method a string containing the names of the stopping methods used in the original call to `ps()`, `mnps()`, or `iptw()`. Examples include `"es.max"` or `"ks.mean"` for `ps` and `mnps` objects. `bal.tab()` will assess balance for the weights created by those stopping methods. The names can be abbreviated as long as the abbreviations are specific enough. If no stopping methods are provided, `bal.tab()` will default to displaying balance for all available stopping methods. Ignored for `ps.cont` objects.
#' @param distance an optional formula or data frame containing distance values (e.g., propensity scores) or a character vector containing their names. If a formula or variable names are specified, `bal.tab()` will look in the argument to `data`, if specified. The propensity scores generated by `ps()` and `iptw()` (but not `mnps()` or `ps.cont()`) are automatically included and named "prop.score.{stop.method}".
#' @param s.d.denom `character`; how the denominator for standardized mean differences should be calculated, if requested. See [col_w_smd()] for allowable options. Abbreviations allowed. If not specified, for `ps` objects, `bal.tab()` will use "treated" if the estimand of the call to `ps()` is the ATT and "pooled" if the estimand is the ATE; for `mnps` objects, `bal.tab()` will use "treated" if `treatATT` was specified in the original call to `mnps` and "pooled" otherwise. Use "all" to get the same values computed by `bal.table()` in \pkg{twang}.
#' @param s.weights Optional; either a vector containing sampling weights for each unit or a string containing the name of the sampling weight variable in `data`. These function like regular weights except that both the adjusted and unadjusted samples will be weighted according to these weights if weights are used. If `sampw` was supplied in the call to `ps()`, `mnps()`, `iptw()`, or `ps.cont()`, they will automatically be supplied to `s.weights` and do not need be specified again (though there is no harm if they are).
#' 
#' @returns
#' For binary or continuous point treatments, if clusters are not specified, an object of class `"bal.tab"` containing balance summaries for the `ps` object. See [bal.tab()] for details.
#' 
#' If clusters are specified, an object of class `"bal.tab.cluster"` containing balance summaries within each cluster and a summary of balance across clusters. See [`bal.tab.cluster()`][class-bal.tab.cluster] for details.
#' 
#' If `mnps()` is used with multi-category treatments, an object of class `"bal.tab.multi"` containing balance summaries for each pairwise treatment comparison and a summary of balance across pairwise comparisons. See [`bal.tab.multi()`][class-bal.tab.multi] for details.
#' 
#' @details
#' `bal.tab.ps()` generates a list of balance summaries for the input object given, and functions similarly to \pkgfun{twang}{bal.table}. The variances used in the denominator of the standardized mean differences computed in `twang::bal.table()` are weighted and computed using `survey::svyvar()` and are unweighted here (except when `s.weights` are specified, in which case [col_w_sd()] is used). \pkg{twang} also uses "all" as the default `s.d.denom` when the estimand is the ATE; the default here is "pooled". For these reasons, results may differ slightly between the two packages. 
#' 
#' @seealso
#' * [bal.tab()] for details of calculations.
#' * [`bal.tab.cluster()`][class-bal.tab.cluster] for more information on clustered data.
#' * [`bal.tab.multi()`][class-bal.tab.multi] for more information on multi-category treatments.
#' * [`bal.tab.msm()`][class-bal.tab.msm] for more information on longitudinal treatments.
#' 
#' @examplesIf requireNamespace("twang", quietly = TRUE)
#' \donttest{library(twang); data("lalonde", package = "cobalt")
#' 
#' ## Using ps() for generalized boosted modeling
#' ps.out <- ps(treat ~ age + educ + married + race +
#'                  nodegree + re74 + re75, data = lalonde, 
#'              stop.method = c("ks.mean", "es.mean"), 
#'              estimand = "ATT", verbose = FALSE)
#' 
#' bal.tab(ps.out, stop.method = "ks.mean", un = TRUE, 
#'         m.threshold = .1, disp.ks = TRUE)
#' }

#' @exportS3Method bal.tab ps
bal.tab.ps <-         function(x, stop.method,
                               stats, int = FALSE, poly = 1, distance = NULL, addl = NULL, data = NULL, continuous, binary, s.d.denom, thresholds = NULL, weights = NULL, cluster = NULL, imp = NULL, pairwise = TRUE, s.weights = NULL, abs = FALSE, subset = NULL, quick = TRUE,
                               ...) {
    
    tryCatch(args <- c(as.list(environment()), list(...))[-1], error = function(e) .err(conditionMessage(e)))
    
    #Adjustments to arguments
    
    args[vapply(args, rlang::is_missing, logical(1L))] <- NULL
    args[vapply(args, is_null, logical(1L)) & names(args) %nin% names(match.call())[-1]] <- NULL
    
    #Initializing variables
    X <- do.call("x2base.ps", c(list(x), args), quote = TRUE)
    
    args[names(args) %in% names(X)] <- NULL
    
    X <- assign.X.class(X)
    
    out <- do.call("base.bal.tab", c(list(X), args),
                   quote = TRUE)
    return(out)
}

#' @exportS3Method bal.tab mnps
bal.tab.mnps <- bal.tab.ps

#' @exportS3Method bal.tab ps.cont
bal.tab.ps.cont <- bal.tab.ps

#' @exportS3Method bal.tab iptw
bal.tab.iptw <- bal.tab.ps
