% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/referencePoint2Polygon.R
\name{referencePoint2Polygon}
\alias{referencePoint2Polygon}
\title{Create a Rectangular Polygon Using Planar XY Coordinates}
\usage{
referencePoint2Polygon(
  x = NULL,
  id = NULL,
  dateTime = NULL,
  point.x = NULL,
  point.y = NULL,
  direction = NULL,
  StartLocation = "UL",
  UpDownRepositionLen = 1,
  LeftRightRepositionLen = 1,
  CenterPoint = FALSE,
  MidPoints = FALSE,
  immobThreshold = 0,
  parallel = FALSE,
  nCores = (parallel::detectCores()/2),
  modelOrientation = 90
)
}
\arguments{
\item{x}{Data frame or list of data frames containing real-time-location 
point data.}

\item{id}{Vector of length nrow(data.frame(x)) or singular character data, 
detailing the relevant colname in x, that denotes what unique ids for 
tracked individuals will be used. If argument == NULL, the function 
assumes a column with the colname "id" exists in x. Defaults to NULL.}

\item{dateTime}{Vector of length nrow(data.frame(x)) or singular character 
data, detailing the relevant colname in x, that denotes what dateTime 
information will be used. If argument == NULL, the function assumes a 
column with the colname "dateTime" exists in x. Defaults to NULL.}

\item{point.x}{Vector of length nrow(data.frame(x)) or singular character 
data, detailing the relevant colname in x, that denotes what planar-x or 
longitude coordinate information will be used. If argument == NULL, the 
function assumes a column with the colname "x" exists in x. Defaults to 
NULL.}

\item{point.y}{Vector of length nrow(data.frame(x)) or singular character 
data, detailing the relevant colname in x, that denotes what planar-y or 
lattitude coordinate information will be used. If argument == NULL, the 
function assumes a column with the colname "y" exists in x. Defaults to 
NULL.}

\item{direction}{Numerical vector of length nrow(data.frame(x)) or singular 
character data detailing the relevant colname in x, that denotes what 
movement-direction information will be used. Observations in this vector
represent the direction (in degrees) that tracked individuals moved to 
reach their position at each time point, NOT the direction that they will
move to reach their subsequent position (i.e., values represent known
orientations at each time point). Note that for the purposes of this 
function, observations of 0, 90, 180, and 270 degrees indicates that an 
individual moved straight Eastward, Northward, Westward, and Southward,
respectively. If NULL, direction will be calculated using observed 
point-locations. Defaults to NULL.}

\item{StartLocation}{Character string taking the values "UL," "UR," "DL," or
"DR" describing where the reference point (i.e., point corresponding to 
xy-coordinates in the data set) lies on the rectangle that this function 
will delineate. Defaults to "UL."}

\item{UpDownRepositionLen}{Numerical. Describes the height, in planar units 
(e.g., meters) of the output polygon. Planar units are inherent to the 
real-time-location input. Defaults to 1.}

\item{LeftRightRepositionLen}{Numerical. Describes the width, in planar 
units (e.g., meters) of the output polygon. Planar units are inherent to 
the real-time-location input. Defaults to 1.}

\item{CenterPoint}{Logical. If TRUE, in addition to the xy-coordinates for 
each polygon vertex, xy-coordinates for centroid of each polygon will be 
reported in the output. Defaults to FALSE.}

\item{MidPoints}{Logical. If TRUE, in addition to the xy-coordinates for 
each polygon vertex, xy-coordinates for mid-point of each polygon edge 
will be reported in the output. Defaults to FALSE.}

\item{immobThreshold}{Numerical. Describes what we call, the immobility 
threshold, which is a movement distance (in planar units) within which 
we assume individuals’ physical locations and orientations remain 
unchanged. This immobility threshold allows us to discount observed 
movements so miniscule that the majority of animals’ physical-space 
usage is likely unaffected (e.g., head shaking). Defaults to 0.}

\item{parallel}{Logical. If TRUE, sub-functions within the 
referencePoint2Polygon wrapper will be parallelized. Defaults to FALSE.}

\item{nCores}{Integer. Describes the number of cores to be dedicated to 
parallel processes. Defaults to half of the maximum number of cores 
available (i.e., (parallel::detectCores()/2)).}

\item{modelOrientation}{Numerical. Describes the relative orientation (in 
degrees) of a planar model (see vignette or Farthing et al. in Press 
(note: when this manuscript is officially published, we will update this 
citation/reference information)) describing vertex locations relative to
tracking-device point-locations. Defaults to 90.}
}
\value{
Output is a data frame with the following columns:
      
   \item{id}{Unique ID of tracked individuals.}
   \item{cornerPoint...x}{Planar x coordinates of polygon-corner vertices.}
   \item{cornerPoint...y}{Planar y coordinates of polygon-corner vertices.}
   \item{startLocation}{Describes the location of input point-locations in 
   the vertex outputs. see \code{StartLocation} argument.}
   \item{upDownRepositionLength}{Describes the vertical movement of 
   point-locations on planar models. see \code{UpDownRepositionLen} 
   argument.}
   \item{leftRightRepositionLength}{Describes the horizontal movement of 
   point-locations on planar models. see \code{leftRightRepositionLen} 
   argument.}
   \item{immob}{If "0", distance between observed movements is 
   < \code{immobThreshold}.}
   \item{immobThreshold}{Returns the value from the \code{immobThreshold}
   argument.}
   \item{dateTime}{Timepoint at which polygons were observed.}
   \item{dt}{The the time between reported xy coordinates in row i to row 
   i + 1 in each individuals' movement path.}
   
   If MidPoints or CenterPoints == TRUE, additional columns will be appended
   to output data frame.
}
\description{
This function creates a square/rectangular polygon from a single reference 
   point by translating its location multiple times using the same method 
   used in repositionReferencePoint. For example, even though calves in our 
   study (see data(calves2018)) were only equiped with RFID tags on their 
   left ear. With this function, we can create polygons that account for the
   total space used by each individual at each time step.This function is 
   different from similar point-to-polygon functions for two reasons:
   1.) It does not assume points lie within the center of the polygon. 
       Rather, the reference point must be a corner of the polygon 
       (Note: "UL" denotes that the reference point lies on the upper-left 
       corner of the polygon, "UR" denotes that reference point lies on the 
       upper-right corner of the polygon,"DL" denotes that reference point 
       lies on the down-left corner of the polygon, "DR" denotes that 
       reference point lies on the down-left corner of the polygon). Note 
       that if you want the reference point to be at the center of the 
       polygon, you can first translate the reference point to a central 
       location on tracked individuals using repositionReferencePoint.
   2.) Polygon angles/directionality are based on observed movements of 
       tracked individuals or gyroscope data.
}
\details{
Currently, this function only supports input data with coordinates 
   representing planar ('Euclidean') space (e.g. units of meters).

In the output, point1.x and point1.y represent the xy coordinates from the 
   input file. Point2-n coordinates move in a clockwise direction from 
   point1. For example: if point1 is located on the upper left ("UL") corner
   of the polygon, point2 would be on the upper right corner, point3 on the 
   bottom right, and point 4 on the bottom left.

If distance == NULL, then function will require information (dist, dx, dy) 
   from 2 points on an individual's path to work properly. Because of this, 
   when no gyroscopic data are provided, at least the first point in each 
   individual's path will be removed (the function will report NAs for 
   adjusted locations). Also note that if the distance between an 
   individual's first point in their path and the second one is 0, the 
   function will also report NAs for the second point's adjusted 
   coordinates. The first non-NA values will only be reported for the 
   instance where dist > 0.

Note that populating the direction argument with gyroscopic accelerometer 
   data (or data collected using similar devices) collected concurrently 
   with point-locations allows us to overcome a couple of assumptions 
   associated with using point-locations alone.
   
   First, unless the direction argument is specifically given (i.e., 
   direction != NULL), vertex locations in output are subject to the 
   assumption that dt values are sufficiently small to capture individuals'
   orientations (i.e., individuals do not face unknown directions 
   inbetween observed relocations). If input was previously processed using 
   tempAggregate with resolutionLevel == "reduced," dt > secondAgg indicates
   that tracked individuals were missing in the original dataset for a 
   period of time. In this case, the assumption that individuals are facing 
   a given direction because they moved from the previous timepoint may not 
   be accurate. Consider removing these rows (rows following one with 
   dt > secondAgg; remember that dt indicates the time between reported xy 
   coordinates in row i to row i + 1) from your data set.

   Second, unless the direction argument is specifically given (i.e., 
   direction != NULL), this function assumes tracked individuals are always 
   forward-facing. This is because by observing only a single point on each 
   individual, we cannot ascertain the true positioning of individuals' 
   bodies. For example, even if we know a point-location moved x distance in
   a 90-degree direction, from this information alone we cannot determine 
   what direction said individual was facing at the time (e.g., this could
   be an example of forward, bawckward, or sideward movement). However, 
   gyroscopic data (or data collected using similar devices) can tell us 
   absolute movement directions, as opposed to relative ones.
}
\examples{
\donttest{
data("calves")
calves.dateTime<-datetime.append(calves, date = calves$date,
   time = calves$time) #add dateTime identifiers for location fixes.

calves.agg<-tempAggregate(calves.dateTime, id = calves.dateTime$calftag, 
   dateTime = calves.dateTime$dateTime, point.x = calves.dateTime$x, 
   point.y = calves.dateTime$y, secondAgg = 300, extrapolate.left = FALSE, 
   extrapolate.right = FALSE, resolutionLevel = "reduced", parallel = FALSE, 
   na.rm = TRUE, smooth.type = 1) #smooth to 5-min fix intervals.

calf_heads <- referencePoint2Polygon(x = calves.agg,
   id = calves.agg$id, dateTime = calves.agg$dateTime,
   point.x = calves.agg$x, point.y = calves.agg$y, direction = NULL,
   StartLocation = "DL", UpDownRepositionLen = 0.333, LeftRightRepositionLen = 0.333,
   CenterPoint = FALSE, MidPoints = FALSE, immobThreshold = 0.1, parallel = FALSE,
   modelOrientation = 90)
   }
}
\references{
Farthing, T.S., Dawson, D.E., Sanderson, M.W., and Lanzas, 
   C. in Press. Accounting for space and uncertainty in real-time-location-
   system-derived contact networks. Ecology and Evolution.
}
\keyword{data-processing}
\keyword{location}
\keyword{planar}
\keyword{point}
\keyword{polygon}
