% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/confusion_matrix.R
\name{confusion_matrix}
\alias{confusion_matrix}
\title{Create a confusion matrix}
\usage{
confusion_matrix(
  targets,
  predictions,
  metrics = list(),
  positive = 2,
  c_levels = NULL,
  do_one_vs_all = TRUE,
  parallel = FALSE
)
}
\arguments{
\item{targets}{\code{vector} with true classes. Either \code{numeric} or \code{character}.}

\item{predictions}{\code{vector} with predicted classes. Either \code{numeric} or \code{character}.}

\item{metrics}{\code{list} for enabling/disabling metrics.

  E.g. \code{list("Accuracy" = TRUE)} would add the regular accuracy metric,
  whie \code{list("F1" = FALSE)} would remove the \code{F1} metric.
  Default values (TRUE/FALSE) will be used for the remaining available metrics.

  You can enable/disable all metrics at once by including
  \code{"all" = TRUE/FALSE} in the \code{list}. This is done prior to enabling/disabling
  individual metrics, why for instance \code{list("all" = FALSE, "Accuracy" = TRUE)}
  would return only the \code{Accuracy} metric.

  The \code{list} can be created with
  \code{\link[cvms:binomial_metrics]{binomial_metrics()}} or
  \code{\link[cvms:multinomial_metrics]{multinomial_metrics()}}.

  Also accepts the string \code{"all"}.}

\item{positive}{Level from \code{`targets`} to predict.
 Either as character (\emph{preferable}) or level index (\code{1} or \code{2} - alphabetically). (\strong{Two-class only})

 E.g. if we have the levels \code{"cat"} and \code{"dog"} and we want \code{"dog"} to be the positive class,
 we can either provide \code{"dog"} or \code{2}, as alphabetically, \code{"dog"} comes after \code{"cat"}.

 \strong{Note:} For \emph{reproducibility}, it's preferable to \strong{specify the name directly}, as
 different \code{\link[base:locales]{locales}} may sort the levels differently.}

\item{c_levels}{\code{vector} with categorical levels in the targets. Should have same type as \code{`targets`}.
 If \code{NULL}, they are inferred from \code{`targets`}.

 N.B. the levels are sorted alphabetically. When \code{`positive`} is numeric (i.e. an index),
 it therefore still refers to the index of the alphabetically sorted levels.}

\item{do_one_vs_all}{Whether to perform \emph{one-vs-all} evaluations
 when working with more than 2 classes (multiclass).

 If you are only interested in the confusion matrix,
 this allows you to skip most of the metric calculations.}

\item{parallel}{Whether to perform the one-vs-all evaluations in parallel. (Logical)

 N.B. This only makes sense when you have a lot of classes or a very large dataset.

 Remember to register a parallel backend first.
 E.g. with \code{doParallel::registerDoParallel}.}
}
\value{
\code{tibble} with:

 Nested \strong{confusion matrix} (tidied version)

 Nested confusion matrix (\strong{table})

 The \strong{Positive Class}.

 Multiclass only: Nested \strong{Class Level Results} with the two-class metrics,
 the nested confusion matrices, and the \strong{Support} metric, which is a
 count of the class in the target column and is used for the weighted average metrics.

 The following metrics are available (see \code{`metrics`}):

 \subsection{Two classes or more}{

 \tabular{rrr}{
  \strong{Metric} \tab \strong{Name} \tab \strong{Default} \cr
  Balanced Accuracy \tab "Balanced Accuracy" \tab Enabled \cr
  Accuracy \tab "Accuracy" \tab Disabled \cr
  F1 \tab "F1" \tab Enabled \cr
  Sensitivity \tab "Sensitivity" \tab Enabled \cr
  Specificity \tab "Specificity" \tab Enabled \cr
  Positive Predictive Value \tab "Pos Pred Value" \tab Enabled \cr
  Negative Predictive Value \tab "Neg Pred Value" \tab Enabled \cr
  Kappa \tab "Kappa" \tab Enabled \cr
  Matthews Correlation Coefficient \tab "MCC" \tab Enabled \cr
  Detection Rate \tab "Detection Rate" \tab Enabled \cr
  Detection Prevalence \tab "Detection Prevalence" \tab Enabled \cr
  Prevalence \tab "Prevalence" \tab Enabled \cr
  False Negative Rate \tab "False Neg Rate" \tab Disabled \cr
  False Positive Rate \tab "False Pos Rate" \tab Disabled \cr
  False Discovery Rate \tab "False Discovery Rate" \tab Disabled \cr
  False Omission Rate \tab "False Omission Rate" \tab Disabled \cr
  Threat Score \tab "Threat Score" \tab Disabled \cr
 }

 The \strong{Name} column refers to the name used in the package.
 This is the name in the output and when enabling/disabling in \code{`metrics`}.
 }

 \subsection{Three classes or more}{

 The metrics mentioned above (excluding \code{MCC})
 has a weighted average version (disabled by default; weighted by the \strong{Support}).

 In order to enable a weighted metric, prefix the metric name with \code{"Weighted "} when specifying \code{`metrics`}.

 E.g. \code{metrics = list("Weighted Accuracy" = TRUE)}.

 \tabular{rrr}{
  \strong{Metric} \tab \strong{Name} \tab \strong{Default} \cr
  Overall Accuracy \tab "Overall Accuracy" \tab Enabled \cr
  Weighted * \tab "Weighted *" \tab Disabled \cr
  Multiclass MCC \tab "MCC" \tab Enabled \cr
 }
 }
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

 Creates a confusion matrix from targets and predictions.
 Calculates associated metrics.

 Multiclass results are based on one-vs-all evaluations.
 Both regular averaging and weighted averaging are available. Also calculates the \code{Overall Accuracy}.

 \strong{Note}: In most cases you should use \code{\link[cvms:evaluate]{evaluate()}} instead. It has additional metrics and
 works in \code{magrittr} pipes (e.g. \code{\%>\%}) and with \code{\link[dplyr:group_by]{dplyr::group_by()}}.
 \code{confusion_matrix()} is more lightweight and may be preferred in programming when you don't need the extra stuff
 in \code{\link[cvms:evaluate]{evaluate()}}.
}
\details{
The following formulas are used for calculating the metrics:

 \code{Sensitivity = TP / (TP + FN)}

 \code{Specificity = TN / (TN + FP)}

 \code{Pos Pred Value = TP / (TP + FP)}

 \code{Neg Pred Value = TN / (TN + FN)}

 \code{Balanced Accuracy = (Sensitivity + Specificity) / 2}

 \code{Accuracy = (TP + TN) / (TP + TN + FP + FN)}

 \code{Overall Accuracy = Correct / (Correct + Incorrect)}

 \code{F1 = 2 * Pos Pred Value * Sensitivity / (Pos Pred Value + Sensitivity)}

 \code{MCC = ((TP * TN) - (FP * FN)) / sqrt((TP + FP) * (TP + FN) * (TN + FP) * (TN + FN))}

 Note for \code{MCC}: Formula is for the \emph{binary} case. When the denominator is \code{0},
 we set it to \code{1} to avoid \code{NaN}.
 See the \code{metrics} vignette for the multiclass version.

 \code{Detection Rate = TP / (TP + FN + TN + FP)}

 \code{Detection Prevalence = (TP + FP) / (TP + FN + TN + FP)}

 \code{Threat Score = TP / (TP + FN + FP)}

 \code{False Neg Rate = 1 - Sensitivity}

 \code{False Pos Rate = 1 - Specificity}

 \code{False Discovery Rate = 1 - Pos Pred Value}

 \code{False Omission Rate = 1 - Neg Pred Value}

 For \strong{Kappa} the counts (\code{TP}, \code{TN}, \code{FP}, \code{FN}) are normalized to percentages (summing to 1).
 Then the following is calculated:

 \code{p_observed = TP + TN}

 \code{p_expected = (TN + FP) * (TN + FN) + (FN + TP) * (FP + TP)}

 \code{Kappa = (p_observed - p_expected) / (1 - p_expected)}
}
\examples{
# Attach cvms
library(cvms)

# Two classes

# Create targets and predictions
targets <- c(0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1)
predictions <- c(1, 1, 0, 0, 0, 1, 1, 1, 0, 1, 0, 0)

# Create confusion matrix with default metrics
cm <- confusion_matrix(targets, predictions)
cm
cm[["Confusion Matrix"]]
cm[["Table"]]

# Three classes

# Create targets and predictions
targets <- c(0, 1, 2, 1, 0, 1, 2, 1, 0, 1, 2, 1, 0)
predictions <- c(2, 1, 0, 2, 0, 1, 1, 2, 0, 1, 2, 0, 2)

# Create confusion matrix with default metrics
cm <- confusion_matrix(targets, predictions)
cm
cm[["Confusion Matrix"]]
cm[["Table"]]

# Enabling weighted accuracy

# Create confusion matrix with Weighted Accuracy enabled
cm <- confusion_matrix(targets, predictions,
  metrics = list("Weighted Accuracy" = TRUE)
)
cm
}
\seealso{
Other evaluation functions: 
\code{\link{binomial_metrics}()},
\code{\link{evaluate_residuals}()},
\code{\link{evaluate}()},
\code{\link{gaussian_metrics}()},
\code{\link{multinomial_metrics}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{evaluation functions}
