#' Recalculate locus metrics when individuals or populations are deleted from a genlight \{adegenet\} object
#'
#' When individuals are deleted from a genlight object generated by DArT, the locus metrics no longer
#' apply. For example, the Call Rate may be different considering the subset of individuals, compared
#' with the full set. This script recalculates those affected locus metrics, namely, avgPIC, CallRate,
#' freqHets, freqHomRef, freqHomSnp, OneRatioRef, OneRatioSnp, PICRef and PICSnp. Metrics that remain
#' unaltered are RepAvg and TrimmedSeq as they are unaffected by the removal of individuals.
#' 
#' The script optionally removes resultant monomorphic loci or loci
#' with all values missing and deletes them (using gl.filter.monomorphs.r). 
#' 
#' The script returns a genlight object with the recalculated locus metadata.
#'
#' @param x -- name of the genlight object containing SNP genotypes [required]
#' @param mono.rm -- if TRUE, removes monomorphic loci [default FALSE]
#' @param verbose -- verbosity: 0, silent or fatal errors; 1, begin and end; 2, progress log ; 3, progress and results summary; 5, full report [default 2 or as specified using gl.set.verbosity]
#' @return A genlight object with the recalculated locus metadata
#' @export
#' @author Arthur Georges (Post to \url{https://groups.google.com/d/forum/dartr})
#' @examples
#'   gl <- gl.recalc.metrics(testset.gl, verbose=2)
#' @seealso \code{\link{gl.filter.monomorphs}}

gl.recalc.metrics <- function(x, mono.rm =FALSE, verbose=NULL){
  
# TRAP COMMAND, SET VERSION
  
  funname <- match.call()[[1]]
  build <- "Jacob"

# SET VERBOSITY
  
  if (is.null(verbose)){ 
    if(!is.null(x@other$verbose)){ 
      verbose <- x@other$verbose
    }else { 
      verbose <- 2
    }
  } 
  
  if (verbose < 0 | verbose > 5){
    cat(paste("  Warning: Parameter 'verbose' must be an integer between 0 [silent] and 5 [full report], set to 2\n"))
    verbose <- 2
  }
  
# FLAG SCRIPT START
  
  if (verbose >= 1){
    if(verbose==5){
      cat("Starting",funname,"[ Build =",build,"]\n")
    } else {
      cat("Starting",funname,"\n")
    }
  }
  
# STANDARD ERROR CHECKING
  
  if(class(x)!="genlight") {
    stop("Fatal Error: genlight object required!\n")
  }
  
  if (all(x@ploidy == 1)){
    if (verbose >= 2){cat("  Processing  Presence/Absence (SilicoDArT) data\n")}
    data.type <- "SilicoDArT"
  } else if (all(x@ploidy == 2)){
    if (verbose >= 2){cat("  Processing a SNP dataset\n")}
    data.type <- "SNP"
  } else {
    stop("Fatal Error: Ploidy must be universally 1 (fragment P/A data) or 2 (SNP data)")
  }

# DO THE JOB

# Recalculate statistics
  
  if (data.type == 'SNP'){
    x <- utils.recalc.avgpic(x,verbose=verbose)
    x <- utils.recalc.callrate(x,verbose=verbose)
    x <- utils.recalc.maf(x,verbose=verbose)
  }
  if (data.type == 'SilicoDArT'){
    x <- utils.recalc.avgpic(x,verbose=verbose)
    x <- utils.recalc.callrate(x,verbose=verbose)
  }
  
  if (verbose >= 2) {
    cat("  Locus metrics recalculated\n")
  }

  if(mono.rm) {
    x <- gl.filter.monomorphs(x,verbose=0)
    if (verbose >= 2) {
      cat("  Monomorphic loci deleted\n")
    }
  }

# ADD TO HISTORY
  nh <- length(x@other$history)
  x@other$history[[nh + 1]] <- match.call() 
  
# FLAG SCRIPT END

  if (verbose > 0) {
    cat("Completed:",funname,"\n")
  }

  return (x)

}  
