% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/categorize.R
\name{categorize}
\alias{categorize}
\alias{categorize.numeric}
\alias{categorize.data.frame}
\title{Recode (or "cut" / "bin") data into groups of values.}
\usage{
categorize(x, ...)

\method{categorize}{numeric}(
  x,
  split = "median",
  n_groups = NULL,
  range = NULL,
  lowest = 1,
  breaks = "exclusive",
  labels = NULL,
  verbose = TRUE,
  ...
)

\method{categorize}{data.frame}(
  x,
  select = NULL,
  exclude = NULL,
  split = "median",
  n_groups = NULL,
  range = NULL,
  lowest = 1,
  breaks = "exclusive",
  labels = NULL,
  append = FALSE,
  ignore_case = FALSE,
  regex = FALSE,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{x}{A (grouped) data frame, numeric vector or factor.}

\item{...}{not used.}

\item{split}{Character vector, indicating at which breaks to split variables,
or numeric values with values indicating breaks. If character, may be one
of \code{"median"}, \code{"mean"}, \code{"quantile"}, \code{"equal_length"}, or \code{"equal_range"}.
\code{"median"} or \code{"mean"} will return dichotomous variables, split at their
mean or median, respectively. \code{"quantile"} and \code{"equal_length"} will split
the variable into \code{n_groups} groups, where each group refers to an interval
of a specific range of values. Thus, the length of each interval will be
based on the number of groups. \code{"equal_range"} also splits the variable
into multiple groups, however, the length of the interval is given, and
the number of resulting groups (and hence, the number of breaks) will be
determined by how many intervals can be generated, based on the full range
of the variable.}

\item{n_groups}{If \code{split} is \code{"quantile"} or \code{"equal_length"}, this defines
the number of requested groups (i.e. resulting number of levels or values)
for the recoded variable(s). \code{"quantile"} will define intervals based
on the distribution of the variable, while \code{"equal_length"} tries to
divide the range of the variable into pieces of equal length.}

\item{range}{If \code{split = "equal_range"}, this defines the range of values
that are recoded into a new value.}

\item{lowest}{Minimum value of the recoded variable(s). If \code{NULL} (the default),
for numeric variables, the minimum of the original input is preserved. For
factors, the default minimum is \code{1}. For \code{split = "equal_range"}, the
default minimum is always \code{1}, unless specified otherwise in \code{lowest}.}

\item{breaks}{Character, indicating whether breaks for categorizing data are
\code{"inclusive"} (values indicate the \emph{upper} bound of the \emph{previous} group or
interval) or \code{"exclusive"} (values indicate the \emph{lower} bound of the \emph{next}
group or interval to begin). Use \code{labels = "range"} to make this behaviour
easier to see.}

\item{labels}{Character vector of value labels. If not \code{NULL}, \code{categorize()}
will returns factors instead of numeric variables, with \code{labels} used
for labelling the factor levels. Can also be \code{"mean"}, \code{"median"},
\code{"range"} or \code{"observed"} for a factor with labels as the mean/median,
the requested range (even if not all values of that range are present in
the data) or observed range (range of the actual recoded values) of each
group. See 'Examples'.}

\item{verbose}{Toggle warnings.}

\item{select}{Variables that will be included when performing the required
tasks. Can be either
\itemize{
\item a variable specified as a literal variable name (e.g., \code{column_name}),
\item a string with the variable name (e.g., \code{"column_name"}), or a character
vector of variable names (e.g., \code{c("col1", "col2", "col3")}),
\item a formula with variable names (e.g., \code{~column_1 + column_2}),
\item a vector of positive integers, giving the positions counting from the left
(e.g. \code{1} or \code{c(1, 3, 5)}),
\item a vector of negative integers, giving the positions counting from the
right (e.g., \code{-1} or \code{-1:-3}),
\item one of the following select-helpers: \code{starts_with()}, \code{ends_with()},
\code{contains()}, a range using \code{:} or \code{regex("")}. \code{starts_with()},
\code{ends_with()}, and  \code{contains()} accept several patterns, e.g
\code{starts_with("Sep", "Petal")}.
\item or a function testing for logical conditions, e.g. \code{is.numeric()} (or
\code{is.numeric}), or any user-defined function that selects the variables
for which the function returns \code{TRUE} (like: \code{foo <- function(x) mean(x) > 3}),
\item ranges specified via literal variable names, select-helpers (except
\code{regex()}) and (user-defined) functions can be negated, i.e. return
non-matching elements, when prefixed with a \code{-}, e.g. \code{-ends_with("")},
\code{-is.numeric} or \code{-(Sepal.Width:Petal.Length)}. \strong{Note:} Negation means
that matches are \emph{excluded}, and thus, the \code{exclude} argument can be
used alternatively. For instance, \code{select=-ends_with("Length")} (with
\code{-}) is equivalent to \code{exclude=ends_with("Length")} (no \code{-}). In case
negation should not work as expected, use the \code{exclude} argument instead.
}

If \code{NULL}, selects all columns. Patterns that found no matches are silently
ignored, e.g. \code{extract_column_names(iris, select = c("Species", "Test"))}
will just return \code{"Species"}.}

\item{exclude}{See \code{select}, however, column names matched by the pattern
from \code{exclude} will be excluded instead of selected. If \code{NULL} (the default),
excludes no columns.}

\item{append}{Logical or string. If \code{TRUE}, recoded or converted variables
get new column names and are appended (column bind) to \code{x}, thus returning
both the original and the recoded variables. The new columns get a suffix,
based on the calling function: \code{"_r"} for recode functions, \code{"_n"} for
\code{to_numeric()}, \code{"_f"} for \code{to_factor()}, or \code{"_s"} for
\code{slide()}. If \code{append=FALSE}, original variables in \code{x} will be
overwritten by their recoded versions. If a character value, recoded
variables are appended with new column names (using the defined suffix) to
the original data frame.}

\item{ignore_case}{Logical, if \code{TRUE} and when one of the select-helpers or
a regular expression is used in \code{select}, ignores lower/upper case in the
search pattern when matching against variable names.}

\item{regex}{Logical, if \code{TRUE}, the search pattern from \code{select} will be
treated as regular expression. When \code{regex = TRUE}, select \emph{must} be a
character string (or a variable containing a character string) and is not
allowed to be one of the supported select-helpers or a character vector
of length > 1. \code{regex = TRUE} is comparable to using one of the two
select-helpers, \code{select = contains("")} or \code{select = regex("")}, however,
since the select-helpers may not work when called from inside other
functions (see 'Details'), this argument may be used as workaround.}
}
\value{
\code{x}, recoded into groups. By default \code{x} is numeric, unless \code{labels}
is specified. In this case, a factor is returned, where the factor levels
(i.e. recoded groups are labelled accordingly.
}
\description{
This functions divides the range of variables into intervals and recodes
the values inside these intervals according to their related interval.
It is basically a wrapper around base R's \code{cut()}, providing a simplified
and more accessible way to define the interval breaks (cut-off values).
}
\section{Splits and breaks (cut-off values)}{
Breaks are by default \emph{exclusive}, this means that these values indicate
the lower bound of the next group or interval to begin. Take a simple
example, a numeric variable with values from 1 to 9. The median would be 5,
thus the first interval ranges from 1-4 and is recoded into 1, while 5-9
would turn into 2 (compare \code{cbind(1:9, categorize(1:9))}). The same variable,
using \code{split = "quantile"} and \code{n_groups = 3} would define breaks at 3.67
and 6.33 (see \code{quantile(1:9, probs = c(1/3, 2/3))}), which means that values
from 1 to 3 belong to the first interval and are recoded into 1 (because
the next interval starts at 3.67), 4 to 6 into 2 and 7 to 9 into 3.

The opposite behaviour can be achieved using \code{breaks = "inclusive"}, in which
case
}

\section{Recoding into groups with equal size or range}{
\code{split = "equal_length"} and \code{split = "equal_range"} try to divide the
range of \code{x} into intervals of similar (or same) length. The difference is
that \code{split = "equal_length"} will divide the range of \code{x} into \code{n_groups}
pieces and thereby defining the intervals used as breaks (hence, it is
equivalent to \code{cut(x, breaks = n_groups)}), while  \code{split = "equal_range"}
will cut \code{x} into intervals that all have the length of \code{range}, where the
first interval by defaults starts at \code{1}. The lowest (or starting) value
of that interval can be defined using the \code{lowest} argument.
}

\section{Selection of variables - the \code{select} argument}{

For most functions that have a \code{select} argument (including this function),
the complete input data frame is returned, even when \code{select} only selects
a range of variables. That is, the function is only applied to those variables
that have a match in \code{select}, while all other variables remain unchanged.
In other words: for this function, \code{select} will not omit any non-included
variables, so that the returned data frame will include all variables
from the input data frame.
}

\examples{
set.seed(123)
x <- sample(1:10, size = 50, replace = TRUE)

table(x)

# by default, at median
table(categorize(x))

# into 3 groups, based on distribution (quantiles)
table(categorize(x, split = "quantile", n_groups = 3))

# into 3 groups, user-defined break
table(categorize(x, split = c(3, 5)))

set.seed(123)
x <- sample(1:100, size = 500, replace = TRUE)

# into 5 groups, try to recode into intervals of similar length,
# i.e. the range within groups is the same for all groups
table(categorize(x, split = "equal_length", n_groups = 5))

# into 5 groups, try to return same range within groups
# i.e. 1-20, 21-40, 41-60, etc. Since the range of "x" is
# 1-100, and we have a range of 20, this results into 5
# groups, and thus is for this particular case identical
# to the previous result.
table(categorize(x, split = "equal_range", range = 20))

# return factor with value labels instead of numeric value
set.seed(123)
x <- sample(1:10, size = 30, replace = TRUE)
categorize(x, "equal_length", n_groups = 3)
categorize(x, "equal_length", n_groups = 3, labels = c("low", "mid", "high"))

# cut numeric into groups with the mean or median as a label name
x <- sample(1:10, size = 30, replace = TRUE)
categorize(x, "equal_length", n_groups = 3, labels = "mean")
categorize(x, "equal_length", n_groups = 3, labels = "median")

# cut numeric into groups with the requested range as a label name
# each category has the same range, and labels indicate this range
categorize(mtcars$mpg, "equal_length", n_groups = 5, labels = "range")
# in this example, each category has the same range, but labels only refer
# to the ranges of the actual values (present in the data) inside each group
categorize(mtcars$mpg, "equal_length", n_groups = 5, labels = "observed")
}
\seealso{
\itemize{
\item Functions to rename stuff: \code{\link[=data_rename]{data_rename()}}, \code{\link[=data_rename_rows]{data_rename_rows()}}, \code{\link[=data_addprefix]{data_addprefix()}}, \code{\link[=data_addsuffix]{data_addsuffix()}}
\item Functions to reorder or remove columns: \code{\link[=data_reorder]{data_reorder()}}, \code{\link[=data_relocate]{data_relocate()}}, \code{\link[=data_remove]{data_remove()}}
\item Functions to reshape, pivot or rotate data frames: \code{\link[=data_to_long]{data_to_long()}}, \code{\link[=data_to_wide]{data_to_wide()}}, \code{\link[=data_rotate]{data_rotate()}}
\item Functions to recode data: \code{\link[=rescale]{rescale()}}, \code{\link[=reverse]{reverse()}}, \code{\link[=categorize]{categorize()}},
\code{\link[=recode_values]{recode_values()}}, \code{\link[=slide]{slide()}}
\item Functions to standardize, normalize, rank-transform: \code{\link[=center]{center()}}, \code{\link[=standardize]{standardize()}}, \code{\link[=normalize]{normalize()}}, \code{\link[=ranktransform]{ranktransform()}}, \code{\link[=winsorize]{winsorize()}}
\item Split and merge data frames: \code{\link[=data_partition]{data_partition()}}, \code{\link[=data_merge]{data_merge()}}
\item Functions to find or select columns: \code{\link[=data_select]{data_select()}}, \code{\link[=extract_column_names]{extract_column_names()}}
\item Functions to filter rows: \code{\link[=data_match]{data_match()}}, \code{\link[=data_filter]{data_filter()}}
}
}
