% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.R
\name{fit_three_layer}
\alias{fit_three_layer}
\title{MCMC sampling for three layer deep GP}
\usage{
fit_three_layer(
  x,
  y,
  nmcmc = 10000,
  D = ifelse(is.matrix(x), ncol(x), 1),
  verb = TRUE,
  w_0 = NULL,
  z_0 = NULL,
  theta_y_0 = 0.01,
  theta_w_0 = 0.1,
  theta_z_0 = 0.1,
  g_0 = 0.001,
  true_g = NULL,
  v = 2.5,
  settings = NULL,
  cov = c("matern", "exp2"),
  vecchia = FALSE,
  m = NULL,
  ord = NULL,
  cores = NULL
)
}
\arguments{
\item{x}{vector or matrix of input locations}

\item{y}{vector of response values}

\item{nmcmc}{number of MCMC iterations}

\item{D}{integer designating dimension of hidden layers, defaults to 
dimension of \code{x}}

\item{verb}{logical indicating whether to print iteration progress}

\item{w_0}{initial value for hidden layer \code{w} (rows must correspond to
rows of \code{x}, requires \code{ncol(w_0) = D}.  Defaults to the 
identity mapping.  If \code{nrow(w_0) < nrow(x)}, missing initial values
are filled-in with the GP posterior mean.}

\item{z_0}{initial value for hidden layer \code{z} (rows must correspond to
rows of \code{x}, requires \code{ncol(z_0) = D}.  Defaults to the 
identity mapping.  If \code{nrow(z_0) < nrow(x)}, missing initial values
are filled-in with the GP posterior mean.}

\item{theta_y_0}{initial value for \code{theta_y} (length scale of outer 
layer)}

\item{theta_w_0}{initial value for \code{theta_w} (length scale of middle 
layer), may be single value or vector of length \code{D}}

\item{theta_z_0}{initial value for \code{theta_z} (length scale of inner 
layer), may be single value or vector of length \code{D}}

\item{g_0}{initial value for \code{g}}

\item{true_g}{if true nugget is known it may be specified here (set to a 
small value to make fit deterministic).  Note - values that are too 
small may cause numerical issues in matrix inversions.}

\item{v}{Matern smoothness parameter (only used if \code{cov = "matern"})}

\item{settings}{hyperparameters for proposals and priors (see details)}

\item{cov}{covariance kernel, either Matern (\code{"matern"}) or squared exponential 
(\code{"exp2"})}

\item{vecchia}{logical indicating whether to use Vecchia approximation}

\item{m}{size of Vecchia conditioning sets, defaults to the lower of 25 or
the maximum available (only used if \code{vecchia = TRUE})}

\item{ord}{optional ordering for Vecchia approximation, must correspond
to rows of \code{x}, defaults to random, is applied to \code{x},
\code{w}, and \code{z}}

\item{cores}{number of cores to use for OpenMP parallelization 
(\code{vecchia = TRUE} only).  Defaults to \code{min(4, maxcores - 1)} 
where \code{maxcores} is the number of detectable available cores.}
}
\value{
a list of the S3 class \code{dgp3} or \code{dgp3vec} with elements:
\itemize{
  \item \code{x}: copy of input matrix
  \item \code{y}: copy of response vector
  \item \code{nmcmc}: number of MCMC iterations
  \item \code{settings}: copy of proposal/prior settings
  \item \code{v}: copy of Matern smoothness parameter (\code{v = 999} 
        indicates \code{cov = "exp2"}) 
  \item \code{g}: vector of MCMC samples for \code{g}
  \item \code{tau2_y}: vector of MLE estimates for \code{tau2} on outer layer
  \item \code{theta_y}: vector of MCMC samples for \code{theta_y} (length 
        scale of outer layer)
  \item \code{theta_w}: matrix of MCMC samples for \code{theta_w} (length 
        scale of middle layer)
  \item \code{theta_z}: matrix of MCMC samples for \code{theta_z} (length 
        scale of inner layer)
  \item \code{w}: list of MCMC samples for middle hidden layer \code{w}
  \item \code{z}: list of MCMC samples for inner hidden layer \code{z}
  \item \code{w_approx}: Vecchia approximation object for outer layer (\code{vecchia = TRUE} only)
  \item \code{z_approx}: Vecchia approximation object for middle layer (\code{vecchia = TRUE} only)
  \item \code{x_approx}: Vecchia approximation object for inner layer (\code{vecchia = TRUE} only)
  \item \code{ll}: vector of MVN log likelihood of the outer layer 
        for reach Gibbs iteration
  \item \code{time}: computation time in seconds
}
}
\description{
Conducts MCMC sampling of hyperparameters, hidden layer 
    \code{z}, and hidden layer \code{w} for a three layer deep GP.  
    Separate length scale parameters \code{theta_z}, 
    \code{theta_w}, and \code{theta_y} govern the correlation 
    strength of the inner layer, middle layer, and outer layer respectively.  
    Nugget parameter \code{g} governs noise on the outer layer.  In Matern 
    covariance, \code{v} governs smoothness.

    Currently, there are no \code{pmx}, \code{monowarp}, or \code{dydx} 
    options.
}
\details{
Maps inputs \code{x} through hidden layer \code{z} then hidden
    layer \code{w} to outputs \code{y}.  Conducts sampling of the hidden 
    layers using elliptical slice sampling.  Utilizes Metropolis Hastings 
    sampling of the length scale and nugget parameters with proposals and 
    priors controlled by \code{settings}.  When \code{true_g} is set to a 
    specific value, the nugget is not estimated.  When 
    \code{vecchia = TRUE}, all calculations leverage the Vecchia 
    approximation with specified conditioning set size \code{m}.
    
    NOTE on OpenMP: The Vecchia implementation relies on OpenMP parallelization
    for efficient computation.  This function will produce a warning message 
    if the package was installed without OpenMP (this is the default for 
    CRAN packages installed on Apple machines).  To set up OpenMP 
    parallelization, download the package source code and install 
    using the gcc/g++ compiler.  
    
    Proposals for \code{g}, 
    \code{theta_y}, \code{theta_w}, and \code{theta_z} follow a uniform 
    sliding window scheme, e.g.,
    
    \code{g_star <- runif(1, l * g_t / u, u * g_t / l)},
    
    with defaults \code{l = 1} and \code{u = 2} provided in \code{settings}.
    To adjust these, set \code{settings = list(l = new_l, u = new_u)}.  
    Priors on \code{g}, \code{theta_y}, \code{theta_w}, and \code{theta_z} 
    follow Gamma distributions with shape parameters (\code{alpha}) and rate 
    parameters (\code{beta}) controlled within the \code{settings} list 
    object.  Default priors differ for noisy/deterministic settings.  
    All default values are 
    visible in the internal \code{deepgp:::check_settings} function.
    These priors are designed for \code{x} scaled to [0, 1] and \code{y} 
    scaled to have mean 0 and variance 1.  These may be adjusted using the 
    \code{settings} input.

    The scale on the latent layers (\code{tau2_z} and \code{tau2_w}) may also 
    be specified in \code{settings}.  Defaults to 1. 
    
    When \code{w_0 = NULL} and/or \code{z_0 = NULL}, the hidden layers are 
    initialized at \code{x} (i.e., the identity mapping).
    If \code{w_0} and/or \code{z_0} is of dimension \code{nrow(x) - 1} by 
    \code{D}, the final row is filled-in using the GP posterior mean. This is helpful in 
    sequential design when adding a new input location and starting the MCMC 
    at the place where the previous MCMC left off.
    
    The output object of class \code{dgp3} or \code{dgp3vec} is designed for 
    use with \code{continue}, \code{trim}, and \code{predict}.
}
\examples{
# Additional examples including real-world computer experiments are available at: 
# https://bitbucket.org/gramacylab/deepgp-ex/
\donttest{
# G function in 2 dimensions (https://www.sfu.ca/~ssurjano/gfunc.html)
f <- function(xx, a = (c(1:length(xx)) - 1) / 2) { 
  new1 <- abs(4 * xx - 2) + a
  new2 <- 1 + a
  prod <- prod(new1 / new2)
  return((prod - 1) / 0.86)
}

# Training data
d <- 2
n <- 30
x <- matrix(runif(n * d), ncol = d)
y <- apply(x, 1, f)

# Testing data
n_test <- 500
xx <- matrix(runif(n_test * d), ncol = d)
yy <- apply(xx, 1, f)

i <- interp::interp(xx[, 1], xx[, 2], yy)
image(i, col = heat.colors(128))
contour(i, add = TRUE)
contour(i, level = -0.5, col = 4, add = TRUE) # potential failure limit
points(x)

# Example 1: nugget fixed, calculating entropy
fit <- fit_three_layer(x, y, nmcmc = 2000, true_g = 1e-6)
plot(fit)
fit <- trim(fit, 1000, 2)
fit <- predict(fit, xx, entropy_limit = -0.5, cores = 1)
plot(fit)
i <- interp::interp(xx[, 1], xx[, 2], fit$entropy)
image(i, col = heat.colors(128), main = "Entropy")

# Example 2: using Vecchia
fit <- fit_three_layer(x, y, nmcmc = 2000, true_g = 1e-6, vecchia = TRUE, m = 10)
plot(fit)
fit <- trim(fit, 1000, 2)
fit <- predict(fit, xx, cores = 1)
plot(fit)
}

}
\references{
Sauer, A. (2023). Deep Gaussian process surrogates for computer experiments. 
     *Ph.D. Dissertation, Department of Statistics, Virginia Polytechnic Institute and State University.*
     \url{http://hdl.handle.net/10919/114845}
     \cr\cr
Sauer, A., Gramacy, R.B., & Higdon, D. (2023). Active learning for deep 
     Gaussian process surrogates. *Technometrics, 65,* 4-18.  arXiv:2012.08015
     \cr\cr
Sauer, A., Cooper, A., & Gramacy, R. B. (2023). Vecchia-approximated deep Gaussian 
     processes for computer experiments. 
     *Journal of Computational and Graphical Statistics, 32*(3), 824-837.  arXiv:2204.02904
     \cr\cr
}
