% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_moments.R
\name{plot_moments}
\alias{plot_moments}
\title{Plot function: Display the influence of a covariate}
\usage{
plot_moments(model, int_var, pred_data = NULL, rug = FALSE,
  samples = FALSE, uncertainty = FALSE, ex_fun = NULL,
  palette = "viridis", vary_by = NULL)
}
\arguments{
\item{model}{A fitted model on which the plots are based.}

\item{int_var}{The variable for which influences of the moments shall be
graphically displayed. Has to be in character form.}

\item{pred_data}{Combinations of covariate data, sometimes also known as
"newdata", including the variable of interest, which will be ignored in
later processing.}

\item{rug}{Should the resulting plot be a rug plot?}

\item{samples}{If the provided model is a bamlss model, should the moment
values be "correctly" calculated, using the transformed samples? See
details for details.}

\item{uncertainty}{If \code{TRUE}, displays uncertainty measures about the
covariate influences. Can only be \code{TRUE} if samples is also
\code{TRUE}.}

\item{ex_fun}{An external function \code{function(par) {...}} which
calculates a measure, whose dependency from a certain variable is of
interest. Has to be specified in character form. See examples for an
example.}

\item{palette}{See \code{\link{plot_dist}}.}

\item{vary_by}{Variable name in character form over which to vary the
mean/reference values of explanatory variables. It is passed to
\link{set_mean}. See that documentation for further details.}
}
\description{
This function takes a dataframe of predictions with one row per prediction
and one column for every explanatory variable. Then, those predictions are
held constant while one specific variable is varied over it's whole range
(min-max). Then, the constant variables with the varied interest variables
are predicted and plotted against the expected value and the variance of the
underlying distribution.
}
\details{
The target of this function is to display the influence of a selected effect
on the predicted moments of the modeled distribution. The motivation for
computing influences on the moments of a distribution is its
interpretability: In most cases, the parameters of a distribution do not
equate the moments and as such are only indirectly location, scale or shape
properties, making the computed effects hard to understand.

Navigating through the disarray of link functions, non-parametric effects and
transformations to moments, \code{plot_moments()} supports a wide range of
target distributions. See \link{dists} for details.

Whether a distribution is supported or not depends on whether the underlying
\code{R} object possesses functions to calculate the moments of the
distribution from the predicted parameters. To achieve this for as many
distributional families as possible, we worked together with both the authors
of \link{gamlss} (Rigby and Stasinopoulos 2005) and \link{bamlss} (Umlauf et
al. 2018) and implemented the moment functions for almost all available
distributions in the respective packages. The \link{betareg} family was
implemented in \link{distreg.vis} as well.
}
\examples{

# Generating some data
dat <- model_fam_data(fam_name = "LOGNO")

# Estimating the model
library("gamlss")
model <- gamlss(LOGNO ~ ps(norm2) + binomial1,
                ~ ps(norm2) + binomial1,
                data = dat, family = "LOGNO")

# Get newdata by either specifying an own data.frame, or using set_mean()
# for obtaining mean vals of explanatory variables
ndata_user <- dat[1:5, c("norm2", "binomial1")]
ndata_auto <- set_mean(model_data(model))

# Influence graphs
plot_moments(model, int_var = "norm2", pred_data = ndata_user) # cont. var
plot_moments(model, int_var = "binomial1", pred_data = ndata_user) # discrete var
plot_moments(model, int_var = "norm2", pred_data = ndata_auto) # with new ndata

# If pred_data argument is omitted plot_moments uses mean explanatory
# variables for prediction (using set_mean)
plot_moments(model, int_var = "norm2")

# Rug Plot
plot_moments(model, int_var = "norm2", rug = TRUE)

# Different colour palette
plot_moments(model, int_var = "binomial1", palette = "Dark2")

# Using an external function
ineq <- function(par) {
  2 * pnorm((par[["sigma"]] / 2) * sqrt(2)) - 1
}
plot_moments(model, int_var = "norm2", pred_data = ndata_user, ex_fun = "ineq")

}
\references{
Rigby RA, Stasinopoulos DM (2005). "Generalized Additive Models
  for Location, Scale and Shape." Journal of the Royal Statistical Society C,
  54(3), 507-554.

  Umlauf, N, Klein N, Zeileis A (2018). "BAMLSS: Bayesian
  Additive Models for Location, Scale and Shape (and Beyond)." Journal of
  Computational and Graphical Statistics, 27(3), 612-627.
}
