drake_context("dependencies")

test_with_dir("unparsable commands are handled correctly", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  x <- "bluh$"
  expect_error(deps_code(x))
})

test_with_dir("dot symbol is illegal", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  expect_equal(
    sort(deps_code("sqrt(x + y + .)")$name),
    sort(c("sqrt", "x", "y"))
  )
  expect_equal(
    sort(deps_code("subset(complete.cases(.))")$name),
    sort(c("complete.cases", "subset"))
  )
  plan <- drake_plan(
    x = 1,
    y = 2,
    a = sqrt(x + y + .),
    b = subset(complete.cases(.))
  )
  e <- environment()
  expect_false(exists(".", envir = e))
  config <- drake_config(plan)
  plan <- drake_plan(
    . = 1,
    y = 2,
    a = sqrt(x + y),
    b = subset(complete.cases(.))
  )
  expect_error(drake_config(plan), "cannot be target names")
})

test_with_dir("file_out() and knitr_in(): commands vs imports", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  skip_if_not_installed("knitr")
  cmd <- quote({
    file_in("x"); file_out("y"); knitr_in("report.Rmd")
  })
  f <- function() {
    file_in("x")
    file_out("y")
    knitr_in("report.Rmd")
  }
  file.create("x")
  file.create("y")
  path <- system.file(
    file.path("examples", "mtcars", "report.Rmd"),
    package = "drake",
    mustWork = TRUE
  )
  file.copy(
    from = path, to = file.path(getwd(), "report.Rmd"), overwrite = TRUE)
  x <- command_dependencies(cmd)
  x <- decode_deps_list(x)
  x0 <- list(
    file_in = "x", file_out = "y", loadd = "large",
    readd = c("small", "coef_regression2_small"),
    knitr_in = "report.Rmd")
  expect_equal(length(x), length(x0))
  for (i in names(x)) {
    expect_equal(sort(x[[i]]), sort(x0[[i]]))
  }
  y <- import_dependencies(f)
  y <- decode_deps_list(y)
  y0 <- list(
    file_in = "x",
    knitr_in = "report.Rmd",
    loadd = "large",
    readd = c("small", "coef_regression2_small")
  )
  expect_equal(length(y), length(y0))
  for (i in names(y)) {
    expect_equal(sort(y[[i]]), sort(y0[[i]]))
  }
  expect_equal(
    sort(deps_code(f)$name), sort(unname(unlist(y))))
  expect_equal(
    sort(deps_code(cmd)$name),
    sort(
      c("coef_regression2_small", "large",
        "report.Rmd", "small", "x", "y"
      )
    )
  )
})

test_with_dir("deps_code() and deps_target()", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  expect_equal(nrow(deps_code("")), 0)
  expect_equal(length(command_dependencies(NA)), 0)
  expect_equal(length(command_dependencies(NULL)), 0)
  expect_equal(length(command_dependencies(character(0))), 0)
  expect_equal(deps_code(base::c)$name, character(0))
  expect_equal(deps_code(base::list)$name, character(0))
  f <- function(x, y) {
    out <- x + y + g(x)
    saveRDS(out, "out.rds")
  }
  expect_false(is_vectorized(f))
  expect_false(is_vectorized("char"))
  expect_equal(
    sort(deps_code(f)$name),
    sort(c("g", "saveRDS"))
  )
  my_plan <- drake_plan(
    x = 1 + some_object,
    my_target = x + readRDS(file_in("tracked_input_file.rds")),
    return_value = f(x, y, g(z + w)),
    botched = read.csv(file_in(nothing)),
    meta = read.table(file_in("file_in"))
  )
  config <- drake_config(
    my_plan,
    session_info = FALSE,
    cache = storr::storr_environment()
  )
  expect_equal(deps_code(my_plan$command[[1]])$name, "some_object")
  expect_equal(
    sort(deps_code(my_plan$command[[2]])$name),
    sort(c("tracked_input_file.rds", "x", "readRDS")))
  expect_equal(
    sort(deps_code(my_plan$command[[3]])$name),
    sort(c("f", "g", "w", "x", "y", "z"))
  )
  expect_equal(
    sort(deps_code(my_plan$command[[4]])$name),
    sort(c("read.csv"))
  )
  expect_equal(
    sort(deps_code(my_plan$command[[5]])$name),
    sort(c("read.table", "file_in"))
  )
  expect_true(!nrow(deps_target(x, config)))
  expect_equal(
    sort(deps_target(my_target, config)$name),
    sort(c("tracked_input_file.rds", "x")))
  expect_equal(
    sort(deps_target(return_value, config)$name),
    sort(c("f", "x"))
  )
  expect_equal(
    sort(deps_target(botched, config)$name),
    character(0)
  )
  expect_equal(
    sort(deps_target(meta, config)$name),
    sort("file_in"))
})

test_with_dir("tracked() works", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  config <- dbug()
  x <- sort(tracked(config))
  y <- sort(c(
    display_keys(encode_path("intermediatefile.rds")),
    "drake_target_1",
    "yourinput", "nextone",
    "combined", "myinput", "final", "j", "i", "h", "g", "f",
    "c", "b", "a",  display_keys(encode_path("input.rds"))
  ))
  expect_equal(x, y)
})

test_with_dir("missing input files", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  config <- dbug()
  expect_silent(tmp <- missing_input_files(config))
  unlink("input.rds", force = TRUE)
  expect_warning(tmp <- missing_input_files(config))
  expect_silent(tmp <- config_checks(config))
  expect_warning(runtime_checks(config), regexp = "missing")
  config$skip_safety_checks <- TRUE
  expect_silent(tmp <- runtime_checks(config))
})

test_with_dir("Vectorized nested functions work", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  e <- new.env(parent = globalenv())
  eval(parse(text = "f <- Vectorize(function(x) g(x), \"x\")"), envir = e)
  eval(parse(text = "g <- function(x) x + y"), envir = e)
  e$y <- 7
  config <- dbug()
  config$envir <- e
  config$plan <- drake_plan(a = f(1:10))
  config$targets <- "a"
  expect_equal(deps_code(e$f)$name, "g")
  expect_equal(deps_code(e$g)$name, "y")

  testrun(config)
  config <- testconfig(config)
  if ("a" %in% ls(config$envir)) {
    rm(a, envir = config$envir)
  }
  expect_equal(readd(a), 8:17)
  k <- readd(f)
  expect_true(is.character(k))
  expect_equal(character(0), outdated(config))
  config$envir$y <- 8
  expect_equal("a", outdated(config))

  # Target "a" should react.
  testrun(config)
  config <- testconfig(config)
  expect_equal(character(0), outdated(config))
  expect_equal(readd(a), 9:18)

  # Change a vectorized function and see target "a" react.
  eval(
    parse(text = "f <- Vectorize(function(x) {g(x) + 3}, \"x\")"),
    envir = e
  )
  testrun(config)
  config <- testconfig(config)
  expect_equal(justbuilt(config), "a")
  expect_equal(readd(a), 12:21)
})

test_with_dir("deps_target()", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  skip_if_not_installed("knitr")
  load_mtcars_example()
  config <- drake_config(my_plan, cache = storr::storr_environment())
  d1 <- deps_target(report, config = config)
  d1 <- as.data.frame(d1[order(d1$name), ])
  d2 <- data.frame(
    name = c(
      "coef_regression2_small", "large", "report.md", "report.Rmd", "small"
    ),
    type = c("readd", "loadd", "file_out", "knitr_in", "readd"),
    stringsAsFactors = FALSE
  )
  d2 <- d2[order(d2$name), ]
  expect_equivalent(d1, d2)
  d <- deps_target(regression1_small, config = config)
  expect_equal(sort(d$name), sort(c("reg1", "small")))
  expect_equal(d$type, rep("globals", 2))
})

test_with_dir("self-referential commands and imports", {
  f <- function(x, ...) {
    x <- f
  }
  x <- data.frame(f = 123)
  plan <- drake_plan(y = f(x, y))
  cache <- storr::storr_environment()
  make(plan, cache = cache, session_info = FALSE)
  o <- drake_config(plan, cache = cache, session_info = FALSE)
  expect_equal(justbuilt(o), "y")
  log1 <- drake_cache_log(cache = cache)
  make(plan, cache = cache, session_info = FALSE)
  o <- drake_config(plan, cache = cache, session_info = FALSE)
  expect_true(nobuild(o))
  log2 <- drake_cache_log(cache = cache)
  expect_equal(log1, log2)
})

test_with_dir("ignore() suppresses updates", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  cache <- storr::storr_environment()
  envir <- new.env(parent = globalenv())
  envir$arg <- 4

  # Without ignore()
  make(
    plan = drake_plan(x = sqrt(arg)),
    envir = envir,
    cache = cache
  )
  con <- drake_config(
    plan = drake_plan(x = sqrt(arg)),
    envir = envir,
    cache = cache
  )
  expect_equal(justbuilt(con), "x")
  con$envir$arg <- con$envir$arg + 1
  make(config = con)
  expect_equal(justbuilt(con), "x")

  # With ignore()
  make(
    plan = drake_plan(x = sqrt( ignore(arg) + 123)), # nolint
    envir = envir,
    cache = cache
  )
  con <- drake_config(
    plan = drake_plan(x = sqrt( ignore(arg) + 123)), # nolint
    envir = envir,
    cache = cache
  )
  expect_equal(justbuilt(con), "x")
  con$envir$arg <- con$envir$arg + 1
  con$cache$clear(namespace = "progress")
  make(config = con)
  expect_equal(justbuilt(con), character(0))

  con$envir$arg2 <- con$envir$arg + 1234
  con$plan <- drake_plan(x = sqrt( ignore  (arg2 ) + 123)) # nolint
  con$cache$clear(namespace = "progress")
  make(config = con)
  expect_equal(justbuilt(con), character(0))
})

test_with_dir("ignore() works on its own", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  expect_equal(ignore(), NULL)
  expect_equal(ignore(1234), 1234)
  expect_identical(ignore_ignore(digest::digest), digest::digest)
})

test_with_dir("Standardized commands have no attributes", {
  expect_null(attributes(standardize_command("")))
  expect_null(attributes(standardize_command("f(x) + y + function(abc){}")))
  expect_null(attributes(standardize_command(quote(NULL))))
  expect_null(attributes(standardize_command(digest::digest)))
  expect_null(attributes(standardize_command(body(digest::digest))))
})

test_with_dir("Can standardize commands", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  expect_true(is.character(standardize_command(parse(text = ""))))
  expect_identical(
    standardize_command(parse(text = "f(x +2) + 2")),
    standardize_command(parse(text = "f(x + 2) + 2"))
  )
  expect_identical(
    standardize_command(quote(f(x + 2) + 2)),
    standardize_command(parse(text = "f(x + 2) + 2"))
  )
  expect_false(
    identical(
      standardize_command(parse(text = "f(x + 2) + 2")),
      standardize_command(parse(text = "f(x + 1 - 1) + 2"))
    )
  )
  expect_identical(
    standardize_command(parse(text = "b->a")),
    standardize_command(parse(text = "a <- b"))
  )
  expect_identical(
    standardize_command(parse(text = "y=sqrt(x=1)")),
    standardize_command(parse(text = "y = sqrt(x = 1)"))
  )
  expect_identical(
    standardize_command(
      parse(text = "abcdefg = hijklmnop <- qrstuvwxyz\n\n")
    ),
    standardize_command(parse(text = "abcdefg = hijklmnop <- qrstuvwxyz"))
  )
  a <- standardize_command(parse(text = "z = {f('#') # comment
    x = 5

    y <-
      'test'
    z <- 4

    x2 <- 'test2'
  }"))
  b <- standardize_command(parse(text = "z = {f('#') # comment X
  x = 5

  y <- 'test'
  z <- 4
  'test2' -> x2
  }"))
  c <- standardize_command(parse(text = "z = {f('#') # comment X
  x = 5

  y <- 'test3'
  z <- 4
  'test2' -> x2
  }"))
  expect_identical(a, b)
  expect_false(identical(b, c))
})

test_with_dir("standardized commands with ignore()", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  expect_equal(
    standardize_command(
      parse(text = "f(     sqrt( ignore(fun(arg) + 7) + 123))")
    ),
    standardize_command(parse(text = "f(sqrt(ignore() + 123))"))
  )
  expect_equal(
    standardize_command(
      parse(text = "f(sqrt( ignore  (fun(arg) + 7)+123) ) # noooop")
    ),
    standardize_command(parse(text = "f(sqrt(ignore() + 123))"))
  )
  expect_equal(
    standardize_command(
      parse(text = " f (sqrt( drake::ignore(fun(arg) + 7) + 123 ))")
    ),
    standardize_command(parse(text = "f(sqrt(ignore() + 123))"))
  )
  expect_equal(
    standardize_command(
      parse(text = "\tf(sqrt( drake ::: ignore  (fun(arg) + 7) + 123))")
    ),
    standardize_command(parse(text = "f(sqrt(ignore() + 123))"))
  )
  expect_equal(
    standardize_command(
      parse(text = "function(x){(sqrt( ignore(fun(arg) + 7) + 123))}")
    ),
    standardize_command(
      parse(text = "function(x) {\n    (sqrt(ignore() + 123))\n}")
    )
  )
  expect_equal(
    standardize_command(
      parse(text = "f(sqrt( ignore(fun(arg) + 7) + 123)); g(ignore(i))")
    ),
    standardize_command(
      parse(text = "f(sqrt( ignore() + 123)); g(ignore())")
    )
  )
  f <- function(x) {
    (sqrt( ignore(fun(arg) + 7) + 123)) # nolint
  }
  b <- body(ignore_ignore(f))
  for (a in names(attributes(b))) {
    attr(b, a) <- NULL
  }
  expect_equal(b, quote({  (sqrt(ignore() + 123)) })) # nolint
})

test_with_dir("Can standardize commands from expr or lang", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  x <- parse(text = "f(x +2) + 2")
  y <- standardize_command(x)
  z <- standardize_command(x)
  w <- standardize_command(x[[1]])
  s <- safe_deparse(quote(f(x + 2) + 2))
  expect_equal(y, s)
  expect_equal(z, s)
  expect_equal(w, s)
})

test_with_dir("ignore() in imported functions", {
  skip_on_cran() # CRAN gets whitelist tests only (check time limits).
  f <- function(x) {
    (sqrt( ignore(sqrt(x) + 7) + 123)) # nolint
  }
  plan <- drake_plan(x = f(1))
  cache <- storr::storr_environment()
  make(plan, cache = cache)
  # Because ignore() affects standardization:
  expect_true(is.character(readd(f, cache = cache)))
  config <- drake_config(plan, cache = cache)
  expect_equal(justbuilt(config), "x")

  str <- readd(f, cache = cache)
  expect_false(any(grepl("sqrt(x)", str, fixed = TRUE)))
  expect_true(any(grepl("(sqrt(ignore() + 123))", str, fixed = TRUE)))
  f <- function(x) {
    (sqrt( ignore(sqrt(x) + 8) + 123)) # nolint
  }
  make(plan, cache = cache)
  config <- drake_config(plan, cache = cache)
  expect_equal(justbuilt(config), character(0))
  f <- function(x) {
    (sqrt( ignore(sqrt(x) + 8) + 124)) # nolint
  }
  make(plan, cache = cache)
  config <- drake_config(plan, cache = cache)
  expect_equal(justbuilt(config), "x")
})
