% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/twdtwApply.R
\docType{methods}
\name{twdtwApply}
\alias{twdtwApply}
\alias{twdtwApply,twdtwTimeSeries-method}
\alias{twdtwApply-twdtwTimeSeries}
\alias{twdtwApply,twdtwRaster-method}
\alias{twdtwApply-twdtwRaster}
\title{Apply TWDTW analysis}
\usage{
twdtwApply(x, y, resample = TRUE, length = NULL, weight.fun = NULL,
  dist.method = "Euclidean", step.matrix = symmetric1, n = NULL,
  span = NULL, min.length = 0, theta = 0.5, ...)

\S4method{twdtwApply}{twdtwTimeSeries}(x, y, resample, length, weight.fun,
  dist.method, step.matrix, n, span, min.length, theta, keep = FALSE, ...)

\S4method{twdtwApply}{twdtwRaster}(x, y, resample, length, weight.fun,
  dist.method, step.matrix, n, span, min.length, theta, breaks = NULL,
  from = NULL, to = NULL, by = NULL, overlap = 0.5, chunk.size = 1000,
  filepath = NULL, silent = FALSE, ...)
}
\arguments{
\item{x}{an object of class twdtw*. This is the target time series. 
Usually, it is a set of unclassified time series.}

\item{y}{an object of class \link[dtwSat]{twdtwTimeSeries}. 
The temporal patterns.}

\item{resample}{resample the patterns to have the same length. Default is TRUE.
See \link[dtwSat]{resampleTimeSeries} for details.}

\item{length}{An integer. Patterns length used with \code{patterns.length}. 
If not declared the length of the output patterns will be the length of 
the longest pattern.}

\item{weight.fun}{A function. Any function that receive and performs a 
computation on a matrix. The function receives a matrix of time differences 
in days and returns a matrix of time-weights. If not declared the time-weight 
is zero. In this case the function runs the standard version of the dynamic 
time warping. See details.}

\item{dist.method}{A character. Method to derive the local cost matrix.
Default is ''Euclidean'' see \code{\link[proxy]{dist}} in package 
\pkg{proxy}.}

\item{step.matrix}{see \code{\link[dtw]{stepPattern}} in package \pkg{dtw} [2].}

\item{n}{An integer. The maximun number of matches to perform. 
NULL will return all matches.}

\item{span}{A number. Span between two matches, \emph{i.e.} the minimum
interval between two matches, for details see [3]. If not declared it removes
all overlapping matches of the same pattern. To include overlapping matches 
of the same pattern use \code{span=0}.}

\item{min.length}{A number between 0 an 1. This argument removes the over fittings.
Minimum length after warping. Percentage of the original pattern length. Default is 0.5, 
meaning that the matching cannot be shorter than half of the pattern length.}

\item{theta}{numeric between 0 and 1. The weight of the time 
for the TWDTW computation. Use \code{theta=0} to cancel the time-weight, 
\emph{i.e.} to run the original DTW algorithm. Default is 0.5, meaning that 
the time has the same weight as the curve shape in the TWDTW analysis.}

\item{...}{arguments to pass to \code{\link[raster]{writeRaster}}}

\item{keep}{preserves the cost matrix, inputs, and other internal structures. 
Default is FALSE. For plot methods use \code{keep=TRUE}.}

\item{breaks}{A vector of class \code{\link[base]{Dates}}. This replaces the arguments \code{from},
\code{to}, and \code{by}.}

\item{from}{A character or \code{\link[base]{Dates}} object in the format "yyyy-mm-dd".}

\item{to}{A \code{\link[base]{character}} or \code{\link[base]{Dates}} object in the format "yyyy-mm-dd".}

\item{by}{A \code{\link[base]{character}} with the intevals size, \emph{e.g.} "6 month".}

\item{overlap}{A number between 0 and 1. The minimum overlapping 
between one match and the interval of classification. Default is 0.5, 
\emph{i.e.} an overlap minimum of 50\%.}

\item{chunk.size}{An integer. Set the number of cells for each block, 
see \code{\link[raster]{blockSize}} for details.}

\item{filepath}{A character. The path to save the raster with results. If not informed the 
function saves in the current work directory.}

\item{silent}{if set to TRUE then hide chunk processing message. Default is FALSE.}
}
\value{
An object of class twdtw*.
}
\description{
This function performs a multidimensional Time-Weighted DTW 
analysis and retrieves the matches between the temporal patterns and 
a set of time series [1].
}
\details{
The linear \code{linearWeight} and \code{logisticWeight} weight functions 
can be passed to \code{twdtwApply} through the argument \code{weight.fun}. This will 
add a time-weight to the dynamic time warping analysis. The time weight 
creates a global constraint useful to analyse time series with phenological cycles
of vegetation that are usually bound to seasons. In previous studies by [1] the 
logistic weight had better results than the linear for land cover classification. 
See [1] for details about the method.
}
\examples{
# Applying TWDTW analysis to objects of class twdtwTimeSeries
log_fun = logisticWeight(-0.1, 100)
ts = twdtwTimeSeries(MOD13Q1.ts.list)
patt = twdtwTimeSeries(MOD13Q1.patterns.list)
mat1 = twdtwApply(x=ts, y=patt, weight.fun=log_fun)
mat1

\dontrun{
# Parallel processin
require(parallel)
mat_list = mclapply(as.list(ts), mc.cores=2, FUN=twdtwApply, y=patt, weight.fun=log_fun)
mat2 = twdtwMatches(alignments=mat_list)
}
\dontrun{
# Run TWDTW analysis for raster time series 
patt = MOD13Q1.MT.yearly.patterns
evi = brick(system.file("lucc_MT/data/evi.tif", package="dtwSat"))
ndvi = brick(system.file("lucc_MT/data/ndvi.tif", package="dtwSat"))
red = brick(system.file("lucc_MT/data/red.tif", package="dtwSat"))
blue = brick(system.file("lucc_MT/data/blue.tif", package="dtwSat"))
nir = brick(system.file("lucc_MT/data/nir.tif", package="dtwSat"))
mir = brick(system.file("lucc_MT/data/mir.tif", package="dtwSat"))
doy = brick(system.file("lucc_MT/data/doy.tif", package="dtwSat"))
timeline = scan(system.file("lucc_MT/data/timeline", package="dtwSat"), what="date")
rts = twdtwRaster(evi, ndvi, red, blue, nir, mir, timeline = timeline, doy = doy)

time_interval = seq(from=as.Date("2007-09-01"), to=as.Date("2013-09-01"), 
                    by="12 month")
log_fun = weight.fun=logisticWeight(-0.1,50)

r_twdtw = twdtwApply(x=rts, y=patt, weight.fun=log_fun, breaks=time_interval, 
          filepath="~/test_twdtw", overwrite=TRUE, format="GTiff")

plot(r_twdtw, type="distance")

r_lucc = twdtwClassify(r_twdtw, format="GTiff", overwrite=TRUE)

plot(r_lucc)

plot(r_lucc, type="distance")

}
}
\references{
[1] Maus  V,  Camara  G,  Cartaxo  R,  Sanchez  A,  Ramos  FM,  de Queiroz, GR.
(2016). A Time-Weighted Dynamic Time Warping method for land use and land cover 
mapping. IEEE Journal of Selected Topics in Applied Earth Observations and Remote 
Sensing, vol.9, no.8, pp.3729-3739.

[2] Giorgino, T. (2009). Computing and Visualizing Dynamic Time Warping Alignments in R: 
The dtw Package. Journal of Statistical Software, 31, 1-24.

[3] Muller, M. (2007). Dynamic Time Warping. In Information Retrieval for Music 
and Motion (pp. 79-84). London: Springer London, Limited.
}
\seealso{
\code{\link[dtwSat]{twdtwMatches-class}}, 
\code{\link[dtwSat]{twdtwTimeSeries-class}}, 
\code{\link[dtwSat]{twdtwRaster-class}}, 
\code{\link[dtwSat]{getTimeSeries}}, and 
\code{\link[dtwSat]{createPatterns}}
}
\author{
Victor Maus, \email{vwmaus1@gmail.com}
}
