% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/twdtwClassify.R
\name{twdtwClassify}
\alias{twdtwClassify}
\alias{twdtwClassify,twdtwMatches-method}
\alias{twdtwClassify-twdtwTimeSeries}
\alias{twdtwClassify,twdtwRaster-method}
\alias{twdtwClassify-twdtwRaster}
\title{Classify time series}
\usage{
twdtwClassify(x, ...)

\S4method{twdtwClassify}{twdtwMatches}(
  x,
  patterns.labels = NULL,
  from = NULL,
  to = NULL,
  by = NULL,
  breaks = NULL,
  overlap = 0.5,
  thresholds = Inf,
  fill = "unclassified"
)

\S4method{twdtwClassify}{twdtwRaster}(
  x,
  patterns.labels = NULL,
  thresholds = Inf,
  fill = 255,
  filepath = "",
  ...
)
}
\arguments{
\item{x}{An object of class twdtw*. This is the target time series. 
Usually, it is a set of unclassified time series.}

\item{...}{Arguments to pass to specific methods for each twdtw* class 
and other arguments to pass to \code{\link[raster]{writeRaster}} and 
\code{\link[raster]{pbCreate}}.}

\item{patterns.labels}{a vector with labels of the patterns.}

\item{from}{A character or \code{\link[base]{Dates}} object in the format "yyyy-mm-dd".}

\item{to}{A \code{\link[base]{character}} or \code{\link[base]{Dates}} object in the format "yyyy-mm-dd".}

\item{by}{A \code{\link[base]{character}} with the interval size, \emph{e.g.} "6 month".}

\item{breaks}{A vector of class \code{\link[base]{Dates}}. This replaces the arguments \code{from},
\code{to}, and \code{by}.}

\item{overlap}{A number between 0 and 1. The minimum overlapping 
between one match and the interval of classification. Default is 0.5, 
\emph{i.e.} an overlap minimum of 50\%.}

\item{thresholds}{A numeric vector the same length as \code{patterns.labels}. 
The TWDTW dissimilarity thresholds, i.e. the maximum TWDTW cost for consideration 
in the classification. Default is \code{Inf} for all \code{patterns.labels}.}

\item{fill}{A character to fill the classification gaps. 
For signature \code{twdtwTimeSeries} the default is \code{fill="unclassified"}, 
for signature \code{twdtwRaster} the default is \code{fill="unclassified"}.}

\item{filepath}{A character. The path at which to save the raster with results. 
If not provided the function saves in the same directory as the input time series raster.}
}
\value{
An object of class twdtw*.
}
\description{
This function classifies the intervals of a time series 
based on the TWDTW results.
}
\examples{
\dontrun{
  
# Example of TWDTW analysis using raster files 
library(dtwSat)
library(caret) 

# Load raster data 
evi  <- brick(system.file("lucc_MT/data/evi.tif",  package = "dtwSat"))
ndvi <- brick(system.file("lucc_MT/data/ndvi.tif", package = "dtwSat"))
red  <- brick(system.file("lucc_MT/data/red.tif",  package = "dtwSat"))
blue <- brick(system.file("lucc_MT/data/blue.tif", package = "dtwSat"))
nir  <- brick(system.file("lucc_MT/data/nir.tif",  package = "dtwSat"))
mir  <- brick(system.file("lucc_MT/data/mir.tif",  package = "dtwSat"))
doy  <- brick(system.file("lucc_MT/data/doy.tif",  package = "dtwSat"))
timeline <- 
  scan(system.file("lucc_MT/data/timeline", package = "dtwSat"), what="date")

# Create raster time series 
rts <- twdtwRaster(evi, ndvi, red, blue, nir, mir, timeline = timeline, doy = doy)

# Load field samples and projection 
field_samples <- 
  read.csv(system.file("lucc_MT/data/samples.csv", package = "dtwSat"))
proj_str <- 
  scan(system.file("lucc_MT/data/samples_projection", package = "dtwSat"), 
       what = "character")

# Split samples for training (10\%) and validation (90\%) using stratified sampling 
set.seed(1)
I <- unlist(createDataPartition(field_samples$label, p = 0.1))
training_samples <- field_samples[I, ]
validation_samples <- field_samples[-I, ]

# Get time series form raster
training_ts <- getTimeSeries(rts, y = training_samples, proj4string = proj_str)
validation_ts <- getTimeSeries(rts, y = validation_samples, proj4string = proj_str)

# Create temporal patterns 
temporal_patterns <- createPatterns(training_ts, freq = 8, formula = y ~ s(x))

# Set TWDTW weight function 
log_fun <- logisticWeight(-0.1, 50)

# Run TWDTW analysis 
system.time(
  r_twdtw <-
    twdtwApply(x = rts, y = temporal_patterns, weight.fun = log_fun, progress = 'text') 
)

# Plot TWDTW distances for the first year 
plot(r_twdtw, type = "distance", time.levels = 1)

# Classify raster based on the TWDTW analysis 
r_lucc <- twdtwClassify(r_twdtw, progress = 'text')

# Plot TWDTW classification results 
plot(r_lucc, type = "map")

# Assess classification 
twdtw_assess <- 
  twdtwAssess(object = r_lucc, y = validation_samples, 
              proj4string = proj_str, conf.int = .95) 

# Plot map accuracy 
plot(twdtw_assess, type = "accuracy")

# Plot area uncertainty 
plot(twdtw_assess, type = "area")

# Plot misclassified samples  
plot(twdtw_assess, type = "map", samples = "incorrect") 

# Get latex table with error matrix 
twdtwXtable(twdtw_assess, table.type = "matrix")

# Get latex table with error accuracy 
twdtwXtable(twdtw_assess, table.type = "accuracy")

# Get latex table with area uncertainty 
twdtwXtable(twdtw_assess, table.type = "area")

}

\dontrun{
  
# Example of TWDTW analysis using raster files 
library(dtwSat)
library(caret) 

# Load raster data 
evi  <- brick(system.file("lucc_MT/data/evi.tif",  package = "dtwSat"))
ndvi <- brick(system.file("lucc_MT/data/ndvi.tif", package = "dtwSat"))
red  <- brick(system.file("lucc_MT/data/red.tif",  package = "dtwSat"))
blue <- brick(system.file("lucc_MT/data/blue.tif", package = "dtwSat"))
nir  <- brick(system.file("lucc_MT/data/nir.tif",  package = "dtwSat"))
mir  <- brick(system.file("lucc_MT/data/mir.tif",  package = "dtwSat"))
doy  <- brick(system.file("lucc_MT/data/doy.tif",  package = "dtwSat"))
timeline <- 
  scan(system.file("lucc_MT/data/timeline", package = "dtwSat"), what="date")

# Create raster time series 
rts <- twdtwRaster(evi, ndvi, red, blue, nir, mir, timeline = timeline, doy = doy)

# Load field samples and projection 
field_samples <- 
  read.csv(system.file("lucc_MT/data/samples.csv", package = "dtwSat"))
proj_str <- 
  scan(system.file("lucc_MT/data/samples_projection", package = "dtwSat"), 
       what = "character")

# Split samples for training (10\%) and validation (90\%) using stratified sampling 
set.seed(1)
I <- unlist(createDataPartition(field_samples$label, p = 0.1))
training_samples <- field_samples[I, ]
validation_samples <- field_samples[-I, ]

# Get time series form raster
training_ts <- getTimeSeries(rts, y = training_samples, proj4string = proj_str)
validation_ts <- getTimeSeries(rts, y = validation_samples, proj4string = proj_str)

# Create temporal patterns 
temporal_patterns <- createPatterns(training_ts, freq = 8, formula = y ~ s(x))

# Set TWDTW weight function 
log_fun <- logisticWeight(-0.1, 50)

# Run TWDTW analysis 
system.time(
  r_twdtw <-
    twdtwApply(x = rts, y = temporal_patterns, weight.fun = log_fun, progress = 'text') 
)

# Plot TWDTW distances for the first year 
plot(r_twdtw, type = "distance", time.levels = 1)

# Classify raster based on the TWDTW analysis 
r_lucc <- twdtwClassify(r_twdtw, progress = 'text')

# Plot TWDTW classification results 
plot(r_lucc, type = "map")

# Assess classification 
twdtw_assess <- 
  twdtwAssess(object = r_lucc, y = validation_samples, 
              proj4string = proj_str, conf.int = .95) 

# Plot map accuracy 
plot(twdtw_assess, type = "accuracy")

# Plot area uncertainty 
plot(twdtw_assess, type = "area")

# Plot misclassified samples  
plot(twdtw_assess, type = "map", samples = "incorrect") 

# Get latex table with error matrix 
twdtwXtable(twdtw_assess, table.type = "matrix")

# Get latex table with error accuracy 
twdtwXtable(twdtw_assess, table.type = "accuracy")

# Get latex table with area uncertainty 
twdtwXtable(twdtw_assess, table.type = "area")

}

}
\references{
\insertRef{Maus:2019}{dtwSat}
  
  \insertRef{Maus:2016}{dtwSat}
}
\seealso{
\code{\link[dtwSat]{twdtwApply}}, 
\code{\link[dtwSat]{twdtwMatches-class}}, 
\code{\link[dtwSat]{twdtwTimeSeries-class}}, and 
\code{\link[dtwSat]{twdtwRaster-class}},
}
\author{
Victor Maus, \email{vwmaus1@gmail.com}
}
