## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(fig.width=10, fig.height=10, fig.asp = 0.618, out.width = "95%", fig.align = "center", fig.dpi = 150, collapse = FALSE, comment = "#") 

## ----  message=FALSE, warning=FALSE-------------------------------------------
#Load packages
require(spatstat)
require(sp)
require(gstat)
require(parallel)
require(eesim)
require(tidyverse)
require(geosphere)
require(ggplot2)
require(rgeos)
#require(rgdal)
require(dynamAedes)

Sys.setlocale("LC_TIME", "en_GB.UTF-8")  

## -----------------------------------------------------------------------------
gridDim <- 20 # 5000m/250 m = 20 columns and rows
xy <- expand.grid(x=1:gridDim, y=1:gridDim)

## ---- message=FALSE-----------------------------------------------------------
varioMod <- vgm(psill=0.005, range=100, model='Exp') # psill = partial sill = (sill-nugget)
# Set up an additional variable from simple kriging
zDummy <- gstat(formula=z~1, 
                locations = ~x+y, 
                dummy=TRUE,
                beta=1, 
                model=varioMod, 
                nmax=1)
# Generate a randomly autocorrelated predictor data field
set.seed(123)
xyz <- predict(zDummy, newdata=xy, nsim=1)

## -----------------------------------------------------------------------------
utm32N <- "+proj=utm +zone=32 +ellps=WGS84 +datum=WGS84 +units=m +no_defs"

r <- raster(nrow=gridDim, ncol=gridDim, crs=utm32N, ext=extent(1220000,1225000, 5700000,5705000))

values(r)=xyz$sim1
plot(r, main="SAC landscape")

df <- data.frame("id"=1:nrow(xyz), raster::coordinates(r))
bbox <- as(extent(r), "SpatialPolygons")
projection(bbox) <- projection(utm32N)

# Store Parameters for autocorrelation
autocorr_factor <- values(r)

## -----------------------------------------------------------------------------
ndays = 365*1 #length of the time series in days
set.seed(123)
sim_temp <- create_sims(n_reps = 1, 
                        n = ndays, 
                        central = 16, 
                        sd = 2, 
                        exposure_type = "continuous", 
                        exposure_trend = "cos1", exposure_amp = -1.0, 
                        average_outcome = 12,
                        outcome_trend = "cos1",
                        outcome_amp = 0.8, 
                        rr = 1.0055)

## -----------------------------------------------------------------------------
hist(sim_temp[[1]]$x, 
     xlab="Temperature (°C)", 
     main="Histogram of simulated temperatures")

plot(sim_temp[[1]]$date,
     sim_temp[[1]]$x,
     main="Simulated temperatures seasonal trend", 
     xlab="Date", ylab="Temperature (°C)"
     )

## -----------------------------------------------------------------------------
mat <-do.call(rbind, lapply(1:ncell(r), function(x) {
	d_t <- sim_temp[[1]]$x*autocorr_factor[[x]]
	return(d_t)
}))

## ----  message=FALSE, warning=FALSE, hide=TRUE--------------------------------
oldpar <- par(mfrow = c(1,2)) 

## -----------------------------------------------------------------------------
par(mfrow=c(2,1))
hist(mat, xlab="Temperature (°C)", main="Histogram of simulated spatial autocorreled temperature")
hist(sim_temp[[1]]$x, xlab="Temperature (°C)", main="Histogram of simulated temperatures", col="red")
par(mfrow=c(1,1))

## ----  message=FALSE, warning=FALSE, hide=TRUE--------------------------------
par(oldpar) 

## -----------------------------------------------------------------------------
names(mat) <- paste0("d_", 1:ndays)
df_temp <- cbind(df, mat)

## -----------------------------------------------------------------------------
set.seed(123)
pts <- spsample(bbox, 5, type="random")
roads <- spLines(pts)

# Check simulated segment
raster::plot(r)
raster::plot(roads, add=T)

## -----------------------------------------------------------------------------
buff <- buffer(roads, width=250)
crs(buff) <- crs(r)
# Check grid, road segment and buffer
raster::plot(r)
raster::plot(buff, add=T)
raster::plot(roads, add=T, col="red")

## ----  message=FALSE----------------------------------------------------------
df_sp <- df
coordinates(df_sp)=~x+y
df_sp <- raster::intersect(df_sp,buff)

# Check selected cells
raster::plot(r)
raster::plot(buff, add=T)
raster::plot(df_sp, add=T, col="red")

## -----------------------------------------------------------------------------
dist_matrix <- as.matrix(dist(coordinates(df_sp)))

## -----------------------------------------------------------------------------
cc <- df_temp[,c("x","y")]

## -----------------------------------------------------------------------------
colnames(dist_matrix) <- row.names(dist_matrix)

## -----------------------------------------------------------------------------
dist_matrix <- apply(dist_matrix,2,function(x) round(x/1000,1)*1000) 

# An histogram showing the distribution of distances of cells along the road network
hist(dist_matrix, xlab="Distance (meters)")

## -----------------------------------------------------------------------------
set.seed(123)
icellcoords <- df[sample(row.names(dist_matrix),1),c(2:3)]
set.seed(123)
icellid <- df[sample(row.names(dist_matrix),1),1]

raster::plot(r)
raster::plot(buff, add=T)
raster::plot(df_sp, add=T, col="red")
raster::plot(SpatialPoints(icellcoords), add=T, col="blue", pch=21)
raster::plot(SpatialPoints(coords=matrix(coordinates(r)[icellid,],ncol=2)), add=T, col="black", pch=21)

## -----------------------------------------------------------------------------
## Define cells along roads into which introduce propagules on day 1
intro.vector <- icellid
## Define the day of introduction (June 1st is day 1)
str = "2000-06-01"
## Define the end-day of life cycle (August 1st is the last day)
endr = "2000-08-01"
## Define the number of adult females to be introduced
ia = 5000
## Define the number of model iterations
it = 1 # The higher the number of simulations the better
## Define the number of liters for the larval density-dependent mortality
habitat_liters=100
##Define average trip distance
mypDist=1000
## Define the number of parallel processes (for sequential iterations set nc=1)
cl = 1
## Define proj4 string
utm32N = "+proj=utm +zone=32 +ellps=WGS84 +datum=WGS84 +units=m +no_defs" 

## -----------------------------------------------------------------------------
w <- sapply(df_temp[,as.POSIXlt(str)$yday:as.POSIXlt(endr)$yday], function(x) as.integer(x*1000))

## ----results='hide', message=FALSE, warning=FALSE-----------------------------
simout=dynamAedes.m(species="albopictus",
            scale="lc",  
            jhwv=habitat_liters,
            temps.matrix=w,
            cells.coords=cc,
            lat=50.80,
            long=4.44,
            coords.proj4=utm32N,
            road.dist.matrix=dist_matrix,
            avgpdisp=mypDist,
            intro.cells=intro.vector,
            startd=str,
            endd=endr,
            n.clusters=cl, 
            iter=it,
            intro.adults=ia,  
            compressed.output=TRUE, 
            cellsize=250,
            maxadisp=600,
            dispbins=10,
            seeding=TRUE,
            verbose=FALSE
            )

## -----------------------------------------------------------------------------
print(it)
print(length(simout))

## -----------------------------------------------------------------------------
length(simout[[1]])

## -----------------------------------------------------------------------------
dim(simout[[1]][[1]])

## -----------------------------------------------------------------------------
psi(input_sim = simout, eval_date = 60)

## -----------------------------------------------------------------------------
plot(psi_sp(coords = cc, input_sim = simout, eval_date = 60, n.clusters=cl))
raster::plot(buff, add=T)
raster::plot(df_sp, add=T, col="red")
raster::plot(SpatialPoints(icellcoords), add=T, col="blue", pch=21)

## ----message=FALSE, warning=FALSE---------------------------------------------
dd <- max(sapply(simout, function(x) length(x)))#retrieve the maximum number of simulated days
egg <- as.data.frame(adci(simout, eval_date=1:dd, breaks=c(0.25,0.50,0.75), st=1))
juv <- as.data.frame(adci(simout, eval_date=1:dd, breaks=c(0.25,0.50,0.75), st=2))
ad <- as.data.frame(adci(simout, eval_date=1:dd, breaks=c(0.25,0.50,0.75), st=3))
eggd <- as.data.frame(adci(simout, eval_date=1:dd, breaks=c(0.25,0.50,0.75), st=4))

# Date to Julian date
strj <- as.numeric(format(as.Date(str),"%j"))
endrj <- as.numeric(format(as.Date(endr),"%j"))-2

egg$myStage='Egg'
egg$Date=seq.Date(sim_temp[[1]]$date[strj], sim_temp[[1]]$date[endrj], by='day')
juv$myStage='Juvenile'
juv$Date=seq.Date(sim_temp[[1]]$date[strj], sim_temp[[1]]$date[endrj], by='day')
ad$myStage='Adult'
ad$Date=seq.Date(sim_temp[[1]]$date[strj], sim_temp[[1]]$date[endrj], by='day')
eggd$myStage='Diapausing egg'
eggd$Date=seq.Date(sim_temp[[1]]$date[strj], sim_temp[[1]]$date[endrj], by='day')

outdf=bind_rows(egg, juv, ad, eggd) %>% 
  as_tibble()

outdf %>% 
  mutate(myStage=factor(myStage, levels= c('Egg', 'Diapausing egg', 'Juvenile', 'Adult'))) %>% 
  ggplot( aes(y=`50%`,x=Date, group=factor(myStage),col=factor(myStage))) +
  ggtitle("Ae. albopictus Interquantile range abundance")+
  geom_line(linewidth=0.8)+
  geom_ribbon(aes(ymin=`25%`,ymax=`75%`,fill=factor(myStage)),
              col="white",
              alpha=0.2,
              outline.type="full")+
  labs(x="Date", y="Interquantile range abundance", col="Stage", fill="Stage")+
  facet_wrap(~myStage, scales = "free")+
  theme_light()+
  theme(legend.pos="bottom",  text = element_text(size=14) , strip.text = element_text(face = "italic"))

## -----------------------------------------------------------------------------
r=adci_sp(simout, coords=cc, eval_date=60, breaks=c(0.025,0.975), stage=3)
plot(r)


## -----------------------------------------------------------------------------
x=icci(simout, eval_date=1:60, breaks=c(0.25,0.50,0.75))
head(x)
tail(x)

## -----------------------------------------------------------------------------
x=dici(simout, coords=cc, eval_date=seq(1,60,length.out=60), breaks=c(0.25,0.50,0.75), space=FALSE)
plot(`0.25`~day,x,type="l",ylab="Population dispersal (in meters) from cell of introduction",xlab="days from introduction")
lines(`0.5`~day,x,type="l", col="red")
lines(`0.75`~day,x,type="l")

