% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/idd.R
\docType{class}
\name{Idd}
\alias{Idd}
\title{Parse, Query and Modify EnergyPlus Input Data Dictionary (IDD)}
\description{
eplusr provides parsing of and programmatic access to EnergyPlus
Input Data Dictionary (IDD) files, and objects. It contains all data needed
to parse EnergyPlus models. \code{Idd} class provides parsing and printing while
\link{IddObject} provides detailed information of curtain class.
}
\section{Overview}{


EnergyPlus operates off of text input files written in its own Input
Data File (IDF) format. IDF files are similar to XML files in that they are
intended to conform to a data schema written using similar syntax. For XML,
the schema format is XSD; for IDF, the schema format is IDD. For each release
of EnergyPlus, valid IDF files are defined by the "Energy+.idd" file shipped
with the release.

eplusr tries to detect all installed EnergyPlus in default installation
locations when loading, i.e. \code{C:\\EnergyPlusVX-X-0} on Windows,
\code{/usr/local/EnergyPlus-X-Y-0} on Linux, and
\code{/Applications/EnergyPlus-X-Y-0} on macOS and stores all found locations
internally. This data is used to locate the distributed "Energy+.idd" file of
each EnergyPlus version. And also, every time an IDD file is parsed, an \code{Idd}
object is created and cached in an environment.

Parsing an IDD file starts from \code{\link[=use_idd]{use_idd()}}. When using \code{\link[=use_idd]{use_idd()}}, eplusr
will first try to find the cached \code{Idd} object of that version, if possible.
If failed, and EnergyPlus of that version is available (see \code{\link[=avail_eplus]{avail_eplus()}}),
the \code{"Energy+.idd"} distributed with EnergyPlus will be parsed and cached. So
each IDD file only needs to be parsed once and can be used when parsing every
IDF file of that version.

Internally, the powerful \href{https://cran.r-project.org/package=data.table}{data.table}
package is used to speed up the whole IDD parsing process and store the
results. However, it will still take about 2-3 sec per IDD. Under the hook,
eplusr uses a SQL-like structure to store both IDF and IDD data in
\link[data.table:data.table]{data.table::data.table} format. Every IDD will be parsed and stored in
four tables:
\itemize{
\item \code{group}: contains group index and group names.
\item \code{class}: contains class names and properties.
\item \code{field}: contains field names and field properties.
\item \code{reference}: contains cross-reference data of fields.
}
}

\section{Usage}{


\preformatted{
idd <- use_idd(ver)
idd$version()
idd$build()
idd$group_index(group = NULL)
idd$group_name()
idd$from_group(class)
idd$class_index(class = NULL, by_group = FALSE)
idd$class_name(index = NULL, by_group = FALSE)
idd$required_class_name()
idd$unique_class_name()
idd$extenesible_class_name()
idd$is_valid_group(group)
idd$is_valid_class(class)
idd$object_relation(class, direction = c("all", "ref_by", "ref_to"))
idd$object(class)
idd$objects(class)
idd$objects_in_relation(class, direction = c("ref_to", "ref_by"))
idd$objects_in_group(group)
idd$ClassName
idd[[ClassName]]
idd$to_table(class, all = FALSE)
idd$to_string(class, leading = 4L, sep_at = 29L, sep_each = 0L, all = FALSE)
idd$print()
print(idd)
}
}

\section{Arguments}{

\itemize{
\item \code{ver}: A valid EnergyPlus IDD version, e.g. 8.8, "8.6.0".
\item \code{idd}: An \code{Idd} object.
\item \code{group}: A character vector of valid group names. For
\code{$objects_in_group()}, a single string of valid group name.
\item \code{index}: An integer vector giving indexes of name appearance in
the IDD file of specified classes.
\item \code{class}: A character vector of valid class names. For \code{$object_relation()}
and \code{$objects_in_relation()}, a single string of valid class name.
\item \code{ClassName}: A single string of valid class name.
\item \code{direciton}: The relation direction to extract. Should be either \code{"all"},
\code{"ref_to"} or "ref_by". For \code{$objects_in_relation()}, only \code{"ref_to"} and
\code{"ref_by"} are acceptable.
\item \code{all}: If \code{TRUE}, all fields in specified classes are returned. If \code{FALSE},
only minimum required fields are returned. Default: \code{FALSE}.
\item \code{leading}: Leading spaces added to each field. Default: \code{4L}.
\item \code{sep_at}: The character width to separate value string and field string.
Default: \code{29L} which is the same as IDF Editor.
\item \code{sep_each}: A single integer of how many empty strings to insert between
different classes. Default: \code{0}.
\item \code{by_group}: If \code{TRUE}, a list is returned which separates class indexes or
names by the group they belongs to. Default: \code{FALSE}.
}
}

\section{Detail}{


\code{$version()} returns the IDD version in
\link[base:numeric_version]{numeric_version}.

\code{$build()} returns the build tag string.

\code{$group_index()} returns integer indexes (indexes of name appearance in
the IDD file) of specified groups.

\code{$group_name()} returns all group names.

\code{$from_group()} returns the names of group that specified classes belongs to.

\code{$is_valid_group()} return \code{TRUE} if the input is a valid group name.

\code{$class_index()} returns integer indexes (indexes of name appearance in
the IDD file) of specified classes. If \code{by_group} is \code{TRUE}, a list is returned
which separate class indexes by the group they belong to. Otherwise, an
integer vector is returned.

\code{$class_name()} returns class names of specified class indexes. If \code{by_group}
is \code{TRUE}, a list is returned which separate class names by the group they
belong to. Otherwise, a character vector is returned.

\code{$required_class_name()} returns the names of all required classes.

\code{$unique_class_name()} returns the names of all unique-object classes.

\code{$extensible_class_name()} returns the names of all extensible classes.

\code{$is_valid_class()} return \code{TRUE} if the input is a valid class name.

\code{$object()} returns an \link{IddObject} of specified class.

\code{$objects()} returns a list of \link{IddObject}s of specified classes.

\code{$object_relation()} returns an \code{IddRelation} object which contains field
data that have relation with specified class. For instance, if
\code{idd$object_relation("Class 1", "ref_to")} gives results below:\preformatted{-- Refer to Others ---------------------
  Class: <Class 1>
  +- Field: <1: Field 1>
  |  v~~~~~~~~~~~~~~~~~~
  |  \- Class: <Class 2>
  |     \- Field: <2: Field 2>
  |
  \- Field: <2: Field 2>
}

This means that \code{Field 2} in class \code{Class 1} does not refer to any other fields.
But \code{Field 1} in class \code{Class 2} refers to \code{Field 2} in class named \code{Class 2}.

\code{$objects_in_relation()} returns a list of \link{IddObject}s that have relations.
The first element is always the \link{IddObject} of that class. If specified class
does not have any relation with other classes, a list that only contains the
\link{IddObject} of that class is returned.  For instance,
\code{idd$objects_in_relation("Construction", "ref_to")} will return a list of an
\link{IddObject} of class \code{Construction} and also all \link{IddObject}s that class
\code{Construction} refers to; similarly, \code{idd$objects_in_relation("Construction", "ref_by")} will return a list of an \link{IddObject} of class \code{Construction} and
also all \link{IddObject}s that refer to class \code{Construction}.

\code{$objects_in_group()} returns a list of \link{IddObject}s of specified group.

eplusr also provides custom S3 method of \code{$} and \code{[[} to make it
more convenient to get a single \link{IddObject}. Basically, \code{idd$ClassName} and
\code{idd[[ClassName]]}, is equivalent to \code{idd$object(ClassName)[[1]]}.
Here, \code{ClassName} is a single valid class name where all characters other
than letters and numbers are replaced by a underscore \code{_}.

For details about \link{IddObject}, please see \link{IddObject} class.

\code{$to_table()} returns a \link[data.table:data.table]{data.table} that contains
core data of specified class. It has 3 columns:
\itemize{
\item \code{class}: Character type. Class names.
\item \code{index}: Integer type. Field indexes.
\item \code{field}: Character type. Field names.
}

\code{$to_string()} returns empty objects of specified class in a character vector
format. It is formatted exactly the same as in IDF Editor.

\code{$print()} prints basic info ob current Idd object, including version, build
tag and total class number.
}

\examples{
\dontrun{
# get the Idd object of EnergyPlus v8.8
idd <- use_idd(8.8, download = "auto")

# version
idd$version()

# build
idd$build()

# all group names
str(idd$group_name())

# all class names
str(idd$class_name())

# all required class names
str(idd$required_class_name())

# all unique class names
str(idd$unique_class_name())

# IddObject of SimulationControl class
idd$object("SimulationControl")
# OR
idd$SimulationControl
# OR
idd[["SimulationControl"]]

# IddObject of Construction and Material class
idd$objects(c("Construction", "Material"))

# show all classes that refer to Material class
length(idd$object_relation("Material", "ref_by"))

# IddObjects that refer to class Construction
length(idd$objects_in_relation("Construction", "ref_by"))

# IddObjects that class Construction refers to
length(idd$objects_in_relation("Construction", "ref_to"))

# All IddObjects in group Schedules
length(idd$objects_in_group("Schedules"))

# Extract core data of class Material and Construction
idd$to_table(c("Material", "Construction"))

# Get empty Material object and Construction object in a character vector
idd$to_string(c("Material", "Construction"))
}
}
\references{
\href{https://github.com/NREL/EnergyPlus/tree/develop/src/IDF_Editor}{IDFEditor},
\href{https://openstudio-sdk-documentation.s3.amazonaws.com/cpp/OpenStudio-2.5.1-doc/utilities/html/idf_page.html}{OpenStudio utilities library}
}
\seealso{
\link{IddObject} class which provides detailed information of
curtain class
}
\author{
Hongyuan Jia
}
