% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/build_resources.R
\docType{methods}
\name{fhir_build_bundle}
\alias{fhir_build_bundle}
\alias{fhir_build_bundle,data.frame-method}
\alias{fhir_build_bundle,list-method}
\title{Build a FHIR bundle}
\usage{
fhir_build_bundle(
  table,
  brackets,
  resource_type,
  bundle_type = "transaction",
  verbose = 1
)

\S4method{fhir_build_bundle}{data.frame}(
  table,
  brackets,
  resource_type,
  bundle_type = "transaction",
  verbose = 1
)

\S4method{fhir_build_bundle}{list}(table, brackets, bundle_type = "transaction", verbose = 1)
}
\arguments{
\item{table}{A wide table as produced by \code{\link[=fhir_crack]{fhir_crack()}}, possibly modified (see details) or a named list
of wide tables, if different resource types have to be included in the same bundle. In this case the names of
the list elements must correspond to the resource type represented in the table!}

\item{brackets}{A character vector of length one. The brackets used for cracking.}

\item{resource_type}{A character vector of length one or \linkS4class{fhir_resource_type} object
indicating which resource type is represented in the table, if a single table is provided. This argument is
ignored when \code{table} is a named list of tables.}

\item{bundle_type}{A character vector of length one defining the bundle type. Will usually be
either \code{"transaction"} (the default) or \code{"batch"}.}

\item{verbose}{An integer vector of length one. If 0, nothing is printed, if > 0 progress message is printed. Defaults to 1.}
}
\value{
A \linkS4class{fhir_bundle_xml} object.
}
\description{
This function takes a table as produced by \code{\link[=fhir_crack]{fhir_crack()}} with \code{format="wide"} and builds a \linkS4class{fhir_bundle_xml} object from it. It is primarily used
to create transaction/batch bundles to POST back to a FHIR server. The column names of the table must represent the XPath expression of the
respective element with indices for repeating items. A table like this is produced when FHIR resources have been cracked with \code{\link[=fhir_crack]{fhir_crack()}} without
assigning explicit column names in the \linkS4class{fhir_design}/\linkS4class{fhir_table_description} and the format has been set to \code{"wide"}.
}
\details{
The typical use case would look like this:
\enumerate{
\item Download resources from a server with \code{\link[=fhir_search]{fhir_search()}}
\item Crack to wide format them with \code{\link[=fhir_crack]{fhir_crack()}}
\item Do something to values (e.g. some kind of anonymization)
\item Translate the data back into FHIR resources with \code{\link[=fhir_build_bundle]{fhir_build_bundle()}}
\item Post the resources to a server
}

A FHIR bundle that can be POSTed to a server is usually of type \code{transaction} or \code{batch}. Each entry of these bundles consists of the resource itself
as well as an instruction for the server of what to to with the resource. A very simple example looks like this:

\if{html}{\out{<div class="sourceCode">}}\preformatted{<Bundle>
   <type value="transaction"/>
  <entry>
     <resource>
        <Patient>
           <id value="id1"/>
	       <address>
	          <city value="Amsterdam"/>
	          <country value="Netherlands"/>
	       </address>
	       <name>
	          <given value="Marie"/>
	       </name>
        </Patient>
    </resource>
    <request>
	   <method value="POST"/>
	   <url value="Patient"/>
    </request>
 </entry>
 <entry>
     <resource>
        <Patient>
           <id value="id2"/>
	       <address>
	          <city value="Paris"/>
	          <country value="France"/>
	       </address>
	       <name>
	          <given value="Anne"/>
	       </name>
        </Patient>
    </resource>
    <request>
	   <method value="POST"/>
	   <url value="Patient"/>
    </request>
 </entry>
</Bundle>
}\if{html}{\out{</div>}}

In this example the bundle contains two Patient resources that are sent to server with a POST. For more information the structure of transaction/batch bundles,
please see the FHIR documentation at https://www.hl7.org/fhir/http.html and https://www.hl7.org/fhir/bundle.html.

In the table, each row corresponds to one resource that is created. To add the information for the \code{request} element of the bundle,
this table has to be augmented with two columns named \code{request.method} and \code{request.url}, which contain the respective HTTP verb and URL for the resource.
If these columns are not added to the table, \code{\link[=fhir_build_bundle]{fhir_build_bundle()}} still builds bundles from it, but those bundles will not be POSTable to a server. See examples.
}
\examples{
#unserialize example
bundles <- fhir_unserialize(bundles = example_bundles1)

#crack fhir resources
table_desc_pat <- fhir_table_description(
    resource = "Patient",
    brackets = c("[", "]"),
    sep      = " ",
    format   = "wide"
)

df <- fhir_crack(bundles = bundles, design = table_desc_pat)

#add request info to table
request <- data.frame(
    request.method = c("POST", "PUT"),
    request.url    = c("Patient", "Patient/id3")
)

request_df <- cbind(df, request)

#build bundle
bundle <- fhir_build_bundle(table          = request_df,
                            brackets       = table_desc_pat@brackets,
                            resource_type  = "Patient",
                            bundle_type    = "transaction")

#print to console
cat(toString(bundle))
}
\seealso{
\code{\link[=fhir_crack]{fhir_crack()}},\code{\link[=fhir_cast]{fhir_cast()}}, \code{\link[=fhir_build_resource]{fhir_build_resource()}}, \code{\link[=fhir_post]{fhir_post()}}
}
