/*  Routines for manipulating B-splines.  These are intended for use with
 *  S or S-PLUS or R.
 *
 *     Copyright (C) 2015 Michel Grzebyk.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * These functions are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, a copy is available at
 *  http://www.r-project.org/Licenses/
 *
 * The routines are loosely based on the pseudo-code in Schumacher (Wiley,
 * 1981) and the CMLIB library DBSPLINES.
 */

#include <R.h>
#include <Rinternals.h>

#ifdef ENABLE_NLS
#include <libintl.h>
#define _(String) dgettext ("splines", String)
#else
#define _(String) (String)
#endif

#ifndef EVAL_SPLINEPARAM
#include "SplineParam.h"
#endif


SEXP eval_lc_spline_basis(SEXP knots, SEXP order, SEXP Matrices, SEXP intercept, SEXP xvals, SEXP beta, SEXP outerok)
{
/* evaluate linear combination of the non-zero B-spline basis functions at xvals, using matrices generated by packag::orthogonalsplinbasis */
/* 
 knots : vector of ordered unreplicated INTERIOR knots 
Matrices : a vectorized array of dim order X nbases X number_of_intervales(knots) 
  where nbases is the number of bases of the non integrated, non derived splines 
order : order of the splines (see package orthogonalsplinbasis
intercept : wehtehr first basis is included
xvals : vector values at which bases are computed
beta : vector of the linear combination
 */
	R_len_t i, j, k, nknots, theorder, nbases, nbeta, nx, oo;
	R_len_t theinterval, firstbasis, mfl;
	double *rknots, *rMatrices, *rxvals, *rbeta, *rcl;
	SEXP cl;
	SEXP dims;
	double temp, tempcl, *U, u, outer_val;
	
	
	PROTECT(knots = coerceVector(knots, REALSXP));
	PROTECT(order = coerceVector(order, INTSXP));
	PROTECT(intercept = coerceVector(intercept, INTSXP));
	PROTECT(Matrices = coerceVector(Matrices, REALSXP));
	PROTECT(xvals = coerceVector(xvals, REALSXP));
	PROTECT(beta = coerceVector(beta, REALSXP));
	PROTECT(outerok = coerceVector(outerok, LGLSXP));


	rknots = REAL(knots); 
	nknots = length(knots);
	theorder = INTEGER(order)[0];

	dims = getAttrib(Matrices, R_DimSymbol);
	if( LENGTH(dims) < 3 ){
		error("'Matrices' must be an array with 3 dim");   
	}
	nbases = INTEGER(dims)[1];
	
	rxvals = REAL(xvals); 
	nx = length(xvals);

	rbeta = REAL(beta); 
	nbeta = length(beta);

	
	firstbasis = (INTEGER(intercept)[0]==0);
	rMatrices = REAL(Matrices);
		
	PROTECT(cl = allocVector(REALSXP, nx));
	rcl = REAL(cl);

	if(nbeta < nbases-firstbasis) {
		error("length of 'beta' must be at least %d", nbases-firstbasis);    
	} 	
	
	oo = asLogical(outerok);
	
	U = (double *) R_alloc( theorder, sizeof(double));
	if(oo == NA_LOGICAL) {
		error("'outer.ok' must be TRUE or FALSE");    
	} else  if (oo) {
		outer_val = 0.0;
	} else {
		outer_val = R_NaN;
	}

	U[0]=1.0;
	for(i = 0; i < nx; i++) {
	    if (ISNAN(rxvals[i])) {
		   rcl[i] = R_NaN;
	    } 
		else {
/* find the interval within the range of all the knots (which include boundaries) 
   of rxvals[i], rightmost_close=TRUE, all_inside = FALSE */ 
/*		    theinterval = findInterval(rknots, nknots, rxvals[i], 1, 0 , theinterval, &mfl );  */
/*		    if (theinterval == 0 || theinterval == nknots  ) {                                 */
		    if (rxvals[i] < rknots[theorder-1] || rxvals[i] > rknots[nknots - theorder]  ) {                            
/* out of the boundary knots interval                                                */             
			    rcl[i] = outer_val;
		    } 
			else {
			    mfl = 0;                                                                            
			    theinterval = findInterval2(rknots, nknots, rxvals[i], 1, 0 , FALSE, theorder, &mfl ); 
			    if( theinterval > nknots - theorder) {
				    /* xx[i] is the rightmost boundary knot */
				    theinterval = nknots - theorder;
			    }
			    u = (rxvals[i] - rknots[theinterval-1])/(rknots[theinterval]-rknots[theinterval-1]);
			    for ( j = 1; j < theorder ; j++) {
				    U[j] = pow(u, (double)j);
			    }
			    /* the usefull matrix is the (theinterval - theorder +1)th matrix of Matrices */
			    theinterval = theinterval - theorder;
				tempcl = 0;
			    for (k = firstbasis; k < nbases; k++) {
				    temp = 0;
				    for (int j = 0; j < theorder ; j++) {
					    temp += U[j] * rMatrices[theorder*nbases*theinterval+ theorder*k + j];
				    }
					tempcl = tempcl + temp * rbeta[k-firstbasis];
			    }
			    rcl[i] = tempcl;
		    }
	    }
	    
	}
	
	UNPROTECT(8);
	return(cl);
}



/* evaluate linear combination of the non-zero B-spline basis functions at one value rxval, 
   using matrices generated by packag::orthogonalsplinbasis                               */
/* pas d'objet R                                                                          */
double r_eval_lc_spline_basis(double *rknots, R_len_t nknots, R_len_t nbases, R_len_t theorder, R_len_t firstbasis, 
			      double *rMatrices, double rxval, double *rbeta, double outer_val){
	double rcl;
	R_len_t j, k, theinterval, mfl;
	double temp, *U, u;
	if (ISNAN(rxval)) {
		rcl = R_NaN;
	} 
	else {
		U = (double *) R_alloc( theorder, sizeof(double));
		U[0]=1.0;

		theinterval= 1;
		mfl = 0;
/* find the interval within the range of all the knots (which include boundaries) 
   of rxval, rightmost_close=TRUE, all_inside = FALSE */ 
/*		theinterval = findInterval(rknots, nknots, rxval, 1, 0 , theinterval, &mfl );  */
/*		if (theinterval == 0 || theinterval == nknots  ) {                             */
                if (rxval < rknots[theorder-1] || rxval > rknots[nknots - theorder]  ) {                         
/* out of the boundary knots interval                                                */             
			rcl = outer_val;
		} 
		else {
			mfl = 0;                                                                            
			theinterval = findInterval2(rknots, nknots, rxval, 1, 0 , FALSE, theorder, &mfl ); 
			if( theinterval > nknots - theorder) {
				/* xx[i] is the rightmost boundary knot */
				theinterval = nknots - theorder;
			}
			u = (rxval - rknots[theinterval-1])/(rknots[theinterval]-rknots[theinterval-1]);
			for ( j = 1; j < theorder ; j++) {
				U[j] = pow(u, (double)j);
			}
			/* the usefull matrix is the (theinterval - theorder +1)th matrix of Matrices */
			theinterval = theinterval - theorder;
			rcl = 0;
			for (k = firstbasis; k < nbases; k++) {
				temp = 0;
				for (int j = 0; j < theorder ; j++) {
					temp += U[j] * rMatrices[theorder*nbases*theinterval+ theorder*k + j];
				}
				rcl = rcl + temp * rbeta[k-firstbasis];
			}
		}
	}
	return(rcl);
}








