% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kernels.R
\name{kernels}
\alias{kernels}
\alias{fmcmc_kernel}
\alias{fmcmc-kernel}
\alias{kernel_new}
\alias{print.fmcmc_kernel}
\alias{kernel_unif}
\alias{kernel_unif_reflective}
\alias{kernel_normal}
\alias{kernel_normal_reflective}
\title{Various kernel functions for MCMC}
\usage{
kernel_new(proposal, logratio = NULL, ...)

\method{print}{fmcmc_kernel}(x, ...)

kernel_unif(min. = -1, max. = 1, fixed = FALSE, scheme = "joint")

kernel_unif_reflective(min. = -1, max. = 1, lb = min., ub = max.,
  fixed = FALSE, scheme = "joint")

kernel_normal(mu = 0, scale = 1, fixed = FALSE, scheme = "joint")

kernel_normal_reflective(mu = 0, scale = 1,
  lb = -.Machine$double.xmax, ub = .Machine$double.xmax,
  fixed = FALSE, scheme = "joint")
}
\arguments{
\item{proposal, logratio}{Functions. The function receives a single argument, an environment.
This functions are called later within \link{MCMC} (see details).}

\item{...}{In the case of \code{kernel_new}, further arguments to be stored with
the kernel.}

\item{x}{An object of class \code{fmcmc_kernel}.}

\item{min., max.}{Passed to \link[stats:runif]{stats::runif}.}

\item{fixed}{Logical scalar or vector. When \code{TRUE} fixes the corresponding
parameter, avoiding new proposals.}

\item{scheme}{scheme in which proposals are made (see details).}

\item{lb, ub}{Either a numeric vector or a scalar. Lower and upper bounds for
bounded kernels.}

\item{mu, scale}{Either a numeric vector or a scalar. Proposal mean and scale.}
}
\description{
Various kernel functions for MCMC
}
\details{
The objects \code{fmcmc_kernels} are environments that in general contain the
following objects:
\itemize{
\item \code{proposal}: The function used to propose changes in the chain based
on the current state. The function must return a vector of length equal
to the number of parameters in the model.
\item \code{logration}: This function is called after a new state has been proposed,
and is used to compute the log of the Hastings ratio.

In the case that the \code{logratio} function is not specified, then it is assumed
that the transition kernel is symmetric, this is, logratio is then implemented
as \code{function(env) {env$f1 - env$f0}}
\item \code{...}: Further objects that are used within those functions.
}

Both functions, \code{proposal} and \code{logratio}, receive a single argument, an
environment, which is passed by the \link{MCMC} function during each step using
the function \link{environment}. The passed
environment is actually the environemnt in which the \code{MCMC} function is running,
in particular, this environment contains the following objects:

\tabular{lcl}{
\strong{Object} \tab \tab \strong{Description} \cr
\code{i} \tab \tab Integer. The current iteration. \cr
\code{theta1} \tab \tab Numeric vector. The last proposed state. \cr
\code{theta0} \tab \tab Numeric vector. The current state \cr
\code{f}\tab \tab The log-unnormalized posterior function (a wrapper of \code{fun} passed
to \link{MCMC}). \cr
\code{f1} \tab \tab The last value of \code{f(theta1)} \cr
\code{f0} \tab \tab The last value of \code{f(theta0)} \cr
\code{kernel} \tab \tab The actual \code{fmcmc_kernel} object. \cr
\code{ans} \tab \tab The matrix of samples defined up to \code{i - 1}.
}

These are the core component of the \code{MCMC} function. The following block
of code is how this is actually implemented in the package:\preformatted{for (i in 1L:nsteps) {
  # Step 1. Propose
  theta1[] <- kernel$proposal(environment())
  f1       <- f(theta1)
  
  # Checking f(theta1) (it must be a number, can be Inf)
  if (is.nan(f1) | is.na(f1) | is.null(f1)) 
    stop(
      "fun(par) is undefined (", f1, ")",
      "Check either -fun- or the -lb- and -ub- parameters.",
      call. = FALSE
    )
  
  # Step 2. Hastings ratio
  if (R[i] < kernel$logratio(environment())) {
    theta0 <- theta1
    f0     <- f1
  }
  
  # Step 3. Saving the state
  ans[i,] <- theta0
  
}
}

For more details see the vignette \code{vignette("user-defined-kernels", "fmcmc")}.
}
\section{Proposal scheme}{


The parameter \code{scheme} present on the currently available kernels sets the way
in which proposals are made. By default, \code{scheme = "joint"}, proposals are done
jointly, this is, at each step of the chain we are proposing new states for
each parameter of the model. When \code{scheme = "ordered"}, a sequential update schema
is followed, in which, at each step of the chain, proposals are made one
variable at a time, If \code{scheme = "random"}, proposals are also made one
variable at a time but in a random scheme.

Finally, users can specify their own sequence of proposals for the variables
by passing a numeric vector to \code{scheme}, for example, if the user wants to make
sequential proposals following the scheme 2, 1, 3, then scheme must be set to
be \code{scheme = c(2, 1, 3)}.
}

\section{Creating your own kernels}{

The function \code{kernel_new} is a helper function that allows creating
\code{fmcmc_kernel} which is used with the \code{MCMC} function. The \code{fmcmc_kernel}
are the backbone of the \link{MCMC} function.
}

\section{Kernels}{

The \code{kernel_unif} function provides a uniform transition kernel. This (symmetric)
kernel function by default adds the current status values between [-1,1].


The \code{kernel_unif_reflective} is similar to \code{kernel_unif} with the
main difference that proposals are bounded to be within \code{[lb, ub]}.


The \code{kernel_normal} function provides the canonical normal kernel
with symmetric transition probabilities.


The \code{kernel_normal_reflective} implements the normal kernel with reflective
boundaries. Lower and upper bounds are treated using reflecting boundaries, this is,
if the proposed \eqn{\theta'} is greater than the \code{ub}, then \eqn{\theta' - ub}
is subtracted from \eqn{ub}. At the same time, if it is less than \code{lb}, then
\eqn{lb - \theta'} is added to \code{lb} iterating until \eqn{\theta} is within
\code{[lb, ub]}.

In this case, the transition probability is symmetric (just like the normal
kernel).
}

\examples{

# Example creating a multivariate normal kernel using the mvtnorm R package
# for a bivariate normal distribution
library(mvtnorm)

# Define your Sigma
sigma <- matrix(c(1, .2, .2, 1), ncol = 2)

# How does it looks like?
sigma
#      [,1] [,2]
# [1,]  1.0  0.2
# [2,]  0.2  1.0

# Create the kernel
kernel_mvn <- kernel_new(
  proposal = function(env) {
  env$theta0 + as.vector(mvtnorm::rmvnorm(1, mean = 0, sigma = sigma.))
  },
  sigma. = sigma
)

# As you can see, in the previous call we passed sigma as it will be used by
# the proposal function
# The logaratio function was not necesary to be passed since this kernel is
# symmetric.


}
