#' @name hap_summary
#' @title Summary Hap Results
#' @description
#' A function used for summarize hapResult to visualization and calculation.
#' @note
#' If the user have changed the default `hapPrefix` in `vcf2hap()` or `seqs2hap()`,
#' then the parameter `hapPrefix` is needed.
#' Furthermore, a multi-letter prefix of hap names is possible.
#' @details
#' It is suggested to use the result of `vcf2hap()` or `seqs2hap()`
#' as input directly.
#' However the user can import previously hap result from local file
#' with `import_hap()`
#'
#' @usage
#' hap_summary(hap,
#'             hapPrefix = "H",
#'             file = file)
#' @examples
#' data("geneHapR_test")
#' hapSummary <- hap_summary(hapResult, hapPrefix = "H")
#' @importFrom utils write.table
#' @param hap object of hapResult class, generated by `vcf2hap()`
#' or `seqs2hap` or `import_hap()`
#' @param hapPrefix prefix of hap names, default as "H"
#' @param file file path where to save the hap summary result.
#' If missing, nothing will be saved to disk.
#' @export
#' @return hapSummary, first four rows are fixed to meta information: CHR, POS, INFO, ALLELE
#' Hap names were placed in first column, Accessions and freqs were placed at the last two columns.
hap_summary <- function(hap,
                        hapPrefix = "H",
                        file = file) {
    if (!inherits(hap, 'hapResult'))
        stop("\n'hap' must be of 'hapResult' class")
    requireNamespace('tidyr')
    hapSummarys <- hap %>% data.frame(check.names = FALSE)
    hapfre <- table(hapSummarys[, 1])
    hapfre <- hapfre[stringr::str_starts(names(hapfre), hapPrefix)]
    hapSummarys <- hapSummarys %>% tidyr::chop(cols = "Accession")
    prob = hapSummarys[5:nrow(hapSummarys), 1]
    hapSummarys$freq[5:nrow(hapSummarys)] <- hapfre[prob]
    Acc <- c()
    nAcc = length(hapSummarys$Accession)
    for (i in seq_len(nAcc)) {
        Acc[i] <- paste(hapSummarys$Accession[[i]], collapse = ";")
    }
    hapSummarys$Accession <- Acc

    # write result to disk
    if (!missing(file))  {
        utils::write.table(
            hapSummarys,
            file = file,
            sep = "\t",
            quote = FALSE,
            row.names = FALSE,
            col.names = FALSE
        )
    }
    class(hapSummarys) <- c("hapSummary", "data.frame")
    attr(hapSummarys, "options") <- attr(hap, "options")
    attr(hapSummarys, "hap2acc") <- attr(hap, "hap2acc")

    return(hapSummarys)
}


#' @name plotHapTable
#' @title plotHapTable
#' @usage
#' plotHapTable(hapSummary,
#'              hapPrefix = "H",
#'              title = "",
#'              geneName = geneName,
#'              INFO_tag = NULL,
#'              tag_split = tag_split,
#'              tag_field = tag_field,
#'              tag_name = tag_name,
#'              displayIndelSize = 0, angle = c(0,45,90),
#'              replaceMultiAllele = TRUE,
#'              ALLELE.color = "grey90")
#' @examples
#' data("geneHapR_test")
#' plotHapTable(hapResult)
#' @description display hap result as a table-like figure
#' @details
#' In **VCF** files, the INFO field are represented as tag-value
#' pairs, where the tag and value are separated by an equal sign, ie "=", and
#' pairs are separated by colons, ie ";".
#'
#' If hapSummarys were generated from sequences, INFO row is null.
#' If hapSummarys were generated from VCF, INFO was take from the INFO column
#' in the source VCF file.
#' Some tag-values may contains more than one value separated by
#' "|", eg.: "ANN" or "snpEFF" added by 'snpeff' or other software. For those
#' fields we need specified value of `tag_field = "ANN"` and `tag_split = "[\|]"`,
#' it's suggest specified the value of `tag_name` for display in figure.
#'
#' 'snpeff', a toolbox for genetic variant annotation and functional effect
#' prediction, will add annotations to INFO filed in VCF file under a tag
#' named as "ANN". The annotations contains several fields separated by "|".
#' eg.:
#'  1.  Allele
#'  2.  Annotation
#'  3.  Annotation_Impact
#'  4.  Gene_Name
#'  5.  Gene_ID
#'  6.  Feature_Type
#'  7.  Feature_ID
#'  8.  Transcript_BioType
#'  9.  Rank
#'  10. HGVS.c
#'  11. HGVS.p
#'  12. cDNA.pos/cDNA.length
#'  ... ...
#'
#' However, the INFO in hapResults may missing annotations that we need.
#' In this case, we can custom INFOs in hapSummarys with `addINFO()`.
#' Once the needed annotations were included in hap results, we can display
#' them with `plotHapTable()` by specify the value of `INFO_tag`.
#'
#' @seealso
#' \code{\link[geneHapR:addINFO]{addINFO()}}
#' @param hapSummary object of hapSummary class
#' @param ALLELE.color the color of ALLELE row, default as "grey90"
#' @param hapPrefix prefix of haplotype names. Default as "H"
#' @param title the main title of the final figure
#' @param displayIndelSize display indels with max size of `displayIndelSize`,
#' If set as 0, all indels will convert into "i*" of which "i" represents "indel".
#' @param angle the angle of coordinates, should be one of 0, 45 and 90
#' @param replaceMultiAllele whether to replace MultiAllele with "T*",
#' default as `TRUE`.
#' @param geneName character, will be used for filter INFO filed of ANN
#' @param INFO_tag The annotations in the INFO field are represented as tag-value
#' pairs, where the tag and value are separated by an equal sign, ie "=", and
#' pairs are separated by colons, ie ";".
#' For more information please see details.
#' @param tag_field integer, if a tag-value contains more than one fields,
#' user need to specified which field should be display. If `tag_field` set as
#' 0, the whole contents will be displayed. Default as 0.
#' @param tag_split usually, the value of tag-value contains one information.
#' However, if a tag contains more than one fields, eg "ANN", then `tag_split` is
#' needed. When `INFO_tag` was set as "ANN" or "SNPEFF", `tag_split` will be set
#' as "|" by default, see details.
#' @param tag_name tag name is displayed in Hap figure. If `tag_name` is
#' missing, will take the value of `INFO_tag`.
#' @importFrom stringr str_starts
#' @importFrom stringr str_length
#' @import ggplot2
#' @export
#' @return ggplot2 object
plotHapTable <- function(hapSummary,
                         hapPrefix = "H",
                         title = "",
                         geneName = geneName,
                         INFO_tag = NULL,
                         tag_split = tag_split,
                         tag_field = tag_field,
                         tag_name = tag_name,
                         displayIndelSize = 0,
                         angle = c(0, 45, 90),
                         replaceMultiAllele = TRUE,
                         ALLELE.color = "grey90")
{
    requireNamespace('tidyr')
    if (!inherits(hapSummary, "hapSummary"))
        if (inherits(hapSummary, 'hapResult'))
            hapSummary <- hap_summary(hapSummary)

    if (!missing(geneName) & title == "")
        title <- geneName

    if ("Accession" %in% colnames(hapSummary)) {
        hapSummary <- hapSummary[, colnames(hapSummary) != 'Accession']
    }


    hps <-
        hapSummary[grepl(paste0(hapPrefix, "[0-9]{1,}"), hapSummary[, 1]), ] %>%
        as.matrix()
    if (nrow(hps) <= 1)
        stop("please check 'hapSummary' and 'hapPrefix'")

    # foot and labs
    ALLELE <- hapSummary[hapSummary$Hap == "ALLELE", ]
    probe_indel <- is.indel.allele(ALLELE)
    probe_mula <- is.multiallelic.allele(ALLELE)

    footi <- ""
    # set indel foots
    if (TRUE %in% probe_indel) {
        # set notes
        displayIndelSize <- displayIndelSize + 1
        allIndel <- ALLELE[probe_indel]
        allIndel <- unlist(stringr::str_split(allIndel, "[,/]"))
        allIndel <-
            allIndel[stringr::str_length(allIndel) > displayIndelSize] %>%
            unique()
        notes <- paste0("i", seq_len(length(allIndel)))
        names(notes) <- allIndel
        if(!is.na(names(notes)[1])){
            m <- paste(names(notes),
                       "->",
                       notes,
                       collapse = "; ",
                       sep = "")
            message("Indel replcements are:\n", m)

            # replace Indels by notes
            hps[hps %in% allIndel] <- notes[hps[hps %in% allIndel]]
            for (i in seq_len(length(ALLELE))) {
                if (probe_indel[i]) {
                    ALi <- ALLELE[i]
                    ALi <- unlist(stringr::str_split(ALi, "[,/]"))
                    p <- ALi %in% names(notes)
                    ALi[p] <- notes[ALi[p]]
                    ALi <- paste(ALi, collapse = ",")
                    ALi <- stringr::str_replace(ALi, ",", "/")
                    ALLELE[i] <- ALi
                }
            }
            # set footi
            footi <- paste(notes,
                           names(notes),
                           sep = ":",
                           collapse = "; ")
        }
    }

    # replace multiallele title
    footT <- ""
    # set multiallele foots
    if (replaceMultiAllele & (TRUE %in% probe_mula)) {
        rept <- ALLELE[probe_mula]
        noteT <- paste0("T", seq_len(length(rept)))
        names(noteT) <- ALLELE[probe_mula]
        ALLELE[probe_mula] <- noteT

        if(! is.na(names(noteT)[1]))
            footT <- paste(noteT,
                           names(noteT),
                           sep = ":",
                           collapse = "; ")

    }

    if (nchar(footi) > 0 & nchar(footT) > 0) {
        foot <- paste(footT, footi, sep = "\n")
    } else {
        if(nchar(footi) == 0 & nchar(footT) == 0){
            foot <- ""
        } else {
            foot <- paste0(footT, footi)
        }
    }
    hps <- rbind(ALLELE, hps)
    # Add extra information
    if (! is.null(INFO_tag)) {
        m <- "length of 'tag_split', 'tag_name' and 'tag_field' should be equal with 'INFO_tag'"
        if(! missing(tag_split))
            if(length(INFO_tag) != length(tag_split))
                stop(m)
        if(missing(tag_name)){
            tag_name <- INFO_tag
        } else {
            if(length(INFO_tag) != length(tag_name))
                stop(m)
        }
        # if(! missing(tag_field))
        #     if(length(INFO_tag) != length(tag_field))
        #         stop(m)


        # get INFOs in hap results
        INFO <- t(hapSummary[hapSummary$Hap == "INFO", ])
        INFO <- strsplit(INFO, ";")

        ntag <- 0
        for (i in INFO_tag) {
            # filter INFO tags with tag_name
            ntag <- ntag + 1
            INFOi <-
                unlist(sapply(INFO, function(x)
                    x[startsWith(x, paste0(i, "="))]))
            # split by "=", removed tag_name
            INFOi <- stringr::str_remove(INFOi, paste0(i, "="))
            if (missing(tag_field)){
                Ii <- INFOi
            } else {
                Ii <- list()
                if (i %in% c("ANN", "SNPEFF")) {
                    if (missing(geneName))
                        stop("'geneName' is missing !!!")
                    INFOi <-
                        sapply(INFOi, function(x)
                            strsplit(x, ","))
                    INFOi <-
                        lapply(INFOi, function(x)
                            strsplit(x, "[|]"))
                    for (j in seq_len(length(INFOi))) {
                        if (is.na(INFOi[[j]])[1])
                            next

                        infi <- ""
                            for (k in seq_len(length(INFOi[[j]]))) {
                                if (INFOi[[j]][[k]][4] == geneName)
                                    infi <-
                                        paste(infi, INFOi[[j]][[k]][tag_field[ntag]], sep = ",")

                            }
                            Ii[[j]] <- stringr::str_remove(infi, ",")
                        }
                    } else {
                        if (missing(tag_split))
                            stop("'tag_split' is missing!!!")
                        Ii <- lapply(INFOi,
                                     function(x) {
                                         unlist(strsplit(x, tag_split))[tag_field[ntag]]
                                     }) %>% unlist()
                    }
                }

            Ii <- unlist(Ii)
            # Ii[Ii == ""] <- NA
            hps <- rbind(c(tag_name[ntag], Ii, ""), hps)

        }
    }


    # set labs for plot
    meltHapRes <- reshape2::melt(hps, 1)
    colnames(meltHapRes) <- c('Var1', 'Var2', "value")
    lab <- meltHapRes

    # set all non-genotype filed as NA
    if (!missing(INFO_tag))
        meltHapRes$value[meltHapRes$Var1 %in% tag_name] <- NA
    meltHapRes$value[meltHapRes$Var1 == "ALLELE"] <- NA
    meltHapRes$value[meltHapRes$Var2 == "freq"] <- NA

    # set levels for row sequences
    levels <- as.vector(unique(meltHapRes$Var1))
    haplevel <- levels[startsWith(levels, hapPrefix)]
    otherlevel <- levels[!startsWith(levels, hapPrefix)]
    levels <-
        c(haplevel[order(haplevel, decreasing = TRUE)], otherlevel)
    meltHapRes$Var1 <- factor(meltHapRes$Var1, levels = levels)

    if (length(angle) > 1)
        warning("using first 'angle': ", angle[1])
    angle <- angle[1]
    if (angle == 0) {
        vjust <- 0.5
        hjust  <- 0.5
    } else
        if (angle == 45) {
            vjust <- 0.1
            hjust  <- 0.1
        } else
            if (angle == 90) {
                vjust <- 1
                hjust  <- 1
            } else
                stop("angle should be one of 0, 45 and 90")
    fig0 <- ggplot2::ggplot(data = meltHapRes,
                            mapping = ggplot2::aes_(
                                x =  ~ Var2,
                                y =  ~ Var1,
                                fill =  ~ value
                            )) +
        ggplot2::geom_tile(color = "white") +
        ggplot2::geom_text(ggplot2::aes_(
            x =  ~ Var2,
            y =  ~ Var1,
            label = lab$value
        )) +
        ggplot2::scale_fill_discrete(na.value = ALLELE.color) +
        ggplot2::labs(caption = foot) +
        ggplot2::ggtitle(label = title) +  ggplot2::scale_y_discrete() +
        ggplot2::scale_x_discrete(guide = ggplot2::guide_axis(position = "top")) +

        ggplot2::theme(
            legend.position = "none",
            axis.title.x =  ggplot2::element_blank(),
            axis.title.y =  ggplot2::element_blank(),
            panel.grid.major =  ggplot2::element_blank(),
            panel.border =  ggplot2::element_blank(),
            panel.background =  ggplot2::element_blank(),
            axis.ticks =  ggplot2::element_blank(),
            axis.text.x = ggplot2::element_text(
                angle = angle,
                vjust = vjust,
                hjust = hjust
            ),
            plot.subtitle = ggplot2::element_text(hjust = 0.5),
            plot.title = ggplot2::element_text(hjust = 0.5)
        ) +
        ggplot2::guides(fill = ggplot2::guide_colorbar(title.position = "top",
                                                       title.hjust = 0.5))
    return(fig0)
}



#' @name displayVarOnGeneModel
#' @title Display Variants on Gene Model
#' @description
#' show variants on gene model using hapSummary and gene annotations
#' @examples
#' \donttest{
#' data("geneHapR_test")
#' hapSummary <- hap_summary(hapResult)
#' displayVarOnGeneModel(hapSummary, gff,
#'                       startPOS = 4100,
#'                       endPOS = 8210,
#'                       cex = 0.75)
#' }
#' @importFrom GenomicRanges GRanges
#' @importFrom GenomicRanges strand
#' @importFrom lolliplot lolliplot
#' @importFrom IRanges IRanges `%over%`
#' @param gff gff
#' @param hapSummary,hap haplotype result
#' @param Chr the chromosome name.
#' If missing, the first element in the hapSummary will be used
#' @param startPOS If missing, will use the min position in hapSummary
#' @param endPOS If missing, will use the max position in hapSummary
#' @param cex a numeric control the size of circle
#' @param type character. Could be "circle", "pie", "pin",
#'  "pie.stack" or "flag"
#' @param CDS_h,fiveUTR_h,threeUTR_h The height of CDS 5'UTR and 3'UTR
#' in gene model
#' @param geneElement ploted elements, eg.: c("CDS","five_prime_UTR")
#' @return No return value
#' @export

displayVarOnGeneModel <- function(hapSummary,
                                  gff,
                                  Chr,
                                  startPOS,
                                  endPOS,
                                  type = "pin",
                                  cex = 0.7,
                                  CDS_h = 0.05,
                                  fiveUTR_h = 0.02,
                                  threeUTR_h = 0.01,
                                  geneElement = geneElement, hap)
{
    # lolliplot
    requireNamespace("tidyr")
    if (missing(gff))
        stop("gff is missing!")

    if (missing('hapSummary')){
        if(missing("hap"))
            stop("hapSummary is missing!") else
                hapSummary <- hap
    }
    if(missing(hapSummary)){
        if(missing(hap)) stop("Please provide your haplotype result!") else
            hapSummary <- hap
    }
    if (!inherits(hapSummary, "hapSummary")){
        if (inherits(hapSummary, 'hapResult')){
            hapSummary <- hap_summary(hapSummary)
        } else {
            stop("please check your inputs")
        }
    }

    s <- sites(hapSummary)
    meta <- hapSummary[seq_len(4), -1]

    if ("Accession" %in% colnames(meta)) {
        meta <- meta[, colnames(meta) != "Accession"]
    }
    if ("freq" %in% colnames(meta)) {
        meta <- meta[, colnames(meta) != "freq"] %>% data.frame()
    }

    POS <- meta[2, ] %>% as.numeric()
    if (missing(Chr))
        Chr <- meta[1, 1]
    if (missing(startPOS))
        startPOS <- min(POS)
    if (missing(endPOS))
        endPOS <- max(POS)
    SNP <- meta[4, ]

    #meta <- hapSummary[1:4,]
    #meta <- meta[,-ncol(meta)]
    #meta[meta == ""] = NA
    #meta <- meta[,!is.na(meta[1,])]
    #POS <- as.numeric(meta[2,])

    if(missing(geneElement)){
        allType <- unique(gff$type) %>% as.vector()
        p1 <- stringr::str_detect(tolower(allType), pattern = "promoter")
        p2 <- stringr::str_detect(tolower(allType), pattern = "utr")
        p3 <- stringr::str_detect(tolower(allType), pattern = "cds")
        p <- p1 | p2 | p3
        geneElement <- allType[p]
    }
    SNP.gr <- GenomicRanges::GRanges(
        Chr,
        IRanges::IRanges(POS, width = 1,
                         names = paste0(POS, "(", SNP, ")")),
        color = sample.int(6, length(SNP), replace = TRUE),
        score = sample.int(5, length(SNP), replace = TRUE),
        angle = 45
    )


    # set plot ranges
    gene <- GenomicRanges::GRanges(Chr,
                                   IRanges::IRanges(
                                       start = min(startPOS, endPOS),
                                       end = max(startPOS, endPOS)
                                   ))

    over <- gff[gff %over% gene]
    over$height[over$type == "CDS"] <- CDS_h
    over$height[over$type != "CDS"] <- CDS_h * 0.5
    over$height[over$type == "three_prime_UTR"] <- threeUTR_h
    over$height[over$type == "five_prime_UTR"] <- fiveUTR_h

    features <- over[over$type %in% geneElement]
    strands <- as.character(GenomicRanges::strand(features))
    layerID <- unlist(features$Parent)
    layerID <- paste0(layerID, "(",
                      ifelse(strands == "+", "5'->3'", "3'<-5'"), ")")
    features$featureLayerID <- layerID
    names(features) <- features$featureLayerID
    l <- length(unique(names(features)))
    if (l < 6) {
        fillc <- c(seq_len(l) + 1)
    } else{
        fillc <- rainbow(l)
    }
    names(fillc) <- unique(names(features))
    features$fill <- fillc[names(features)]


    # set ranges of features
    lolliplot::lolliplot(
        SNP.gr,
        features,
        gene,
        type = type,
        jitter = NULL,
        cex = cex,
        ylab = "",
        yaxis = FALSE
    )
}
