% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/annotation_ticks.R
\name{annotation_ticks}
\alias{annotation_ticks}
\title{Add ticks as ggplot annotation}
\source{
The code is a slightly modified version of the answer to this
\href{https://stackoverflow.com/questions/58485334}{Stack Overflow} question,
which is itself a refactored version of this
\href{https://github.com/hrbrmstr/ggalt/blob/master/R/annotation_ticks.r}{\code{annotation_ticks()}}
function.
}
\usage{
annotation_ticks(
  sides = "b",
  type = "both",
  outside = FALSE,
  tick.length = unit(4.8, "pt"),
  minor.length = unit(2.4, "pt"),
  linewidth = 0.6,
  colour = "black",
  color = NULL,
  linetype = 1,
  lineend = "butt",
  alpha = 1,
  data = data.frame(x = NA)
)
}
\arguments{
\item{sides}{\code{string}. Indicates which sides of the plot should ticks
appear. Can be any of \code{"trbl"}, for top, right, bottom, left.}

\item{type}{\code{string}. Types of ticks that appear. One of
\code{"major"}, \code{"minor"}, or \code{"both"}. Control number of ticks
by controlling the \code{breaks} and \code{minor_breaks} arguments in the
various ggplot2 \verb{scale_(x|y)_} functions.}

\item{outside}{\code{logical}. Should the ticks point outside of the plotting
area? If \code{TRUE} clipping must be turned off.}

\item{tick.length}{a \code{\link[grid]{unit}} object specifying the length
of major ticks.}

\item{minor.length}{a \code{\link[grid]{unit}} object specifying the length
of minor ticks.}

\item{linewidth}{\code{numeric}. Linewidth of ticks.}

\item{colour, color}{\code{string}. Colour of ticks.}

\item{linetype}{\code{string} or \code{numeric}. Linetype of tick marks.}

\item{lineend}{\code{string}. Lineend of ticks. One of \code{"square"}
(default), \code{"butt"}, or \code{"round"}.}

\item{alpha}{\code{numeric}. Transparency of ticks.}

\item{data}{\code{data.frame}. Use this argument to control the appearance of
ticks on different facets. Pass a data.frame containing the levels from the
faceting variable you want to annotate specifically.
See \href{https://stackoverflow.com/questions/20128582}{here} for an example.}
}
\value{
Returns a \emph{layer} ggproto object with \code{geom = GeomTicks}.
}
\description{
This is an annotation function to add tick marks (major, minor, or both) to
a ggplot. Clipping must be turned off if the ticks are to appear outside the
plotting area, for example with: \code{coord_cartesian(clip = "off")}.
}
\examples{
## Generally it is better to use the guide_prism_minor function.
## However annotation_ticks is useful in a few specific situations.
library(ggplot2)

## easily put ticks without labels around a plot with a border
ggplot(mtcars, aes(x = mpg, y = disp)) +
  geom_point() +
  theme_prism(border = TRUE) +
  coord_cartesian(clip = "off") +
  annotation_ticks(sides = "tr", type = "major", outside = TRUE) +
  theme(plot.margin = unit(c(4, 4, 4, 4), "mm"))

# the same but with minor ticks as well
ggplot(mtcars, aes(x = mpg, y = disp)) +
  geom_point() +
  scale_x_continuous(guide = "prism_minor") +
  scale_y_continuous(guide = "prism_minor") +
  theme_prism(border = TRUE) +
  coord_cartesian(clip = "off") +
  annotation_ticks(sides = "tr", type = "both", outside = TRUE) +
  theme(plot.margin = unit(c(4, 4, 4, 4), "mm"))

# you can adjust the appearance of annotation_ticks
ggplot(mtcars, aes(x = mpg, y = disp)) +
  geom_point() +
  theme_prism(border = TRUE) +
  coord_cartesian(clip = "off") +
  annotation_ticks(
    sides = "tr",
    type = "major",
    outside = TRUE,
    tick.length = unit(10, "pt"),
    colour = "red",
    linewidth = 2,
    linetype = "dashed",
    lineend = "round"
  ) +
  theme(plot.margin = unit(c(4, 4, 4, 4), "mm"))

## Unfortunately, due to the way they work, secondary axes don't always play
## well with the minor tick axes guides in this package.
## So we can use annotation_ticks instead.
sample.data <- data.frame(
  day = as.Date("2019-01-01") + 0:99,
  temperature = runif(100) + seq(1, 100)^2.5 / 10000,
  price = runif(100) + seq(100, 1)^1.5 / 10
)

# sample graph with secondary axis
ggplot(sample.data, aes(x = day)) +
  geom_line(aes(y = temperature), colour = "magenta") +
  geom_line(aes(y = price / 10), colour = "blue") +
  scale_y_continuous(sec.axis = sec_axis(~. * 10, name = "price")) +
  theme_prism(border = TRUE) +
  coord_cartesian(clip = "off")

# guide_prism_minor only works with the main axis in this case
ggplot(sample.data, aes(x = day)) +
  geom_line(aes(y = temperature), colour = "magenta") +
  geom_line(aes(y = price / 10), colour = "blue") +
  scale_y_continuous(
    sec.axis = sec_axis(~. * 10, name = "price"),
    guide = "prism_minor"
  ) +
  theme_prism(border = TRUE) +
  coord_cartesian(clip = "off")

# we use annotation_ticks to draw the minor ticks on the secondary axis
ggplot(sample.data, aes(x = day)) +
  geom_line(aes(y = temperature), colour = "magenta") +
  geom_line(aes(y = price / 10), colour = "blue") +
  scale_y_continuous(
    sec.axis = sec_axis(~. * 10, name = "price"),
    guide = "prism_minor"
  ) +
  theme_prism(border = TRUE) +
  coord_cartesian(clip = "off") +
  annotation_ticks(sides = "r", type = "minor", outside = TRUE)
}
