% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prune.R
\name{prune_meta}
\alias{prune_meta}
\alias{prune_meta.git_repository}
\title{Prune Metadata Files}
\usage{
prune_meta(root = ".", path = NULL, recursive = TRUE, ...)

\method{prune_meta}{git_repository}(root, path = NULL, recursive = TRUE, ..., stage = FALSE)
}
\arguments{
\item{root}{The root of a project. Can be a file path or a \code{git-repository}.
Defaults to the current working directory (\code{"."}).}

\item{path}{the directory in which to clean all the data files. The directory
is relative to \code{root}.}

\item{recursive}{remove files in subdirectories too.}

\item{...}{parameters used in some methods}

\item{stage}{stage the changes after removing the files. Defaults to \code{FALSE}.}
}
\value{
returns invisibly a vector of removed files names. The paths are
relative to \code{root}.
}
\description{
Removes all \strong{valid} metadata (\code{.yml} files) from the \code{path} when they don't
have accompanying data (\code{.tsv} file). \strong{Invalid} metadata triggers a warning
without removing the metadata file.

Use this function with caution since it will remove all valid metadata files
without asking for confirmation. We strongly recommend to use this
function on files under version control. See
\code{vignette("workflow", package = "git2rdata")} for some examples on how to use
this.
}
\examples{
## on file system

# create a directory
root <- tempfile("git2rdata-")
dir.create(root)

# store a dataframe as git2rdata object. Capture the result to minimise
# screen output
junk <- write_vc(iris[1:6, ], "iris", root, sorting = "Sepal.Length")
# write a standard tab separate file (non git2rdata object)
write.table(iris, file = file.path(root, "standard.tsv"), sep = "\t")
# write a YAML file
yml <- list(
  authors = list(
   "Research Institute for Nature and Forest" = list(
       href = "https://www.inbo.be/en")))
yaml::write_yaml(yml, file = file.path(root, "_pkgdown.yml"))

# list the git2rdata objects
list_data(root)
# list the files
list.files(root, recursive = TRUE)

# remove all .tsv files from valid git2rdata objects
rm_data(root, path = ".")
# check the removal of the .tsv file
list.files(root, recursive = TRUE)
list_data(root)

# remove dangling git2rdata metadata files
prune_meta(root, path = ".")
# check the removal of the metadata
list.files(root, recursive = TRUE)
list_data(root)


## on git repo

# initialise a git repo using git2r
repo_path <- tempfile("git2rdata-repo-")
dir.create(repo_path)
repo <- git2r::init(repo_path)
git2r::config(repo, user.name = "Alice", user.email = "alice@example.org")

# store a dataframe
write_vc(iris[1:6, ], "iris", repo, sorting = "Sepal.Length", stage = TRUE)
# check that the dataframe is stored
status(repo)
list_data(repo)

# commit the current version and check the git repo
commit(repo, "add iris data", session = TRUE)
status(repo)

# remove the data files from the repo
rm_data(repo, path = ".")
# check the removal
list_data(repo)
status(repo)

# remove dangling metadata
prune_meta(repo, path = ".")
# check the removal
list_data(repo)
status(repo)

# clean up
junk <- file.remove(
  list.files(root, full.names = TRUE, recursive = TRUE), root)
junk <- file.remove(
  rev(list.files(repo_path, full.names = TRUE, recursive = TRUE,
                 include.dirs = TRUE, all.files = TRUE)),
  repo_path)
}
\seealso{
Other storage: 
\code{\link{list_data}()},
\code{\link{read_vc}()},
\code{\link{relabel}()},
\code{\link{rename_variable}()},
\code{\link{rm_data}()},
\code{\link{write_vc}()}
}
\concept{storage}
