% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glca.R
\name{glca}
\alias{glca}
\title{Fits Latent Class Models for Data Containing Group Variable and Covariates}
\usage{
glca(formula, group = NULL, data = NULL, nclass = 3, ncluster = 0,
  measure.inv = TRUE, std.err = TRUE, init.param = NULL,
  n.init = 1, testiter = 50, maxiter = 1000, eps = 1e-06,
  na.rm = FALSE, random.seed = NULL, verbose = TRUE)
}
\arguments{
\item{formula}{a formula for specifying manifest items and covariates using the "\code{item}" function}

\item{group}{an optional vector specifying a group of observations. Given group variable, group covariates can be incorporated.}

\item{data}{a data frame containing the manifest item, covariates and group variable}

\item{nclass}{number of level-1 (individual-level) latent classes}

\item{ncluster}{number of level-2 (group-level) latent classes. When \code{group} and \code{ncluster} (>1) are given the multilevel latent class models will be fitted.}

\item{measure.inv}{a logical value of the measurement invariance assumption across groups}

\item{std.err}{a logical value for whether calculating standard errors for estimates}

\item{init.param}{a list which contains user-defined initial parameter}

\item{n.init}{number of random initial parameter sets}

\item{testiter}{an integer for maximum number of test set iteration}

\item{maxiter}{an integer for maximum number of iteration}

\item{eps}{positive convergence tolerance}

\item{na.rm}{a logical value for whether or not to remove observations who has at least 1 item missing}

\item{random.seed}{random seed to have the equivalent solution for every trials}

\item{verbose}{a logical value for whether or not to print the result of a function's execution}
}
\value{
\code{glca} returns an object of class "\code{glca}".

The function \code{summary} prints estimates for parameters and \code{glca.gof} function gives goodness of fit measures for the model.

An object of class "\code{glca}" is a list containing the following components:

\item{call}{the matched call}
\item{model}{a list of model description}
\item{datalist}{a list of data used for fitting}
\item{param}{a list of parameter estimates}
\item{std.err}{a list of standard errors for estimates}
\item{coefficient}{a list of logistic regression coefficients for prevalence of level-1 class}
\item{posterior}{a data frame of posterior probablities of each individaul for latent classes and each group for latent clusters}
\item{gof}{a list of goodness of fit measures}
\item{convergence}{a list containing information about convergence}
}
\description{
Function for fitting latent class models with multiple groups, which may or may not include latent class structure for group variable.
}
\details{
The \code{glca} is the function for implementing LCA consist of two-type latent categorical variables (i.e., level-1 and level-2 latent class). The level-1 (individual-level) latent class is identified by the association among the individuals' responses to multiple manifest items, but level-2 (group-level) latent class is categorized by the prevalence of level-1 latent class for group variable. The function \code{glca} can handle two types of covariates: level-1 and level-2 covariates. If covariates vary across individuals, they are considered as level-1 covariates. When \code{group} and \code{ncluster} (>1) are given, covariates which are varying across groups are considered as level-2 covariates. Both types of covariates have effect on level-1 class prevalence.

The formula should consist of an \code{~} operator between two sides. Manifest items should be indicated in LHS of formula using \code{item} function and covariates should be specified in RHS of formula. For example, \cr
\code{item(y1, y2, y3) ~ 1}\cr
\code{item(y1, y2, y3) ~ x1 + x2}
where the first fomula indicates LCA with three manifest variables (\code{y1}, \code{y2}, and \code{y3}) and no covariate, and the second formula includes two covariates (\code{x1} and \code{x2}). Two types of covariates (i.e., level-1 and level-2 covariates) will be automatically detected by \code{glca}.

The estimated parameters in \code{glca} are \code{rho}, \code{gamma}, \code{delta}, and \code{beta}. The set of item response probabilities for each level-1 class is \code{rho}. The sets of prevalences for level-1 and level-2 class are \code{gamma} and \code{delta}, respectively. The prevalence for level-1 class (i.e., \code{gamma}) can be modeled as logistic regression using level-1 and/or level-2 covariates. The set of logistic regression coefficients is \code{beta} in \code{glca} output.
}
\examples{
##
## Example 1. GSS dataset
##
data("gss")
# LCA
lca = glca(item(ABDEFECT, ABHLTH, ABRAPE, ABPOOR, ABSINGLE) ~ 1,
            data = gss, nclass = 3)
summary(lca)

# LCA with covariate(s)
lcr = glca(item(ABDEFECT, ABHLTH, ABRAPE, ABPOOR, ABSINGLE) ~ AGE,
           data = gss, nclass = 3)
summary(lcr)
coef(lcr)

# Multiple-group LCA (MGLCA)
mglca = glca(item(ABDEFECT, ABHLTH, ABRAPE, ABPOOR, ABSINGLE) ~ 1,
             group = DEGREE, data = gss, nclass = 3)
summary(mglca)


# Multiple-group LCA with covariate(s) (MGLCR)
mglcr = glca(item(ABDEFECT, ABHLTH, ABRAPE, ABPOOR, ABSINGLE) ~ SEX,
             group = DEGREE, data = gss, nclass = 3)
summary(mglcr)
coef(mglcr)

\donttest{
##
## Example 2. BRFSS data
##
data("brfss")
# Multilevel LCA (MLCA)
brfss1000 = brfss[sample(1:nrow(brfss), 1000),]
mlca = glca(item(OBESE, PA300, FRTLT1A, VEGLT1A, SMOKER, DRNK30) ~ 1,
            group = STATE, data = brfss1000, nclass = 3, ncluster = 2)
summary(mlca)

# MLCA with covariate(s) (MLCR)
# (SEX: level-1 covariate, REGION: level-2 covariate)
mlcr = glca(item(OBESE, PA300, FRTLT1A, VEGLT1A, SMOKER, DRNK30) ~ SEX + PARTY,
            group = STATE, data = brfss1000, nclass = 3, ncluster = 2)
summary(mlcr)
coef(mlcr)
}

}
\references{
Vermunt, J.K. (2003) Multilevel latent class models. \emph{Sociological Methodology}, \bold{33}, 213--239. \doi{10.1111/j.0081-1750.2003.t01-1-00131.x}

Collins, L.M. and Lanza, S.T. (2009) \emph{Latent Class and Latent Transition Analysis: With Applications in the Social, Behavioral, and Health Sciences}. John Wiley & Sons Inc.
}
\seealso{
\code{\link{gss}} \code{\link{brfss}}
}
\author{
Youngsun Kim
}
