#' Authenticate with Google Analytics OAuth2
#'
#' A wrapper for \link[googleAuthR]{gar_auth} and \link[googleAuthR]{gar_auth_service}
#'
#' @param token An existing token or file location of a token to authenticate with
#' @param new_user If TRUE, reauthenticate via Google login screen
#' @param no_auto Skip auto authentication
#' 
#' @details
#' 
#' Run this function first time to authenticate with Google in your browser.  
#' 
#' After initial authentication, a \code{.httr-oauth} will be saved to your working directory, where your authentication details are kept.  Keep this file safe.
#' 
#' If you want to reauthenticate, delete this file from your directory or run \code{ga_auth(new_user = TRUE)}
#' 
#' @section Multiple accounts:
#' 
#' You can authenticate with a new auth file for each account. Supply argument \code{token} with the name of the cache file you want to use e.g. \code{ga_auth(token = "one.httr-oauth")} for one account, 
#'   \code{ga_auth(token = "another.httr-oauth")} for a different account.
#'
#' 
#' @section Auto-authentication:
#' 
#' You can choose to auto-authenticate by moving your \code{.httr-oauth} or by 
#'   creating a Google OAuth service account JSON file.
#' 
#' Specify an environment variable in R via a \code{.Renviron} file or using \link{Sys.setenv} which point to the file location of your chosen authentication file.  See \link{Startup}
#' 
#' Once you have set the environment variable \code{GA_AUTH_FILE} to a valid file location,
#'   the function will look there for authentication details upon loading the library meaning 
#'   you will not need to call \code{ga_auth()} yourself as you would normally.
#' 
#' An example \code{.Renviron} file is below:
#' 
#' \code{GA_AUTH_FILE = "/Users/bob/auth/googleAnalyticsR.httr-oauth"}
#'
#' \code{GA_AUTH_FILE} can be either a auth file for a token generated by \link[googleAuthR]{gar_auth} or
#'   service account JSON ending with file extension \code{.json}
#'
#' @section Your own Google Project:
#' 
#' Be default the Google Project used is shared by all users, so you may find it runs out of API calls.  
#' To mitigate that, create your own Google Project and turn on the Analytics APIs.
#' 
#' You can then create your own client ID and client secret, to place in options or environment arguments (whichever is easiest)
#' 
#' The environment args are below.  Similar to auto-authentication, you can place your entries in an \code{.Renviron} file
#' 
#' \code{
#' GA_CLIENT_ID="XXXX"
#' GA_CLIENT_SECRET="XXX"
#' GA_WEB_CLIENT_ID="XXX"
#' GA_WEB_CLIENT_SECRET="XXX"
#' }
#' 
#' @section Service accounts:
#'   
#' If you use the service account JSON, you will need to add the service account email 
#'   to your Google Analytics users to see data e.g. \code{xxxx@yyyyyy.iam.gserviceaccount.com}
#'
#' @return Invisibly, the token that has been saved to the session
#' @import googleAuthR
#' @importFrom tools file_ext
#' @export
ga_auth <- function(token = NULL, new_user = FALSE, no_auto = FALSE){
  
  if(!is.null(token)){
    return(gar_auth(token = token))
  }
  
  needed <- c("https://www.googleapis.com/auth/analytics", 
              "https://www.googleapis.com/auth/analytics.readonly",
              "https://www.googleapis.com/auth/analytics.manage.users.readonly",
              "https://www.googleapis.com/auth/analytics.edit",
              "https://www.googleapis.com/auth/analytics.manage.users",
              "https://www.googleapis.com/auth/analytics.provision"	)
  
 gar_auto_auth(needed,
               new_user = new_user,
               no_auto = no_auto,
               environment_var = "GA_AUTH_FILE")
 
}