#' Plot quantile-quantile (QQ) graphs from residuals of linear models.
#'
#' This function takes a linear model (simple or mixed effects) and plots a QQ graph after running `rstudent` from \code{\link[stats]{rstudent}} to generate a table of studentized model residuals on an ordinary (\code{\link{simple_model}}),  mixed model (\code{\link{mixed_model}} or \code{\link{mixed_model_slopes}}. The graph plots studentized residuals from the model (sample) on Y axis & Theoretical quantiles on X axis.
#'
#' The function uses \code{\link{ggplot2}[geom_qq]} and \code{\link{ggplot2}[geom_qq_line]} geometries. Symbols receive "ok_orange" colour by default.
#'
#' @param Model name of a saved model generated by \code{simple_model} or \code{mixed_model}.
#' @param symsize size of symbols, default set to 3.
#' @param symthick thickness of symbol border, default set to 1.
#' @param s_alpha fractional opacity of symbols, default set to 1 (i.e. maximum opacity & zero transparency).
#' @param fontsize parameter of \code{base_size} of fonts in \code{theme_classic}, default set to size 20.
#'
#' @return This function returns a \code{ggplot2} object of class "gg" and "ggplot".
#' @export plot_qqmodel
#' @import ggplot2
#' @importFrom stats rstudent
#'
#' @examples
#' #Basic usage
#' m1 <- simple_model(data = data_2w_Festing, 
#' Y_value = "GST", 
#' Fixed_Factor = c("Treatment", "Strain"))
#' plot_qqmodel(m1)
#'

plot_qqmodel <- function(Model, symsize = 2.5, symthick = 1, s_alpha = 1, fontsize = 20){
  mod <- rstu <- NULL
  mod <- data.frame(rstu = stats::rstudent(Model))
  P <- ggplot2::ggplot(mod,
                       aes(sample = rstu))+
  geom_qq_line(na.rm = TRUE,
               size = 1)+
  geom_qq(na.rm = TRUE,
          shape = 21, fill = "#E69F00",
          size = {{ symsize }},
          stroke = {{ symthick }},
          alpha = {{ s_alpha }})+
  labs(y = "Studentized residual",
       x = "Theoretical quantile")+
  theme_classic(base_size = {{ fontsize }})+
  theme(strip.background = element_blank())
  suppressWarnings(P)
}
