/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5O_FRIEND     
#include "H5Tmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Iprivate.h" 
#include "H5Opkg.h"     
#include "H5Tpkg.h"     

static htri_t     H5O__dtype_isa(const H5O_t *loc);
static void      *H5O__dtype_open(const H5G_loc_t *obj_loc, H5I_type_t *opened_type);
static void      *H5O__dtype_create(H5F_t *f, void *_crt_info, H5G_loc_t *obj_loc);
static H5O_loc_t *H5O__dtype_get_oloc(hid_t obj_id);

const H5O_obj_class_t H5O_OBJ_DATATYPE[1] = {{
    H5O_TYPE_NAMED_DATATYPE, 
    "named datatype",        
    NULL,                    
    NULL,                    
    H5O__dtype_isa,          
    H5O__dtype_open,         
    H5O__dtype_create,       
    H5O__dtype_get_oloc,     
    NULL,                    
    NULL                     
}};

static htri_t
H5O__dtype_isa(const H5O_t *oh)
{
    htri_t ret_value = FAIL; 

    FUNC_ENTER_PACKAGE

    assert(oh);

    if ((ret_value = H5O_msg_exists_oh(oh, H5O_DTYPE_ID)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to read object header");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__dtype_open(const H5G_loc_t *obj_loc, H5I_type_t *opened_type)
{
    H5T_t *type      = NULL; 
    void  *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(obj_loc);

    *opened_type = H5I_DATATYPE;

    
    if (NULL == (type = H5T_open(obj_loc)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, NULL, "unable to open datatype");

    ret_value = (void *)type;

done:
    if (NULL == ret_value)
        if (type && H5T_close(type) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CLOSEERROR, NULL, "unable to release datatype");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__dtype_create(H5F_t *f, void *_crt_info, H5G_loc_t *obj_loc)
{
    H5T_obj_create_t *crt_info  = (H5T_obj_create_t *)_crt_info; 
    void             *ret_value = NULL;                          

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(crt_info);
    assert(obj_loc);

    
    if (H5T__commit(f, crt_info->dt, crt_info->tcpl_id) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to commit datatype");

    
    if (NULL == (obj_loc->oloc = H5T_oloc(crt_info->dt)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "unable to get object location of named datatype");
    if (NULL == (obj_loc->path = H5T_nameof(crt_info->dt)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "unable to get path of named datatype");

    
    ret_value = crt_info->dt;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5O_loc_t *
H5O__dtype_get_oloc(hid_t obj_id)
{
    H5T_t     *type      = NULL; 
    H5T_t     *dt        = NULL;
    H5O_loc_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (dt = (H5T_t *)H5I_object(obj_id)))
        HGOTO_ERROR(H5E_OHDR, H5E_BADID, NULL, "couldn't get object from ID");

    
    type = (H5T_t *)H5T_get_actual_type(dt);

    
    if (NULL == (ret_value = H5T_oloc(type)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, NULL, "unable to get object location from object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
