/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 

#include "H5private.h"   
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5MFprivate.h" 
#include "H5VMprivate.h" 

static herr_t H5D__none_idx_create(const H5D_chk_idx_info_t *idx_info);
static herr_t H5D__none_idx_open(const H5D_chk_idx_info_t *idx_info);
static herr_t H5D__none_idx_close(const H5D_chk_idx_info_t *idx_info);
static herr_t H5D__none_idx_is_open(const H5D_chk_idx_info_t *idx_info, bool *is_open);
static bool   H5D__none_idx_is_space_alloc(const H5O_storage_chunk_t *storage);
static herr_t H5D__none_idx_get_addr(const H5D_chk_idx_info_t *idx_info, H5D_chunk_ud_t *udata);
static herr_t H5D__none_idx_load_metadata(const H5D_chk_idx_info_t *idx_info);
static int    H5D__none_idx_iterate(const H5D_chk_idx_info_t *idx_info, H5D_chunk_cb_func_t chunk_cb,
                                    void *chunk_udata);
static herr_t H5D__none_idx_remove(const H5D_chk_idx_info_t *idx_info, H5D_chunk_common_ud_t *udata);
static herr_t H5D__none_idx_delete(const H5D_chk_idx_info_t *idx_info);
static herr_t H5D__none_idx_copy_setup(const H5D_chk_idx_info_t *idx_info_src,
                                       const H5D_chk_idx_info_t *idx_info_dst);
static herr_t H5D__none_idx_size(const H5D_chk_idx_info_t *idx_info, hsize_t *size);
static herr_t H5D__none_idx_reset(H5O_storage_chunk_t *storage, bool reset_addr);
static herr_t H5D__none_idx_dump(const H5O_storage_chunk_t *storage, FILE *stream);

const H5D_chunk_ops_t H5D_COPS_NONE[1] = {{
    false,                        
    NULL,                         
    H5D__none_idx_create,         
    H5D__none_idx_open,           
    H5D__none_idx_close,          
    H5D__none_idx_is_open,        
    H5D__none_idx_is_space_alloc, 
    NULL,                         
    H5D__none_idx_get_addr,       
    H5D__none_idx_load_metadata,  
    NULL,                         
    H5D__none_idx_iterate,        
    H5D__none_idx_remove,         
    H5D__none_idx_delete,         
    H5D__none_idx_copy_setup,     
    NULL,                         
    H5D__none_idx_size,           
    H5D__none_idx_reset,          
    H5D__none_idx_dump,           
    NULL                          
}};

static herr_t
H5D__none_idx_create(const H5D_chk_idx_info_t *idx_info)
{
    hsize_t nbytes;              
    haddr_t addr;                
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(idx_info);
    assert(idx_info->f);
    assert(idx_info->pline);
    assert(idx_info->pline->nused == 0); 
    assert(idx_info->layout);
    assert(idx_info->layout->u.chunk.max_nchunks);
    assert(!H5_addr_defined(
        idx_info->layout->storage.u.chunk.idx_addr)); 

    
    nbytes = idx_info->layout->u.chunk.max_nchunks * idx_info->layout->u.chunk.size;

    
    addr = H5MF_alloc(idx_info->f, H5FD_MEM_DRAW, nbytes);
    if (!H5_addr_defined(addr))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "file allocation failed");

    
    idx_info->layout->storage.u.chunk.idx_addr = addr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__none_idx_open(const H5D_chk_idx_info_t H5_ATTR_UNUSED *idx_info)
{
    FUNC_ENTER_PACKAGE_NOERR

    

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__none_idx_close(const H5D_chk_idx_info_t H5_ATTR_UNUSED *idx_info)
{
    FUNC_ENTER_PACKAGE_NOERR

    

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__none_idx_is_open(const H5D_chk_idx_info_t H5_ATTR_NDEBUG_UNUSED *idx_info, bool *is_open)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(idx_info);
    assert(idx_info->layout);
    assert(H5D_CHUNK_IDX_NONE == idx_info->layout->storage.u.chunk.idx_type);
    assert(is_open);

    *is_open = true;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static bool
H5D__none_idx_is_space_alloc(const H5O_storage_chunk_t *storage)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(storage);

    FUNC_LEAVE_NOAPI((bool)H5_addr_defined(storage->idx_addr))
} 

static herr_t
H5D__none_idx_get_addr(const H5D_chk_idx_info_t *idx_info, H5D_chunk_ud_t *udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(idx_info);
    assert(idx_info->f);
    assert(idx_info->pline);
    assert(idx_info->pline->nused == 0);
    assert(idx_info->layout);
    assert(udata);
    assert(H5_addr_defined(idx_info->layout->storage.u.chunk.idx_addr));

    
    udata->chunk_idx = H5VM_array_offset_pre((idx_info->layout->u.chunk.ndims - 1),
                                             idx_info->layout->u.chunk.max_down_chunks, udata->common.scaled);

    
    udata->chunk_block.offset =
        idx_info->layout->storage.u.chunk.idx_addr + udata->chunk_idx * idx_info->layout->u.chunk.size;

    
    udata->chunk_block.length = idx_info->layout->u.chunk.size;
    udata->filter_mask        = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__none_idx_load_metadata(const H5D_chk_idx_info_t H5_ATTR_UNUSED *idx_info)
{
    FUNC_ENTER_PACKAGE_NOERR

    

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static int
H5D__none_idx_iterate(const H5D_chk_idx_info_t *idx_info, H5D_chunk_cb_func_t chunk_cb, void *chunk_udata)
{
    H5D_chunk_rec_t chunk_rec;                
    unsigned        ndims;                    
    unsigned        u;                        
    int             curr_dim;                 
    hsize_t         idx;                      
    int             ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE

    
    assert(idx_info);
    assert(idx_info->f);
    assert(idx_info->pline);
    assert(!idx_info->pline->nused);
    assert(idx_info->layout);
    assert(chunk_cb);
    assert(chunk_udata);
    assert(H5_addr_defined(idx_info->layout->storage.u.chunk.idx_addr));

    
    memset(&chunk_rec, 0, sizeof(chunk_rec));
    chunk_rec.nbytes      = idx_info->layout->u.chunk.size;
    chunk_rec.filter_mask = 0;

    ndims = idx_info->layout->u.chunk.ndims - 1;
    assert(ndims > 0);

    
    for (u = 0; u < idx_info->layout->u.chunk.nchunks && ret_value == H5_ITER_CONT; u++) {
        
        idx = H5VM_array_offset_pre(ndims, idx_info->layout->u.chunk.max_down_chunks, chunk_rec.scaled);

        
        chunk_rec.chunk_addr =
            idx_info->layout->storage.u.chunk.idx_addr + idx * idx_info->layout->u.chunk.size;

        
        if ((ret_value = (*chunk_cb)(&chunk_rec, chunk_udata)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CALLBACK, H5_ITER_ERROR,
                        "failure in generic chunk iterator callback");

        
        curr_dim = (int)(ndims - 1);
        while (curr_dim >= 0) {
            
            chunk_rec.scaled[curr_dim]++;

            
            if (chunk_rec.scaled[curr_dim] >= idx_info->layout->u.chunk.chunks[curr_dim]) {
                
                chunk_rec.scaled[curr_dim] = 0;
                curr_dim--;
            } 
            else
                break;
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__none_idx_remove(const H5D_chk_idx_info_t H5_ATTR_UNUSED *idx_info,
                     H5D_chunk_common_ud_t H5_ATTR_UNUSED    *udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__none_idx_delete(const H5D_chk_idx_info_t *idx_info)
{
    hsize_t nbytes;              
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(idx_info);
    assert(idx_info->f);
    assert(idx_info->pline);
    assert(!idx_info->pline->nused); 
    assert(idx_info->layout);
    assert(H5_addr_defined(idx_info->layout->storage.u.chunk.idx_addr)); 

    
    nbytes = idx_info->layout->u.chunk.max_nchunks * idx_info->layout->u.chunk.size;
    if (H5MF_xfree(idx_info->f, H5FD_MEM_DRAW, idx_info->layout->storage.u.chunk.idx_addr, nbytes) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, H5_ITER_ERROR, "unable to free dataset chunks");

    idx_info->layout->storage.u.chunk.idx_addr = HADDR_UNDEF;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__none_idx_copy_setup(const H5D_chk_idx_info_t H5_ATTR_NDEBUG_UNUSED *idx_info_src,
                         const H5D_chk_idx_info_t                       *idx_info_dst)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(idx_info_src);
    assert(idx_info_src->f);
    assert(idx_info_src->pline);
    assert(!idx_info_src->pline->nused);
    assert(idx_info_src->layout);
    assert(H5_addr_defined(idx_info_src->layout->storage.u.chunk.idx_addr));

    assert(idx_info_dst);
    assert(idx_info_dst->f);
    assert(idx_info_dst->pline);
    assert(!idx_info_dst->pline->nused);
    assert(idx_info_dst->layout);

    
    H5_BEGIN_TAG(H5AC__COPIED_TAG)

    
    if (H5D__none_idx_create(idx_info_dst) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize chunked storage");

    
    H5_END_TAG

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__none_idx_size(const H5D_chk_idx_info_t H5_ATTR_UNUSED *idx_info, hsize_t *index_size)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(index_size);

    *index_size = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__none_idx_reset(H5O_storage_chunk_t *storage, bool reset_addr)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(storage);

    
    if (reset_addr)
        storage->idx_addr = HADDR_UNDEF;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5D__none_idx_dump(const H5O_storage_chunk_t *storage, FILE *stream)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(storage);
    assert(stream);

    Rfprintf(stream, "    Address: %" PRIuHADDR "\n", storage->idx_addr);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
