/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5HFpkg.h"    

herr_t
H5HF_stat_info(const H5HF_t *fh, H5HF_stat_t *stats)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fh);
    assert(stats);

    
    stats->man_size       = fh->hdr->man_size;
    stats->man_alloc_size = fh->hdr->man_alloc_size;
    stats->man_iter_off   = fh->hdr->man_iter_off;
    stats->man_nobjs      = fh->hdr->man_nobjs;
    stats->man_free_space = fh->hdr->total_man_free;
    stats->huge_size      = fh->hdr->huge_size;
    stats->huge_nobjs     = fh->hdr->huge_nobjs;
    stats->tiny_size      = fh->hdr->tiny_size;
    stats->tiny_nobjs     = fh->hdr->tiny_nobjs;
    

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF_size(const H5HF_t *fh, hsize_t *heap_size)
{
    H5HF_hdr_t *hdr;                 
    H5B2_t     *bt2       = NULL;    
    hsize_t     meta_size = 0;       
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fh);
    assert(heap_size);

    
    hdr = fh->hdr;

    
    *heap_size += hdr->heap_size;      
    *heap_size += hdr->man_alloc_size; 
    *heap_size += hdr->huge_size;      

    
    if (H5_addr_defined(hdr->man_dtable.table_addr) && hdr->man_dtable.curr_root_rows != 0)
        if (H5HF__man_iblock_size(hdr->f, hdr, hdr->man_dtable.table_addr, hdr->man_dtable.curr_root_rows,
                                  NULL, 0, heap_size) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL,
                        "unable to get fractal heap storage info for indirect block");

    
    if (H5_addr_defined(hdr->huge_bt2_addr)) {
        
        if (NULL == (bt2 = H5B2_open(hdr->f, hdr->huge_bt2_addr, hdr->f)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTOPENOBJ, FAIL,
                        "unable to open v2 B-tree for tracking 'huge' objects");

        
        if (H5B2_size(bt2, heap_size) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve B-tree storage info");
    } 

    
    if (H5_addr_defined(hdr->fs_addr)) {
        if (H5HF__space_size(hdr, &meta_size) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "can't retrieve FS meta storage info");
        *heap_size += meta_size;
    } 

done:
    
    if (bt2 && H5B2_close(bt2) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTCLOSEOBJ, FAIL, "can't close v2 B-tree for tracking 'huge' objects");

    FUNC_LEAVE_NOAPI(ret_value)
} 
