% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/HARmodel.R
\name{HARmodel}
\alias{HARmodel}
\title{Heterogeneous autoregressive (HAR) model for realized volatility model estimation}
\usage{
HARmodel(
  data,
  periods = c(1, 5, 22),
  periodsJ = c(1, 5, 22),
  periodsQ = c(1),
  leverage = NULL,
  RVest = c("rCov", "rBPCov", "rQuar"),
  type = "HAR",
  inputType = "RM",
  jumpTest = "ABDJumptest",
  alpha = 0.05,
  h = 1,
  transform = NULL,
  externalRegressor = NULL,
  periodsExternal = c(1),
  ...
)
}
\arguments{
\item{data}{an \code{xts} object containing either: intraday (log-)returns or realized measures already computed from such returns. 
In case more than one realized measure is needed, the object should have the as many columns as realized measures needed. 
The first column should always be the realized variance proxy. 
In case type is either \code{"HARQJ"} or \code{"CHARQ"} the order should be 
\code{"RV"}, \code{"BPV"}, \code{"RQ"}, or the relevant proxies.}

\item{periods}{a vector of integers indicating over how days the realized measures in the model should be aggregated. 
By default  \code{periods = c(1,5,22)}, which corresponds to one day, one week and one month respectively. This default is in line with Andersen et al. (2007).}

\item{periodsJ}{a vector of integers indicating over what time periods the jump components in the model should be aggregated. 
By default \code{periodsJ = c(1,5,22)}, which corresponds to one day, one week and one month respectively.}

\item{periodsQ}{a vector of integers indicating over what time periods the realized quarticity in the model should be aggregated. 
By default \code{periodsQ = c(1,5,22)}, which corresponds to one day, one week and one month respectively.}

\item{leverage}{a vector of integers indicating over what periods the negative returns should be aggregated.
See Corsi and Reno (2012) for more information. By default \code{leverage = NULL} and the model assumes the absence of a  leverage effect. 
Set \code{leverage = c(1,5,22)} to mimic the analysis in Corsi and Reno (2012).}

\item{RVest}{a character vector with one, two, or three elements. The first element always refers to the name of the function to estimate the daily integrated variance (non-jump-robust).
The second and third element depends on which type of model is estimated: 
If \code{type = "HARJ"}, \code{type = "HARCJ"}, \code{type = "HARQJ"} the second element refers to the name of the function to estimate the continuous component of daily volatility (jump robust). 
If type = "HARQ", the second element refers to the name of the function used to estimate the integrated quarticity.
If \code{type = "HARQJ"} the third element always refers to the name of the function used to estimate integrated quarticity.
By default \code{RVest = c("rCov","rBPCov","rQuar")}, i.e. using the realized volatility, realized bi-power variance, and realized quarticity.}

\item{type}{a string referring to the type of HAR model you would like to estimate. By default \code{type = "HAR"}, the most basic model. 
Other valid options for type are \code{"HARJ"}, \code{"HARCJ"}, \code{"HARQ"}, \code{"HARQJ"}, \code{"CHAR"}, or \code{"CHARQ"}.}

\item{inputType}{a string denoting if the input data consists of realized measures or high-frequency returns. 
Default "RM" is the only way to denote realized measures and everything else denotes returns.}

\item{jumpTest}{the function name of a function used to test whether the test statistic which determines whether the jump variability is significant that day. By default \code{jumpTest = "ABDJumptest"}, hence using the test statistic in Equation or Equation (18) of Andersen et al. (2007).
It is also possible to provide pre-computed test statistics for jump tests by setting \code{jumpTest} to \code{"testStat"}. These test statistics should still be passed as the third column.}

\item{alpha}{a real indicating the confidence level used in testing for jumps. By default \code{alpha = 0.05}.}

\item{h}{an integer indicating the number over how many days the dependent variable should be aggregated.
By default, \code{h = 1}, i.e. no aggregation takes place, you just model the daily realized volatility.}

\item{transform}{optionally a string referring to a function that transforms both the dependent and explanatory variables in the model. By default \code{transform = NULL}, so no transformation is done. Typical other choices in this context would be "log" or "sqrt".}

\item{externalRegressor}{an \code{xts} object of same number of rows as \code{data}, and one column. This is used as an external regressor. Default is \code{NULL}.}

\item{periodsExternal}{a vector of integers indicating over how days \code{externalRegressor} should be aggregated.}

\item{...}{extra arguments for jump test.}
}
\value{
The function outputs an object of class \code{HARmodel} and \code{\link{lm}} (so \code{HARmodel} is  a subclass of \code{\link{lm}}). Objects
of class \code{HARmodel} has the following methods \code{\link{plot.HARmodel}}, \code{\link{predict.HARmodel}}, \code{\link{print.HARmodel}}, and \code{\link{summary.HARmodel}}.
}
\description{
Function returns the estimates for the heterogeneous autoregressive model (HAR)
for realized volatility  discussed in Andersen et al. (2007) and Corsi (2009).
This model is mainly used to forecast the next day's volatility based on the high-frequency returns of the past.
}
\details{
The basic specification in Corsi (2009) is as follows. 
Let \eqn{RV_{t}} be the realized variances at day \eqn{t} and \eqn{RV_{t-k:t}} the average
realized variance in between \eqn{t-k} and \eqn{t}, \eqn{k \geq 0}. 

The dynamics of the model are given by
\deqn{
RV_{t+1} = \beta_0 + \beta_1 \ RV_{t} + \beta_2 \ RV_{t-4:t} + \beta_3 \ RV_{t-21:t} + \varepsilon_{t+1}
}
which is estimated by ordinary least squares under the assumption that at time \eqn{t},
the conditional mean of  \eqn{\varepsilon_{t+1}} is equal to zero.

For other specifications, please refer to the cited papers.

The standard errors reporting in the \code{print} and \code{summary} methods are Newey-West standard errors calculated with the \pkg{sandwich} package.
}
\examples{
\dontshow{if (!grepl("debian", sessionInfo()["platform"], fixed = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\dontshow{data.table::setDTthreads(2)}
# Example 1: HAR
# Forecasting daily Realized volatility for the S&P 500 using the basic HARmodel: HAR
library(xts)
RVSPY <- as.xts(SPYRM$RV5, order.by = SPYRM$DT)

x <- HARmodel(data = RVSPY , periods = c(1,5,22), RVest = c("rCov"),
              type = "HAR", h = 1, transform = NULL, inputType = "RM")
class(x)
x
summary(x)
plot(x)
predict(x)


# Example 2: HARQ
# Get the highfrequency returns
dat <- as.xts(sampleOneMinuteData[, makeReturns(STOCK), by = list(DATE = as.Date(DT))])
x <- HARmodel(dat, periods = c(1,5,10), periodsJ = c(1,5,10),
            periodsQ = c(1), RVest = c("rCov", "rQuar"),
              type="HARQ", inputType = "returns")
# Estimate the HAR model of type HARQ
class(x)
x
# plot(x)
# predict(x)

# Example 3: HARQJ with already computed realized measures
dat <- SPYRM[, list(DT, RV5, BPV5, RQ5)]
x <- HARmodel(as.xts(dat), periods = c(1,5,22), periodsJ = c(1),
              periodsQ = c(1), type = "HARQJ")
# Estimate the HAR model of type HARQJ
class(x)
x
# plot(x)
predict(x)

# Example 4: CHAR with already computed realized measures
dat <- SPYRM[, list(DT, RV5, BPV5)]

x <- HARmodel(as.xts(dat), periods = c(1, 5, 22), type = "CHAR")
# Estimate the HAR model of type CHAR
class(x)
x
# plot(x)
predict(x)

# Example 5: CHARQ with already computed realized measures
dat <- SPYRM[, list(DT, RV5, BPV5, RQ5)]

x <- HARmodel(as.xts(dat), periods = c(1,5,22), periodsQ = c(1), type = "CHARQ")
# Estimate the HAR model of type CHARQ
class(x)
x
# plot(x)
predict(x)

# Example 6: HARCJ with pre-computed test-statistics
## BNSJumptest manually calculated.
testStats <- sqrt(390) * (SPYRM$RV1 - SPYRM$BPV1)/sqrt((pi^2/4+pi-3 - 2) * SPYRM$medRQ1)
model <- HARmodel(cbind(as.xts(SPYRM[, list(DT, RV5, BPV5)]), testStats), type = "HARCJ")

\dontshow{\}) # examplesIf}
}
\references{
Andersen, T. G., Bollerslev, T., and Diebold, F. (2007). Roughing it up: Including jump components in the measurement, modelling and forecasting of return volatility. \emph{The Review of Economics and Statistics}, 89, 701-720.

Corsi, F. (2009). A simple approximate long memory model of realized volatility. \emph{Journal of Financial Econometrics}, 7, 174-196.

Corsi, F. and Reno R. (2012). Discrete-time volatility forecasting with persistent leverage effect and the link with continuous-time volatility modeling. \emph{Journal of Business & Economic Statistics}, 30, 368-380.

Bollerslev, T., Patton, A., and Quaedvlieg, R. (2016).  Exploiting the errors: A simple approach for improved volatility forecasting, \emph{Journal of Econometrics}, 192, 1-18.
}
\author{
Jonathan Cornelissen, Kris Boudt, Onno Kleen, and Emil Sjoerup.
}
\keyword{forecasting}
