# Written by Catoline Ring
#'Converts the HTTK-Pop population data table to a table of the parameters 
#'needed by HTTK, for a specific chemical.
#'
#'Takes the data table generated by \code{httkpop_generate}, and converts it to
#'the corresponding table of HTTK model parameters for a specified chemical and
#'HTTK model.
#'
#'@export
#'
#'@param indiv_dt A data table generated by \code{httkpop_generate()}.
#'@param chemcas The CAS number of one of the HTTK chemicals (see 
#'  \code{\link[httk]{get_cheminfo}}).
#' @param parameters A list of chemical-specific model parameters containing at least
#' Funbound.plasma, Clint, and Fhep.assay.correction. 
#'@param model One of the HTTK models: "1compartment", "3compartmentss", 
#'  "3compartment", or "pbtk".
#'@param poormetab TRUE (include poor metabolizers) or FALSE (exclude poor 
#'  metabolizers)
#'@param fup.censor TRUE (draw \code{Funbound.plasma} from a censored 
#'  distribution) or FALSE (draw \code{Funbound.plasma} from a non-censored 
#'  distribution)
#'@param sigma.factor The coefficient of variation to use for \code{Clint} and 
#'  \code{Funbound.plasma} distributions. Default value is 0.3.
#'@param Clint.vary TRUE (sample \code{Clint} values) or FALSE (hold 
#'  \code{Clint} fixed). Default TRUE. If \code{Clint.vary} is FALSE, then 
#'  \code{poormetab} will have no effect.
#'@param lod The average limit of detection for Funbound.plasma. if 
#'  \code{fup.censor == TRUE}, the \code{Funbound.plasma} distribution will be 
#'  censored below \code{lod/2}. Default value is 0.01.
#'@param well.stirred.correction If TRUE (default) then the well-stirred 
#'correction (Rowland et al., 1973) is 
#'used in the calculation of hepatic clearance for the models that do not
#'include flows for first-pass metabolism (currently, 1compartment and 
#' 3compartmentss). This assumes clearance relative to amount unbound in whole 
#'blood instead of plasma, but converted for use with plasma concentration.
#'  
#'@return A data.table whose columns correspond to the parameters of the HTTK 
#'  model specified in \code{model}, and whose rows correspond to the 
#'  individuals (rows) of \code{indiv_dt}.                                                                  #' @param well.stirred.correction Uses correction in calculation of hepatic clearance 
#' for well-stirred model if TRUE for hepatic.model well-stirred. This assumes 
#' clearance relative to amount unbound in whole blood instead of plasma, but 
#' converted to use with plasma concentration.
#' @param restrictive.clearance Protein binding not taken into account (set to 1) in 
#' liver clearance if FALSE.
#' @param clint.pvalue.threshold Hepatic clearance for chemicals where the in vitro 
#' clearance assay result has a p-values greater than the threshold are set to zero.
#' @param adjusted.Funbound.plasma Uses adjusted Funbound.plasma when set to TRUE.
#' @param regression Whether or not to use the regressions in calculating partition 
#' coefficients.             
#'  
#'  @examples 
#'  set.seed(42)
#'  indiv_examp <- httkpop_generate(method="d", 
#'  nsamp=100)
#'  httk_param <- get_httk_params(indiv_dt=indiv_examp, 
#'  chemcas="80-05-7", 
#'  model="1compartment", 
#'  poormetab=TRUE, 
#'  fup.censor=TRUE)
#'
#' @author Caroline Ring and John Wambaugh
#'
#'@references Ring, Caroline L., et al. "Identifying populations sensitive to 
#'environmental chemicals by simulating toxicokinetic variability." Environment 
#'International 106 (2017): 105-118
#'
#'@references Rowland, Malcolm, Leslie Z. Benet, and Garry G. Graham. "Clearance 
#' concepts in pharmacokinetics." Journal of Pharmacokinetics and 
#' Biopharmaceutics 1.2 (1973): 123-136.

get_httk_params <- function(indiv_dt,
                            chemcas=NULL,
                            parameters=NULL,
                            model,
                            poormetab,
                            fup.censor,
                            sigma.factor=0.3,
                            Clint.vary=TRUE,
                            lod=0.01,
                            adjusted.Funbound.plasma=T,
                            regression=T,
                            well.stirred.correction=T,
                            restrictive.clearance=T,
                            clint.pvalue.threshold=0.05){
  
  #First convert to physiological parameters used by HTTK
  indiv_bio <- httkpop_bio(indiv_dt = indiv_dt)
  
  #Next add chemical-specific Funbound.plasma and CLint values
  #Just cbind them together for now
  indiv_fc <- cbind(indiv_bio,
                    draw_fup_clint(this.chem=chemcas,
                      parameters=parameters,
                      nsamp=nrow(indiv_bio),
                      sigma.factor=sigma.factor,
                      poormetab=poormetab,
                      fup.censor=fup.censor,
                      Clint.vary=Clint.vary,
                      lod=lod,
                      adjusted.Funbound.plasma=adjusted.Funbound.plasma,
                      clint.pvalue.threshold=clint.pvalue.threshold))
  
  #Next convert the whole thing to the HTTK parameters for a specified model
  indiv_httk <- convert_httk(indiv.model.bio=indiv_fc, 
                 model=model,
                 this.chem=chemcas,
                 parameters=parameters,
                 adjusted.Funbound.plasma=adjusted.Funbound.plasma,regression=regression,
                 well.stirred.correction=well.stirred.correction,
                 restrictive.clearance=restrictive.clearance,
                 clint.pvalue.threshold=clint.pvalue.threshold)
  
  return(indiv_httk)
}