#' Generate and store population values of time-dependent input variables
#' 
#' @description
#' Generates and stores a population using \code{\link{create_mc_samples}} for 
#' reference by \code{\link{get_input_param_timeseries}}. Specifically, this 
#' function generates parameters for a sample population and stores those
#' parameters which are listed in the specified model's \code{input.var.names}.
#' 
#' @details
#' This function has no output; it writes (or re-writes) an Rds file in 
#' \code{httk/data} of the form \code{input_params_<model>_<CAS>.Rds} where <model>
#' and <CAS> are the model name and chemical CAS, respectively.
#' 
#' 
#' @param model A model which incorporates time-dependent parameters as specified
#' in the \code{model.list[[model]]$input.var.names}. 
#' 
#' @param chem.cas Chemical Abstract Services Registry Number (CAS-RN) -- the 
#' chemical must be identified by either CAS, name, or DTXISD
#' 
#' @param chem.name Chemical name (spaces and capitalization ignored) --  the 
#' chemical must be identified by either CAS, name, or DTXISD
#' 
#' @param dtxsid EPA's 'DSSTox Structure ID (\url{https://comptox.epa.gov/dashboard})   
#' -- the chemical must be identified by either CAS, name, or DTXSIDs
#' 
#' @param samples Size of the stored population; default 10000. Overridden by
#' the size of \code{httkpop.dt}, if specified, or by \code{nsamp} in 
#' \code{httkpop.generate.arg.list}, if specified. 
#' 
#' @param httkpop.dt A data table generated by \code{\link{httkpop_generate}}.
#' This defaults to NULL, in which case \code{\link{httkpop_generate}} is 
#' called to generate this table.
#' 
#' @param httkpop.generate.arg.list Arguments passed to \code{\link{httkpop_generate}}
#' for generating a population if \code{httkpop.dt} is not specified. Defaults 
#' to \code{list(nsamp = 10000, method = "direct resampling")}.
#' 
#' @param seed Sets a seed for repeatable generation of populations. Defaults to
#' null, in which case no seed is set. 
#' 
#' @param input.param.dir The path to the \code{input_params_data_files} directory,
#' which is used to store all \code{input_param} data files. If \code{input_params_data_files}
#' does not exist, this function will create it in the specified path. Default \code{NULL}, 
#' in which case the present working directory is used as default.
#' 
#' @author Colin Thomson
#' 
#' @seealso \code{\link{get_input_param_timeseries}}
#' 
#' @keywords lifestage
#'
#' @export gen_input_params
#' 
#' @useDynLib httk

gen_input_params <- function(model,
                             chem.cas=NULL,
                             chem.name=NULL,
                             dtxsid=NULL,
                             samples = 10000,
                             httkpop.dt = NULL,
                             httkpop.generate.arg.list =
                               list(nsamp = 10000,
                                    method = "direct resampling"),
                             seed = NULL,
                             input.param.dir = NULL) 
{
  ## Defining data.table arguments mistaken by check for global variables ##
  ..param <- weight_class <- gfr_class <- reth <- param.gen.function <- NULL
  ####
    
  # Required for reproducibility
  if (!is.null(seed)) {
    set.seed(seed)
  }
  
  # Model must be in model list and have input parameters listed; if no input
  #   parameters are listed then the model does not use time-dependent variables
  if (!(model %in% names(model.list))) {
    stop(paste('Model must be chosen from:', names(model.list)))
  } else if (is.null(model.list[[model]]$input.var.names)) {
    stop(paste('No input parameters are listed in modelinfo_', model, '.R.', 
               sep = ""))
  }
  
  if (is.null(chem.cas) & is.null(chem.name) & is.null(dtxsid)) {
    stop('chem.name, chem.cas, or dtxsid must be specified.')
  } else{
    chem.cas <- get_chem_id(chem.cas = chem.cas,
                            chem.name = chem.name,
                            dtxsid = dtxsid)$chem.cas
  }
  
  if (is.null(httkpop.dt)) {
    httkpop.dt <- do.call(httkpop_generate, httkpop.generate.arg.list)
  }
  generated.params <- create_mc_samples(model = model,
                                        chem.cas = chem.cas,
                                        samples = samples,
                                        suppress.messages = TRUE,
                                        httkpop.dt = httkpop.dt)
  
  # Retain demographic and age data
  stored.input.params <- data.frame(gender = httkpop.dt$gender,
                                    reth = httkpop.dt$reth,
                                    age_years = httkpop.dt$age_years,
                                    age_months = httkpop.dt$age_months,
                                    weight_class = httkpop.dt$weight_class,
                                    gfr_class = httkpop.dt$gfr_class,
                                    stringsAsFactors = FALSE)
  
  
  for (input.param in model.list[[model]]$input.var.names) {
    # Remove prepended 'd_' and get R name for parameter
    param <- model.list[[model]]$Rtosolvermap[[substring(input.param,3)]]
    if (param %in% names(generated.params)) {
      stored.input.params[,substring(input.param,3)] <- generated.params[,..param]
    }
  }

  if (is.null(input.param.dir)) {
    input.param.dir <- getwd()
  } else if (!file.exists(input.param.dir)) {
    stop("Specified directory for storing input parameters does not exist.")
  }
  
  storage.dir <- file.path(input.param.dir, "input_param_data_files")
  if (!file.exists(storage.dir)) {
    dir.create(storage.dir)
  }
  param.data.file <- file.path(storage.dir, paste("input_params_",
                                                  model,
                                                  "_",
                                                  chem.cas,
                                                  ".Rds",
                                                  sep = ""))
  saveRDS(as.data.table(stored.input.params), param.data.file)
    
}