% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/define_implementation.R, R/utils-dom-states.R,
%   R/utils-export.R
\name{define_impl}
\alias{define_impl}
\alias{make_impl_args_list}
\alias{define_domains}
\alias{define_pop_state}
\alias{define_env_state}
\title{Helpers for IPM construction}
\usage{
define_impl(proto_ipm, kernel_impl_list)

make_impl_args_list(kernel_names, int_rule, state_start, state_end)

define_domains(proto_ipm, ...)

define_pop_state(proto_ipm, ..., pop_vectors = list())

define_env_state(proto_ipm, ..., data_list = list())
}
\arguments{
\item{proto_ipm}{The name of the model.}

\item{kernel_impl_list}{A named list. Names correspond to kernel names. Each
kernel should have 3 slots defined - the \code{int_rule} (integration rule),
the \code{state_start} (the domain the kernel begins on), and the \code{state_end}
(the domain the kernel ends on). For more complicated models, it is usually
safest to use \code{make_impl_args_list} to generate this.}

\item{kernel_names}{A character vector with the names of the kernels
that parameters are being defined for.}

\item{int_rule}{The integration rule to be used for the kernel. The default is
"midpoint". "b2b" (bin to bin) and "cdf" (cumulative density functions) will
be implemented as well.}

\item{state_start}{The name of the state variable for the kernel that the
kernel acts on at time \emph{t}.}

\item{state_end}{The name of the state variable that the kernel produces
at time \emph{t+1}.}

\item{...}{Named expressions. See Details for more information on their usage in
each \code{define_*} function.}

\item{pop_vectors}{If the population vectors are already pre-defined (i.e. are
not defined by a function passed to \code{...}), then they can
be passed as a named list here.}

\item{data_list}{A list of named values that contain data used in the expressions
in \code{...} in \code{define_env_state()}.}
}
\value{
All \code{define_*} functions return a proto_ipm. \code{make_impl_args_list}
returns a list, and so must be used within a call to \code{define_impl} or
before initiating the model creation procedure.
}
\description{
Helpers for IPM construction
}
\details{
These are helper functions to define IPMs. They are used after defining the kernels,
but before calling \code{make_ipm()} They are meant to be called in the
following order:

\enumerate{

  \item \code{define_impl()}

  \item \code{define_domains()}

  \item \code{define_pop_state()}

  \item \code{define_env_state()}

}

The order requirement is so that information is correctly matched to each kernel.
Below are specific details on the way each works.

\strong{\code{define_impl}}

This has two arguments - \code{proto_ipm} (the model object you wish to work with),
and the \code{kernel_impl_list}. The format of the \code{kernel_impl_list} is:
names of the list should be kernel names, and each kernel should have 3 entries:
\code{int_rule}, \code{state_start}, and \code{state_end}. See examples.

\strong{\code{define_domains}}

If the \code{int_rule = "midpoint"}, the \code{...} entries are vectors of
length 3 where the name corresponds to the
state variable, the first entry is the lower bound of the domain, the second
is the upper bound of the domain, and the third entry is the number of
meshpoints. Other \code{int_rule}s are not yet implemented, so for now this is the
only format they can take. See examples.

\strong{\code{define_pop_state}}

This takes either calls to functions in the \code{...}, or a pre-generated
list of vectors in the \code{pop_vectors}. The names used
for each entry in \code{...} and/or for the \code{pop_vectors} should be
\code{n_<state_variable>}. See examples.

\strong{\code{define_env_state}}

Takes expressions that generate values for environmental covariates at each
iteration of the model in \code{...}. The \code{data_list} should contain any
parameters that the function uses, as well as the function itself. The
functions should return named lists. Names in that list can be referenced in
vital rate expressions and/or kernel formulas.
}
\examples{

# Example with kernels named "P" and "F", and a domain "z"

kernel_impl_list <- list(P = list(int_rule = "midpoint",
                                  state_start = "z",
                                  state_end = "z"),
                         F = list(int_rule = "midpoint",
                                  state_start = "z",
                                  state_end = "z"))

# an equivalent version using make_impl_args_list

kernel_impl_list <- make_impl_args_list(
     kernel_names = c("P", "F"),
     int_rule     = c("midpoint", "midpoint"),
     state_start  = c("z", "z"),
     state_end    = c("z", "z")
)

data(sim_di_det_ex)

proto_ipm <- sim_di_det_ex$proto_ipm

# define_domains

lower_bound <- 1
upper_bound <- 100
n_meshpoints <- 50


define_domains(proto_ipm, c(lower_bound, upper_bound, n_meshpoints))

# define_pop_state with a state variable named "z". Note that "n_" is prefixed
# to denote that it is a population state function!

define_pop_state(proto_ipm, n_z = runif(100))

# alternative, we can make a list before starting to make the IPM

pop_vecs <- list(n_z = runif(100))

define_pop_state(proto_ipm, pop_vectors = pop_vecs)

# define_env_state. Generates a random draw from a known distribution
# of temperatures.

env_sampler <- function(env_pars) {

  temp <- rnorm(1, env_pars$temp_mean, env_pars$temp_sd)

  return(list(temp = temp))

}

env_pars <- list(temp_mean = 12, temp_sd = 2)

define_env_state(
 proto_ipm,
 env_values = env_sampler(env_pars),
 data_list  = list(env_sampler = env_sampler,
                   env_pars    = env_pars)

)


}
