% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iprior.R
\name{iprior}
\alias{iprior}
\alias{iprior.default}
\alias{iprior.formula}
\alias{iprior.ipriorKernel}
\alias{iprior.ipriorMod}
\title{Fit an I-prior regression model}
\usage{
\method{iprior}{default}(y, ..., kernel = "linear", method = "direct",
  control = list(), interactions = NULL, est.lambda = TRUE,
  est.hurst = FALSE, est.lengthscale = FALSE, est.offset = FALSE,
  est.psi = TRUE, fixed.hyp = NULL, lambda = 1, psi = 1,
  nystrom = FALSE, nys.seed = NULL, model = list(), train.samp,
  test.samp)

\method{iprior}{formula}(formula, data, kernel = "linear",
  one.lam = FALSE, method = "direct", control = list(),
  est.lambda = TRUE, est.hurst = FALSE, est.lengthscale = FALSE,
  est.offset = FALSE, est.psi = TRUE, fixed.hyp = NULL, lambda = 1,
  psi = 1, nystrom = FALSE, nys.seed = NULL, model = list(),
  train.samp, test.samp, ...)

\method{iprior}{ipriorKernel}(object, method = "direct",
  control = list(), ...)

\method{iprior}{ipriorMod}(object, method = NULL, control = list(),
  iter.update = 100, ...)
}
\arguments{
\item{y}{Vector of response variables}

\item{...}{Only used when fitting using non-formula, enter the variables
(vectors or matrices) separated by commas.}

\item{kernel}{Character vector indicating the type of kernel for the
variables. Available choices are: \itemize{ \item{\code{"linear"} -
(default) for the linear kernel} \item{\code{"canonical"} - alternative
name for \code{"linear"}} \item{\code{"fbm"}, \code{"fbm,0.5"} - for the
fBm kernel with Hurst coefficient 0.5 (default)} \item{\code{"se"},
\code{"se,1"} - for the SE kernel with lengthscale 1 (default)}
\item{\code{"poly"}, \code{"poly2"}, \code{"poly2,0"} - for the polynomial
kernel of degree 2 with offset 0 (default)} \item{\code{"pearson" - for the
Pearson kernel}}} The \code{kernel} argument can also be a vector of length
equal to the number of variables, therefore it is possible to specify
different kernels for each variables. Note that factor type variables are
assigned the Pearson kernel by default, and that non-factor types can be
forced to use the Pearson kernel (not recommended).}

\item{method}{The estimation method. One of: \itemize{ \item{\code{"direct"}
- for the direct minimisation of the marginal deviance using
\code{optim()}'s L-BFGS method} \item{\code{"em"} - for the EM algorithm}
\item{\code{"mixed"} - combination of the direct and EM methods}
\item{\code{"fixed"} - for just obtaining the posterior regression function
with fixed hyperparameters (default method when setting \code{fixed.hyp =
TRUE})} \item{\code{"canonical"} - an efficient estimation method which
takes advantage of the structure of the linear kernel} }}

\item{control}{(Optional) A list of control options for the estimation
procedure: \describe{ \item{\code{maxit}}{The maximum number of iterations
for the quasi-Newton optimisation or the EM algorithm. Defaults to
\code{100}.} \item{\code{em.maxit}}{For \code{method = "mixed"}, the number
of EM steps before switching to direct optimisation. Defaults to \code{5}.}
\item{\code{stop.crit}}{The stopping criterion for the EM and L-BFGS
algorithm, which is the difference in successive log-likelihood values.
Defaults to \code{1e-8}.} \item{\code{theta0}}{The initial values for the
hyperparameters. Defaults to random starting values.}
\item{\code{report}}{The interval of reporting for the \code{optim()}
function.} \item{\code{restarts}}{The number of random restarts to perform.
Defaults to \code{0}. It's also possible to set it to \code{TRUE}, in which
case the number of random restarts is set to the total number of available
cores.} \item{\code{no.cores}}{The number of cores in which to do random
restarts. Defaults to the total number of available cores.}
\item{\code{omega}}{The overrelaxation parameter for the EM algorithm - a
value between 0 and 1.}}}

\item{interactions}{Character vector to specify the interaction terms. When
using formulas, this is specified automatically, so is not required. Syntax
is \code{"a:b"} to indicate variable \code{a} interacts with variable
\code{b}.}

\item{est.lambda}{Logical. Estimate the scale parameters? Defaults to
\code{TRUE}.}

\item{est.hurst}{Logical. Estimate the Hurst coefficients for fBm kernels?
Defaults to \code{FALSE}.}

\item{est.lengthscale}{Logical. Estimate the lengthscales for SE kernels?
Defaults to \code{FALSE}.}

\item{est.offset}{Logical. Estimate the offsets for polynomial kernels?
Defaults to \code{FALSE}.}

\item{est.psi}{Logical. Estimate the error precision? Defaults to
\code{TRUE}.}

\item{fixed.hyp}{Logical. If \code{TRUE}, then no hyperparameters are
estimated, i.e. all of the above \code{est.x} are set to \code{FALSE}, and
vice versa. If \code{NULL} (default) then all of the \code{est.x} defaults
are respected.}

\item{lambda}{Initial/Default scale parameters. Relevant especially if
\code{est.lambda = FALSE}.}

\item{psi}{Initial/Default value for error precision. Relevant especially if
\code{est.psi = FALSE}.}

\item{nystrom}{Either logical or an integer indicating the number of Nystrom
samples to take. Defaults to \code{FALSE}. If \code{TRUE}, then
approximately 10\% of the sample size is used for the Nystrom
approximation.}

\item{nys.seed}{The random seed for the Nystrom sampling. Defaults to
\code{NULL}, which means the random seed is not fixed.}

\item{model}{DEPRECATED.}

\item{train.samp}{(Optional) A vector indicating which of the data points
should be used for training, and the remaining used for testing.}

\item{test.samp}{(Optional) Similar to \code{train.samp}, but on test samples
instead.}

\item{formula}{The formula to fit when using formula interface.}

\item{data}{Data frame containing variables when using formula interface.}

\item{one.lam}{Logical. When using formula input, this is a convenient way of
letting the function know to treat all variables as a single variable (i.e.
shared scale parameter). Defaults to \code{FALSE}.}

\item{object}{An \code{ipriorKernel} or \code{ipriorMod} object.}

\item{iter.update}{The number of iterations to perform when calling the
function on an \code{ipriorMod} object. Defaults to \code{100}.}
}
\value{
An \code{ipriorMod} object. Several accessor functions have been
  written to obtain pertinent things from the \code{ipriorMod} object. The
  \code{print()} and \code{summary()} methods display the relevant model
  information.
}
\description{
A function to perform regression using I-priors. The I-prior model parameters
may be estimated in a number of ways: direct minimisation of the marginal
deviance, EM algorithm, fixed hyperparameters, or using a Nystrom kernel
approximation.
}
\details{
The \code{iprior()} function is able to take formula based input and
non-formula. When not using formula, the syntax is as per the default S3
method. That is, the response variable is the vector \code{y}, and any
explanatory variables should follow this, and separated by commas.

As described \link[=kernL]{here}, the model can be loaded first into an
\code{ipriorKernel} object, and then passed to the \code{iprior()} function
to perform the estimation.
}
\section{Methods (by class)}{
\itemize{
\item \code{ipriorKernel}: Takes in object of type \code{ipriorKernel}, a loaded and
prepared I-prior model, and proceeds to estimate it.

\item \code{ipriorMod}: Re-run or continue running the EM algorithm from last
attained parameter values in object \code{ipriorMod}.
}}

\examples{

# Formula based input
(mod.stackf <- iprior(stack.loss ~ Air.Flow + Water.Temp + Acid.Conc.,
                      data = stackloss))
mod.toothf <- iprior(len ~ supp * dose, data = ToothGrowth)
summary(mod.toothf)

# Non-formula based input
mod.stacknf <- iprior(y = stackloss$stack.loss,
                      Air.Flow = stackloss$Air.Flow,
                      Water.Temp = stackloss$Water.Temp,
                      Acid.Conc. = stackloss$Acid.Conc.)
mod.toothnf <- iprior(y = ToothGrowth$len, ToothGrowth$supp, ToothGrowth$dose,
                      interactions = "1:2")

# Formula based model option one.lam = TRUE
# Sets a single scale parameter for all variables
modf <- iprior(stack.loss ~ ., data = stackloss, one.lam = TRUE)
modnf <- iprior(y = stackloss$stack.loss, X = stackloss[1:3])
all.equal(coef(modnf), coef(modnf))  # both models are equivalent

# Fit models using different kernels
dat <- gen_smooth(n = 100)
mod <- iprior(y ~ X, dat, kernel = "fbm")  # Hurst = 0.5 (default)
mod <- iprior(y ~ X, dat, kernel = "poly3")  # polynomial degree 3

# Fit models using various estimation methods
mod1 <- iprior(y ~ X, dat)
mod2 <- iprior(y ~ X, dat, method = "em")
mod3 <- iprior(y ~ X, dat, method = "canonical")
mod4 <- iprior(y ~ X, dat, method = "mixed")
mod5 <- iprior(y ~ X, dat, method = "fixed", lambda = coef(mod1)[1],
               psi = coef(mod1)[2])
c(logLik(mod1), logLik(mod2), logLik(mod3), logLik(mod4),
  logLik(mod5))

\dontrun{

# For large data sets, it is worth trying the Nystrom method
mod <- iprior(y ~ X, gen_smooth(5000), kernel = "se", nystrom = 50,
              est.lengthscale = TRUE)  # a bit slow
plot_fitted(mod, ci = FALSE)
}

}
\seealso{
\link[=optim]{optim}, \link[=update.ipriorMod]{update},
  \link[=check_theta]{check_theta}, print, summary, plot, coef, sigma,
  fitted, predict, logLik, deviance.
}
