% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scale_lm.R
\name{center_mod}
\alias{center_mod}
\alias{center_lm}
\title{Center variables in fitted regression models}
\usage{
center_mod(
  model,
  binary.inputs = "0/1",
  center.response = FALSE,
  data = NULL,
  apply.weighted.contrasts = getOption("jtools-weighted.contrasts", FALSE),
  ...
)
}
\arguments{
\item{model}{A regression model of type \code{lm}, \code{glm}, or
\code{\link[survey]{svyglm}}; others may work as well but have not been
tested.}

\item{binary.inputs}{Options for binary variables. Default is \code{0/1};
\code{0/1} keeps original scale; \code{-0.5,0.5} rescales 0 as -0.5 and 1
as 0.5; \code{center} subtracts the mean; and \code{full} treats them
like other continuous variables.}

\item{center.response}{Should the response variable also be centered?
Default is \code{FALSE}.}

\item{data}{If you provide the data used to fit the model here, that data
frame is used to re-fit the model instead of the \code{\link[stats:model.frame]{stats::model.frame()}}
of the model. This is particularly useful if you have variable
transformations or polynomial terms specified in the formula.}

\item{apply.weighted.contrasts}{Factor variables cannot be scaled, but you
can set the contrasts such that the intercept in a regression model will
reflect the true mean (assuming all other variables are centered). If set
to TRUE, the argument will apply weighted effects coding to all factors.
This is similar to the R default effects coding, but weights according to
how many observations are at each level. An adapted version of
\code{contr.wec()} from the \code{wec} package is used to do this. See
that package's documentation and/or Grotenhuis et al. (2016) for more
info.}

\item{...}{Arguments passed on to \code{\link[=gscale]{gscale()}}.}
}
\value{
The functions returns a \code{lm} or \code{glm} object, inheriting
from whichever class was supplied.
}
\description{
\code{center_mod} (previously known as \code{center_lm}) takes fitted regression models
and mean-centers the
continuous variables in the model to aid interpretation, especially in
the case of models with interactions. It is a wrapper to
\code{\link{scale_mod}}.
}
\details{
This function will mean-center all continuous variables in a
regression model for ease of interpretation, especially for those models
that have
interaction terms. The mean for \code{svyglm} objects is calculated using
\code{svymean}, so reflects the survey-weighted mean. The weight variables
in \code{svyglm} are not centered, nor are they in other \code{lm} family
models.

This function re-estimates the model, so for large models one should
expect
a runtime equal to the first run.
}
\examples{

fit <- lm(formula = Murder ~ Income * Illiteracy,
          data = as.data.frame(state.x77))
fit_center <- center_mod(fit)

# With weights
fitw <- lm(formula = Murder ~ Income * Illiteracy,
           data = as.data.frame(state.x77),
           weights = Population)
fitw_center <- center_mod(fitw)

# With svyglm
if (requireNamespace("survey")) {
library(survey)
data(api)
dstrat <- svydesign(id = ~1, strata = ~stype, weights = ~pw,
                    data = apistrat, fpc =~ fpc)
regmodel <- svyglm(api00 ~ ell * meals, design = dstrat)
regmodel_center <- center_mod(regmodel)
}

}
\references{
Bauer, D. J., & Curran, P. J. (2005). Probing interactions in fixed and
multilevel regression: Inferential and graphical techniques.
\emph{Multivariate Behavioral Research}, \emph{40}(3), 373-400.

Cohen, J., Cohen, P., West, S. G., & Aiken, L. S. (2003). \emph{Applied
multiple regression/correlation analyses for the behavioral sciences} (3rd
ed.). Mahwah, NJ: Lawrence Erlbaum Associates, Inc.
}
\seealso{
\code{\link{sim_slopes}} performs a simple slopes analysis.

\code{\link{interact_plot}} creates attractive, user-configurable plots of
interaction models.

Other standardization: 
\code{\link{center}()},
\code{\link{gscale}()},
\code{\link{scale_mod}()},
\code{\link{standardize}()}
}
\author{
Jacob Long \email{jacob.long@sc.edu}
}
\concept{standardization}
