% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/learning_rate_schedules.R
\name{learning_rate_schedule_exponential_decay}
\alias{learning_rate_schedule_exponential_decay}
\title{A LearningRateSchedule that uses an exponential decay schedule}
\usage{
learning_rate_schedule_exponential_decay(
  initial_learning_rate,
  decay_steps,
  decay_rate,
  staircase = FALSE,
  ...,
  name = NULL
)
}
\arguments{
\item{initial_learning_rate}{A scalar \code{float32} or \code{float64} \code{Tensor} or a R
number. The initial learning rate.}

\item{decay_steps}{A scalar \code{int32} or \code{int64} \code{Tensor} or an R number. Must
be positive.  See the decay computation above.}

\item{decay_rate}{A scalar \code{float32} or \code{float64} \code{Tensor} or an R number.
The decay rate.}

\item{staircase}{Boolean.  If \code{TRUE} decay the learning rate at discrete
intervals.}

\item{...}{For backwards and forwards compatibility}

\item{name}{String. Optional name of the operation.  Defaults to
'ExponentialDecay'.}
}
\description{
A LearningRateSchedule that uses an exponential decay schedule
}
\details{
When training a model, it is often useful to lower the learning rate as
the training progresses. This schedule applies an exponential decay function
to an optimizer step, given a provided initial learning rate.

The schedule is a 1-arg callable that produces a decayed learning
rate when passed the current optimizer step. This can be useful for changing
the learning rate value across different invocations of optimizer functions.
It is computed as:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{decayed_learning_rate <- function(step)
  initial_learning_rate * decay_rate ^ (step / decay_steps)
}\if{html}{\out{</div>}}

If the argument \code{staircase} is \code{TRUE}, then \code{step / decay_steps} is
an integer division (\code{\%/\%}) and the decayed learning rate follows a
staircase function.

You can pass this schedule directly into a optimizer
as the learning rate (see example)
Example: When fitting a Keras model, decay every 100000 steps with a base
of 0.96:

\if{html}{\out{<div class="sourceCode R">}}\preformatted{initial_learning_rate <- 0.1
lr_schedule <- learning_rate_schedule_exponential_decay(
    initial_learning_rate,
    decay_steps = 100000,
    decay_rate = 0.96,
    staircase = TRUE)

model \%>\% compile(
  optimizer= optimizer_sgd(learning_rate = lr_schedule),
  loss = 'sparse_categorical_crossentropy',
  metrics = 'accuracy')

model \%>\% fit(data, labels, epochs = 5)
}\if{html}{\out{</div>}}
}
\seealso{
\itemize{
\item \url{https://www.tensorflow.org/api_docs/python/tf/keras/optimizers/schedules/ExponentialDecay}
}
}
