% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semTable.R
\name{semTable}
\alias{semTable}
\title{Creates Structural Equation Modeling Tables}
\usage{
semTable(object, file = NULL, params = "all", fit = c("chi-square", "cfi",
  "tli", "rmsea"), names_fit = fit, standardized = FALSE,
  names_upper = TRUE, single_spaced = TRUE, type = "latex",
  group = NULL, longtable = FALSE)
}
\arguments{
\item{object}{A lavaan object returned by cfa() or sem().}

\item{file}{Character string for file name.  Default is NULL,
meaning no output file.}

\item{params}{Parameters to be included. Valid values are
"loadings", "slopes", "intercepts", "means", "residuals",
"covariances", "latentvariances", "latentmeans" and
"thresholds". Defaults to "all" which includes all available
parameters (but excludes "means"). See Details.}

\item{fit}{A vector of fit measures that to be included in the
note. Listing "chi-square" will do special formatting to the
chi-square value in the note. Any other measures listed must
correspond to measures found in fitMeasures(object).}

\item{names_fit}{Names for the fit measures requested by the fit
parameter.  Must have same number of elements as fit.  For
example, fit = c("cfi.scaled", "tli.scaled"), names_fit =
c("CFI", "TLI").}

\item{standardized}{Should standarized results be presented along
with unstandardized?  Default is FALSE. See Details.}

\item{names_upper}{Should the names of the model fit parameters be
forced to be uppercase.  The default is TRUE.  This will also
affect whatever is specified in names_fit.}

\item{single_spaced}{Default = TRUE. If a double-spaced table is
needed, set single_spaced = FALSE.}

\item{type}{Type of output table ("latex" or "html"). Defaults to
"latex".}

\item{group}{Group for which parameters should be
reported. Provide the value in the data that indicates the
desired group. Only necessary for multiple group
models. Defaults to NULL.}

\item{longtable}{Should a latex longtable be generated? Defaults
to FALSE, which makes the table tabular. Ignored if type =
"html".}
}
\value{
SEM table of desired type.
}
\description{
Creates LaTeX markup for structural equation modeling output
tables in the style of the American Psychological
Association(APA). Input objects should be created by the
"\code{lavaan}" package.
}
\details{
The argument params determines the inclusion of estimate sections.
\itemize{
\item "loadings" are the factor loadings in the model.
\item "slopes" are the regression slopes in the model.
\item "intercepts" are
the observed variable intercepts.
\item "means" are the observed
variable means.
\item "residuals" are the observed
variable residual variances.
\item "covariances" are the observed
variable covariances.
\item "latentvariances" are the latent
variable variances and covariances.
\item "latentmeans" are the latent variable
means (or intercepts).
\item "thresholds" arise in latent
response variates (non-numeric indicator data).
}

The stantardized parameter regulates the number of columns to be
included.  standardized=FALSE implies there will be four columns:
\enumerate{
\item the estimate
\item the standard error
\item the z-value, and
\item the p-value.
}

When standardized = TRUE, the columns are:
\enumerate{
\item the parameter estimates,
\item the standard errors,
\item standardized parameter estimates, and
\item standardized standard errors.
}

The standardized parameters are obtained by updating the output
with the options std.lv = TRUE and std.ov = TRUE.  If these
options were used when originally creating output, setting
standardized = TRUE will yield two identical sets of two columns.
}
\examples{
\donttest{
## These run longer than 5 seconds
## CFA model
require(lavaan)
HS.model <- ' visual  =~ x1 + x2 + x3
textual =~ x4 + x5 + x6
speed   =~ x7 + x8 + x9'
output1 <- cfa(HS.model, data = HolzingerSwineford1939, std.lv = TRUE)
semTable(output1, fit = "rmsea",
standardized = TRUE, type = "latex")
## Basic SEM model
regmodel <- "x1 ~ x2 + x3
x1 ~1"
output1a <- sem(regmodel, data = HolzingerSwineford1939, std.lv = TRUE)
semTable(output1a, fit = "rmsea", type = "html")
#### Example with file output
##semTable(output1, file = "exampleTable.html", fit = "rmsea",
##standardized = TRUE, params = c("loadings", "latentvariances"),
##type = "html")

model <- "factor =~ .7*y1 + .7*y2 + .7*y3 + .7*y4
y1 | -1*t1 + 1*t2
y2 | -.5*t1 + 1*t2
y3 | -.2*t1 + 1*t2
y4 | -1*t1 + 1*t2
"
dat <- simulateData(model, sample.nobs = 300)
testmodel <- "ExampleFactor =~ y1 + y2 + y3 + y4"
output2 <- cfa(testmodel, data = dat, ordered = colnames(dat),
    std.lv = FALSE)
semTable(output2,
    params = c("loadings", "thresholds", "residuals"),
    fit = c("tli", "chi-square"),
    names_fit = c("TLI", "chi-square"), type = "html")

## Example with file output
## semTable(output, file = "catTable.tex",
##    params = c("loadings", "thresholds", "residuals"),
##    fit = c("tli", "chi-square"),
##    names_fit = c("TLI", "chi-square"), type = "latex")
}
}
\author{
Ben Kite <bakite@ku.edu>
}
