#' Render Dockerized R Markdown Documents
#'
#' Render dockerized R Markdown documents using Docker containers.
#' Rabix tools/workflows will be ran if there is a \code{Rabixfile}
#' generated by \link{lift} under the same directory.
#'
#' Before using \code{drender()}, run \link{lift} on the document
#' first to generate \code{Dockerfile}.
#' See \code{vignette('liftr-intro')} for details about the extended
#' YAML front-matter metadata format and system requirements for
#' rendering dockerized R Markdown documents.
#'
#' @param input Input file to render in Docker container.
#' @param tag Docker image name to build, sent as docker argument \code{-t}.
#' If not specified, it will use the same name as the input file.
#' @param build_args A character string specifying additional
#' \code{docker build} arguments. For example,
#' \code{--pull=true -m="1024m" --memory-swap="-1"}.
#' @param container_name Docker container name to run.
#' If not specified, we will generate and use a random name.
#' @param reset Should we cleanup the Docker container and
#' Docker image after getting the rendered result?
#' @param ... Additional arguments passed to
#' \code{\link[rmarkdown]{render}}.
#'
#' @return Rendered file is written to the same directory of the input file.
#' A character vector with the image name and container name will be
#' returned. You will be able to manage them with \code{docker}
#' commands later if \code{reset = FALSE}.
#'
#' @export drender
#'
#' @importFrom rmarkdown render
#'
#' @examples
## Included in \dontrun{} since users need
## Docker and Rabix installed to run them.
#' # 1. Dockerized R Markdown document
#' # Docker is required to run the example,
#' # so make sure we can use `docker` in shell.
#' dir_docker = paste0(tempdir(), '/drender_docker/')
#' dir.create(dir_docker)
#' file.copy(system.file("docker.Rmd", package = "liftr"), dir_docker)
#' docker_input = paste0(dir_docker, "docker.Rmd")
#' lift(docker_input)
#' \dontrun{
#' drender(docker_input)
#' # view rendered document
#' browseURL(paste0(dir_docker, "docker.html"))}
#'
#' # 2. Dockerized R Markdown document with Rabix options
#' # Docker and Rabix are required to run the example,
#' # so make sure we can use `docker` and `rabix` in shell.
#' dir_rabix  = paste0(tempdir(), '/drender_rabix/')
#' dir.create(dir_rabix)
#' file.copy(system.file("rabix.Rmd", package = "liftr"), dir_rabix)
#' rabix_input = paste0(dir_rabix, "rabix.Rmd")
#' lift(rabix_input)
#' \dontrun{
#' drender(rabix_input)
#' # view rendered document
#' browseURL(paste0(dir_rabix, "rabix.html"))}
drender = function (input = NULL,
                    tag = NULL, build_args = NULL, container_name = NULL,
                    reset = TRUE, ...) {

  if (is.null(input))
    stop('missing input file')
  if (!file.exists(normalizePath(input)))
    stop('input file does not exist')

  # run rabix first if Rabixfile is found
  rabixfile_path = paste0(file_dir(input), '/Rabixfile')

  if (file.exists(rabixfile_path)) {

    if (Sys.which('rabix') == '')
      stop('Cannot find `rabix` on system search path,
           please ensure we can use `rabix` from shell')

    rabix_cmd = paste0(readLines(rabixfile_path), collapse = '\n')
    system(rabix_cmd)
  }

  # docker build
  dockerfile_path = paste0(file_dir(input), '/Dockerfile')

  if (!file.exists(dockerfile_path))
    stop('Cannot find Dockerfile in the same directory of input file')

  if (Sys.which('docker') == '')
    stop('Cannot find `docker` on system search path,
         please ensure we can use `docker` from shell')

  image_name = ifelse(is.null(tag), file_name_sans(input), tag)
  docker_build_cmd = paste0("docker build --no-cache=true --rm=true ",
                            build_args, " -t=\"", image_name, "\" ",
                            file_dir(dockerfile_path))

  # docker run
  container_name = ifelse(is.null(container_name),
                          paste0('liftr_container_', uuid()),
                          container_name)

  docker_run_cmd_base =
    paste0("docker run --rm -i -t --name \"", container_name,
           "\" -u `id -u $USER` -v \"",
           file_dir(dockerfile_path), ":", "/liftrroot/\" ",
           image_name,
           " /usr/bin/Rscript -e \"library('knitr');library('rmarkdown');",
           "library('shiny');setwd('/liftrroot/');")

  # process additional arguments passed to rmarkdown::render()
  dots_arg = list(...)

  if (length(dots_arg) == 0L) {

    docker_run_cmd = paste0(docker_run_cmd_base, "render(input = '",
                            file_name(input), "')\"")

  } else {

    if (!is.null(dots_arg$input))
      stop('input can only be specified once')

    if (!is.null(dots_arg$output_file) |
        !is.null(dots_arg$output_dir) |
        !is.null(dots_arg$intermediates_dir)) {
      stop('`output_file`, `output_dir`, and `intermediates_dir`
           are not supported to be changed now, we will consider
           this in the next versions.')
    }

    dots_arg$input = file_name(input)
    tmp = tempfile()
    dput(dots_arg, file = tmp)
    render_args = paste0(readLines(tmp), collapse = '\n')
    render_cmd = paste0("do.call(render, ", render_args, ')')

    docker_run_cmd = paste0(docker_run_cmd_base, render_cmd, "\"")

    }

  system(docker_build_cmd)
  system(docker_run_cmd)

  # cleanup docker containers and images
  # TODO: needs exception handling
  if (reset) {
    system(paste0("docker stop \"", container_name, "\""))
    system(paste0("docker rm -f \"", container_name, "\""))
    system(paste0("docker rmi -f \"", image_name, "\""))
  }

  return(c('image_name' = image_name, 'container_name' = container_name))

}
