#--------------------------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------------------------
# 15th March 2014, Adrian Timpson

# All functions required to produce both main reports. Utilising package 'rtf' to produce .doc outputs.
# A premininary 'allele report' is generated by allele.report(), using only 'admin'. This is intended to assist the user in choosing parameter values for a proper run of the likeLTD program.
# Once the user has obtained Pros and Def results, a full output report may be generated by output.report().
# Many sections are common to both reports, so for simplicity these are performed by common.report.section()
# Most other functions perform a specific task in preparing a section of the report

# The exception to this is latex.maker() and its three helper functions, which outputs a separate .tex file containing the two allele tables (crime scene and references)

# The admin report draws its administrative data (default parameters, filepaths etc) from the 'admin' object
# In contrast, the final output report draws its administrative data from the values that were ACTUALLY used (since there are various opportunities downstream of the 'admin' object for these to be changed by the user).
# This means that the object 'genetics' generated by pack.genetics.for.allele.report() for the allele report is different to the 'genetics' object generated by pack.genetics.for.output.report() for the final report.
#--------------------------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------------------------
rounder <- function(x,n){
	# deals with three repetitive issues (that are more insipid than might be expected! :
	# numerics are best converted to text, to prevent MS Word from formatting them
	# rounding
	# rounding a number with a few zeros results in 0 when we actually want 0.00 
	if(is.numeric(x))x <- round(x,n) # method for vectors
	if(is.data.frame(x)){	# method for data.frames, each column at a time
		for(c in 1:ncol(x))if(is.numeric(x[,c]))x[,c] <- round(x[,c],n)
		}
	result <- format(x,nsmall=n,scientific=F)
return(result)}
#--------------------------------------------------------------------------------------------------------------------
# shorthand to avoid bothering with the argument n, for clarity in the code. Can handle vectors and data.frames
round.0 <- function(x)rounder(x,0)
round.1 <- function(x)rounder(x,1)
round.3 <- function(x)rounder(x,3)

#--------------------------------------------------------------------------------------------------------------------
latex.table.header <- function(genetics){
	# helper function for latex.maker()
	N <- ncol(genetics$summary$table$latex)+1
	text <- c('
	\\begin{sidewaystable}[p]\n
	%\\setcounter{table}{1}
	',
	paste('\\begin{center}\\begin{tabular}{|',paste(rep('c|',N),collapse=''),'}\\hline',sep=''),
	paste('&',paste(names(genetics$summary$table$latex),collapse='&'),'\\\\\\hline',sep=''),
	paste('\\multicolumn{',N,'}{|l|}{Crime scene profiles}\\\\\\hline',sep=''))
return(text)}

#--------------------------------------------------------------------------------------------------------------------
csp.table.to.latex <- function(genetics){
	# helper function for latex.maker()
	# formats the CSP table of alleles for latex
	table.csp <- table.collapser(genetics$cspData)
	table.unc <- table.collapser(genetics$uncData)
	table <- NULL; 
	for(n in 1:genetics$nrep){
		table <- rbind(table,table.csp[n,])
		table <- rbind(table,table.unc[n,])
		}
	colnames(table) <- colnames(genetics$cspData)
	row.names(table) <- rep(c('csp','unc'),genetics$nrep)
	text = c()
	N.col <- ncol(table)
	N.row <- nrow(table)
	for(row in 1:N.row){
		text.line <- character(N.col)
		for(col in 1:N.col){
			if(as.character(table[row,col])=='')text.line[col] <- '--'
			if(as.character(table[row,col])!='')text.line[col] <- gsub(' ',',',table[row,col])
			}
		if(row%%2==1)text=c(text,paste(paste(row.names(table)[row],paste(text.line,collapse='&'),sep='&'),'\\\\',sep=''))
		if(row%%2==0)text=c(text,paste(paste(row.names(table)[row],paste(text.line,collapse='&'),sep='&'),'\\\\\\hline',sep=''))
		}
return(text)}

#--------------------------------------------------------------------------------------------------------------------
ref.table.to.latex <- function(genetics){
	# helper function for latex.maker()
	# table: Reference table, from genetics$summary$table$latex
	table <- genetics$summary$table$latex
	text = paste('\\multicolumn{',ncol(table)+1,'}{|l|}{SUMMARY:}\\\\\\hline',sep='')
	for(row in 1:nrow(table)){
		text=c(text,paste(paste(row.names(table)[row],paste(table[row,],collapse='&'),sep='&'),'\\\\\\hline',sep=''))
		}
return(text)}

latex.maker <- function(genetics,filename){
# table1: CSP table produced by allele.table() 
# table2: Reference table, from genetics$summary$table$latex
	text.1 <- latex.table.header(genetics)
	text.2 <- csp.table.to.latex(genetics)
	text.3 <- ref.table.to.latex(genetics)
	text <- c(text.1,text.2,text.3)

	file <- file(filename)
	writeLines(text,file)
	close(file)
	}

#--------------------------------------------------------------------------------------------------------------------
pack.admin.input <- function(cspFile, refFile, caseName='dummy',databaseFile=NULL, outputPath=getwd() ) {
	# Packs and verifies administrative information.
	# Documentation in man directory.
    	paths <- c(cspFile, refFile) 
	if(!is.null(databaseFile)) paths <- c(databaseFile, paths, recursive=TRUE)
	for(path in paths) {
		if(!file.exists(path))
			stop(paste(path, "does not exist."))
		else { 
			info <- file.info(path)
			if(info$isdir) stop(paste(path, "is not a file."))
      		}
    		} # loop over files.
	if(file.exists(outputPath) & !file.info(outputPath)$isdir) 
	stop(paste(outputPath, " exists and is not a directory."))
	admin <- list( caseName=caseName,
                databaseFile=databaseFile,
                cspFile=cspFile,
                refFile=refFile,
                outputPath=outputPath )
	return(admin)}

#--------------------------------------------------------------------------------------------------------------------
load.allele.database <- function(path=NULL) {
	# Loads allele database
	# Documentation is in man directory.
	if(is.null(path)) { # Load default database
	dummyEnv <- new.env()
	data('lgc-allele-freqs-wbp', package='likeLTD', envir=dummyEnv)
	return(dummyEnv[['lgc-allele-freqs-wbp']])
	}
	if(!file.exists(path)) stop(paste(path, "does not exist."))
	read.table(path, sep="\t", header=TRUE)
	}

#--------------------------------------------------------------------------------------------------------------------
unattributable.plot.maker <- function(genetics){
    with(genetics$summary$counts,
	    plot <- ggplot(data=genetics$summary$counts, aes(x=loci,y=counts,fill=status))+
		    geom_bar(stat='identity')+
		    scale_fill_grey()
		)
return(plot)}

#--------------------------------------------------------------------------------------------------------------------
table.collapser <- function(table){
	# collapses a table so that fields stored as lists become comma separated strings
	result <- array(,dim(table))
	for(row in 1:nrow(table)){
		for(locus in 1:ncol(table)){
			result[row,locus] <- paste(unlist(table[row,locus]),collapse=',')
			}}
return(result)}

#--------------------------------------------------------------------------------------------------------------------
unusual.alleles.per.table <- function(table,afreq){
	# finds unusual alleles in any specific table, (provided in original listed format)
	rare <- data.frame(locus=NULL,allele=NULL,EA1.freq=NULL,EA3.freq=NULL,EA4.freq=NULL,error=NULL)
	loci <- colnames(table); loci <- loci[loci!='queried']# ref table has a unwanted column 'queried'

	for(locus in loci){
		if(!locus%in%afreq$Marker){ # check the loci are even in the database
			frame <- data.frame(locus=locus, allele=NA,EA1.freq=NA,EA3.freq=NA,EA4.freq=NA,error='Entire locus missing from database')
            	rare <- rbind(rare, frame)
			}

		if(locus%in%afreq$Marker){ # only continue if the locus is in the database
			for(row in 1:nrow(table)){
				alleles <- unique(unlist(table[row,locus]))
				for(allele in alleles[!is.na(alleles)]){ # ignore NAs in the csv file
					condition <- afreq$Marker==locus & afreq$Allele==allele
					x <- afreq[condition,]
					if(nrow(x)==1){ # if the allele is present once in the database (should be!)
						if(x$EA1<2 | x$EA3<2 | x$EA4<2) {
            					frame <- data.frame(locus=locus, allele=allele, EA1.freq=x[,4], EA3.freq=x[,5], EA4.freq=x[,6], error=NA)
            					rare <- rbind(rare, frame)}
          						}
					if(nrow(x)==0){ # if the allele is absent from database it is probably a typo	
							frame <- data.frame(locus=locus, allele=allele, EA1.freq=NA,EA3.freq=NA,EA4.freq=NA ,error='Allele absent from database,check for typo')
            					rare <- rbind(rare, frame)
          						}
					if(nrow(x)>1){ # if the allele is more than once there is a problem with the database!	
							frame <- data.frame(locus=locus,allele=allele,EA1.freq=NA,EA3.freq=NA,EA4.freq=NA ,error='Allele present multiple times in database')
            					rare <- rbind(rare, frame)
          						}
        				}  # loop over alleles
     				} # loop over replicates
			}} # loop over loci				
return(unique(rare))}

#--------------------------------------------------------------------------------------------------------------------
unusual.alleles <- function(genetics){
	#  creates and formats a combined table of unusual alleles

	t1.tmp <- unusual.alleles.per.table(genetics$refData,genetics$afreq)
	t1 <- cbind(data.frame(source=rep('Reference profiles',nrow(t1.tmp))),t1.tmp)
	t2.tmp <- unusual.alleles.per.table(genetics$cspData,genetics$afreq)
	t2 <- cbind(data.frame(source=rep('Crime scene certain',nrow(t2.tmp))),t2.tmp)
	t3.tmp <- unusual.alleles.per.table(genetics$uncData,genetics$afreq)
	t3 <- cbind(data.frame(source=rep('Crime scene uncertain',nrow(t3.tmp))),t3.tmp)
	table <- rbind(t1,t2,t3)
	if(nrow(table)==0)table <- data.frame(status='No unusual alleles present')
return(table)}

#--------------------------------------------------------------------------------------------------------------------
csp.table.reformatter <- function(genetics){
	table.csp <- table.collapser(genetics$cspData)
	table.unc <- table.collapser(genetics$uncData)
	table <- NULL; 
	for(n in 1:genetics$nrep){
		table <- rbind(table,table.csp[n,])
		table <- rbind(table,table.unc[n,])
		}
	colnames(table) <- colnames(genetics$cspData)
	extra <- data.frame(rep=rep(1:genetics$nrep,each=2),status=rep(c("certain","uncertain"),genetics$nrep))
	combined <- cbind(extra,table)
return(combined)}

#--------------------------------------------------------------------------------------------------------------------
reference.table.reformatter <- function(genetics){
	table <- genetics$summary$table$rtf
	extra <- data.frame(profile=row.names(table))
	combined <- cbind(extra,table)
return(combined)}

#--------------------------------------------------------------------------------------------------------------------
local.likelihood.table.reformatter <- function(prosecutionHypothesis,defenceHypothesis,prosecutionResults,defenceResults){
	P <- log10(locus.likes(prosecutionHypothesis,prosecutionResults))
	D <- log10(locus.likes(defenceHypothesis,defenceResults))
	table  <- t(data.frame(Prosecution.log10=P,Defence.log10=D,Ratio.log10=(P-D),Ratio=10^(P-D)))
	extra <- data.frame(Likelihood=row.names(table))
	result <- round.3(cbind(extra,table))
return(result)}

#--------------------------------------------------------------------------------------------------------------------
hypothesis.generator <- function(genetics){
	# replicated alleles may not be explained by dropin, therefore this dictates minimum number of Us (minU)
	# maximum Us determined by whichever locus has the most unattributables (rep and unrep combined)
	# reports all U + dropin combos between minU and maxU
	table <- genetics$summary$counts
	rep <- subset(table,table$status=='replicated')$counts
	unrep <- subset(table,table$status=='unreplicated')$counts
	minU <- ceiling(max(rep)/2)
	maxU <- ceiling(max(rep+unrep)/2) 
	unknowns <- minU:maxU
	N <- length(unknowns)
	dropins <- numeric(N)
	recommendation <- character(N)
	for(n in 1:N){
		dropins[n] <- sum(pmax(0,rep+unrep-2*unknowns[n]))
		if(dropins[n]<=2)recommendation[n]<-"strongly recommended"
		if(dropins[n]==3)recommendation[n]<- "worth considering"
		if(unknowns[n]==3)recommendation[n]<-"Can only be evaluated by removing the additional U from defence"
		if(unknowns[n]>3)recommendation[n]<-"Too many U's to evaluate"
		}
	result <- data.frame(nUnknowns=unknowns, doDropin=dropins, Recommendation=recommendation)
return(result)}

#--------------------------------------------------------------------------------------------------------------------
system.info <- function(){
	date <- date()
	package.info <- sessionInfo()$otherPkgs$likeLTD
	sys.info <- Sys.info()
	Details <- t(data.frame(Details=c(date,package.info,sys.info) ))
	Details  <- gsub("\n","",Details ,fixed=T)
	Details  <- gsub("   ","",Details ,fixed=T)
	Type <- data.frame(Type=c('Date report generated:',names(package.info),names(sys.info)))
	all <- cbind(Type,Details)
return(all)}

#--------------------------------------------------------------------------------------------------------------------
filename.maker <- function(outputPath,file,type=NULL){
	# type: report type, one of 'allele' or 'results'
	if(is.null(file)){ 
		if(is.null(type)) title <- 'Report' # shouldn't remain as NULL, but just incase
		if(type=='allele')title <- 'Allele Report'
		if(type=='results')title <- 'DNA profile evaluation report'
		n <- 1
		filename <- file.path(outputPath,paste(title,n,'doc',sep='.'))
		while(file.exists(filename)){
			n <- n + 1
			filename <- file.path(outputPath,paste(title,n,'doc',sep='.'))
			}
		}
return(list(filename=filename,title=title))}

#--------------------------------------------------------------------------------------------------------------------
hyp.P <- function(genetics){
	Q <- paste(genetics$nameQ,'(Q)') 
	HP <- paste('Prosecution Hypothesis:',paste(c(Q,genetics$nameK),collapse=' + ')  )
	if(is.null(genetics$nameK))HP <- NULL # genetics object is different for allele report or final report
return(HP)}

hyp.D <- function(genetics){
	X <- 'Unknown (X)'
	HD <- paste('Defence Hypothesis:',paste(c(X,genetics$nameK),sep=' + ') )
	if(is.null(genetics$nameK))HD <- NULL # genetics object is different for allele report or final report
return(HD)}

#--------------------------------------------------------------------------------------------------------------------
locus.likes <- function(hypothesis,results,...){
	# Generate locus likelihoods from overall likelihood
	# 	hypothesis: generated by either defence.hypothesis() or 
      #     prosecution.hypothesis()
	# results: results from do.call(optim,params)
	model <- create.likelihood.vectors(hypothesis)
	arguments <- relistArguments(results$optim$bestmem, hypothesis, ...)
	likes <- do.call(model,arguments)
	likes <- likes$objectives * likes$penalties
	}

#--------------------------------------------------------------------------------------------------------------------
pack.genetics.for.allele.report <- function(admin){
	# packs together all the genetic information required for the allele.report()

	# primary objects
	cspData <- read.csp.profile(admin$cspFile)
	uncData <- read.unc.profile(admin$cspFile)
	refData <- read.known.profiles(admin$refFile)
	afreq <- load.allele.database(admin$databaseFile)

	# secondary objects
	summary <- summary.generator(refData, cspData)
	estimates <- estimate.csp(refData, cspData)
	nrep	<- nrow(cspData)
	allele.report.genetics <- list( 
	cspData = cspData, 
	uncData = uncData,
	refData = refData,
	afreq = afreq,
	summary = summary,
	estimates = estimates,
	nrep = nrep)
return(allele.report.genetics)}

#--------------------------------------------------------------------------------------------------------------------	
pack.genetics.for.output.report <- function(P.hyp,D.hyp){
	# packs together all the genetic information required for the output.report()

	# Comparison checks between P.hyp and D.hyp 
	compare.hypothesis.inputs(P.hyp,D.hyp)

	# primary objects
	cspData <- read.csp.profile(P.hyp$cspFile)
	uncData <- read.unc.profile(P.hyp$cspFile)
	refData <- read.known.profiles(P.hyp$refFile)
	afreq <- load.allele.database(P.hyp$databaseFile)

	# secondary objects
	summary <- summary.generator(refData, cspData)
	estimates <- estimate.csp(refData, cspData)
	nrep	<- nrow(cspData)
	QvK <- queried.vs.known(P.hyp$refFile)
	nameQ <- row.names(refData)[QvK]
	nameK <- row.names(refData)[!QvK]
	output.report.genetics <- list( 
	cspData = cspData, 
	uncData = uncData,
	refData = refData,
	afreq = afreq,
	summary = summary,
	estimates = estimates,
	nrep = nrep,
	nameQ = nameQ,
	nameK = nameK, 
	P.hyp = P.hyp,
	D.hyp = D.hyp)  
return(output.report.genetics)}

#--------------------------------------------------------------------------------------------------------------------
compare.hypothesis.inputs <- function(P.hyp,D.hyp){
	# Checks the input data and parameters were the same for P and D
	if(!identical(P.hyp$cspFile,D.hyp$cspFile))warning("P and D hypotheses were constructed using two different crime scene files!!")
	if(!identical(P.hyp$refFile,D.hyp$refFile))warning("P and D hypotheses were constructed using two different reference files!!")
	if(!identical(P.hyp$databaseFile,D.hyp$databaseFile))warning("P and D hypotheses were constructed using two different allele database files!!")
	if(!identical(P.hyp$outputPath,D.hyp$outputPath))warning("P and D hypotheses were given two different output paths!!")
	if(!identical(P.hyp$ethnic,D.hyp$ethnic))warning("P and D hypotheses were constructed using two different ethnic codes!!")
	if(!identical(P.hyp$ethnic,D.hyp$ethnic))warning("P and D hypotheses were constructed using two different ethnic codes!!")
	if(!identical(P.hyp$adj,D.hyp$adj))warning("P and D hypotheses were constructed using two different locus adjustment parameter vectors!!")
	if(!identical(P.hyp$fst,D.hyp$fst))warning("P and D hypotheses were constructed using two different fsts!!")
	}

#--------------------------------------------------------------------------------------------------------------------
queried.vs.known <- function(path) {
	# Reads profile from path and returns queried vs known column.
	# Args:
	#	path: Path to file with the profile. 
	raw <- read.table(path, header=T, colClasses='character', row.names=1, sep=',', quote = "\"")
	if(is.null(raw$known.queried))stop("Reference csv must contain a column 'known/queried'. This column must contain one field 'queried'. ")
	if(sum(raw$known.queried=='queried')!=1)stop("The reference csv column 'known/queried' must contain one field 'queried'. ")
	return(raw$known.queried == 'queried')
	}

#--------------------------------------------------------------------------------------------------------------------
estimate.csp <- function(refData, cspData) {
  # Estimate how well each reference profile is represented in the CSP

	nrep <- nrow(cspData)
	# Constructs the result data frame.
	result <- data.frame(array(0, c(nrow(refData), nrep+1)), row.names=rownames(refData))
	colnames(result)[1:nrep] = sapply(1:nrep, function(n) {paste('Rep', n)})
	colnames(result)[nrep+1] = 'Total' 
  
	# now add data.
	for(person in row.names(refData)){
		for(rep in 1:nrep){
			# number of alleles in common for given person and CSP replicate, across all loci
			represented <- c()
			for(locus in 1:ncol(cspData)){
				if(!is.null(cspData[rep,locus])) {
					# figure out unique alleles in reference 
					ref.alleles  <- unique(unlist(refData[person,locus+1]))# +1 ignores first column(queried)
					csp.alleles <- unique(unlist(cspData[rep,locus]))
					# figure out how many of these allele are in CSP
					represented <- c(represented, ref.alleles %in% csp.alleles)
					}
				}
			if(length(represented) > 0) 
			result[person, rep] <- 100*sum(represented)/length(represented)
			}
  		}
	if(nrep==1)  result[, nrep+1] <- result[, nrep]
	else         result[, nrep+1] <- round.1(rowSums(result)/nrep)
   
	# Reorders rows 
	result <- round.1(result[order(result[, nrep+1], decreasing=T), ])
return(result)}

#--------------------------------------------------------------------------------------------------------------------
estimates.reformatter <- function(genetics){
	table <- genetics$estimates
	extra <- data.frame(Contributor=row.names(table))
	result <- cbind(extra,table)
return(result)}

#--------------------------------------------------------------------------------------------------------------------
summary.generator <- function(refData, cspData){

	# summary table for Q and K, showing which of their alleles are replicated, unreplicated, or absent
	table <- as.data.frame(array(,c(nrow(refData)+1,ncol(cspData))))
	colnames(table) <- colnames(cspData)
	row.names(table) <- c(row.names(refData),'Unattributable')

	# two identical tables, will be slightly different formats for rtf and latex
	table.rtf <- table.latex <- table

	# count of unattributable alleles, to establish if they are replicated or not
	rep.counts <- unrep.counts <- c()

	# generate the results 
	for(locus in colnames(cspData)){
		# for unattributable alleles are rep, unrep, absent
		cspAlleles <- unlist(cspData[,locus])
		cspAlleles <- cspAlleles[!is.na(cspAlleles)] # remove NAs
		refAlleles <- refData[,locus][[1]]
		unattributableAlleles <- cspAlleles[!cspAlleles%in%refAlleles]
		table.rtf['Unattributable',locus] <- summary.helper(unattributableAlleles,cspAlleles)$rtf
		table.latex['Unattributable',locus] <- summary.helper(unattributableAlleles,cspAlleles)$latex

		# for unattributable alleles calculate how many are replicated /unreplicated
		rep.counts <- c(rep.counts, summary.helper(unattributableAlleles,cspAlleles)$rep )
		unrep.counts <- c(unrep.counts, summary.helper(unattributableAlleles,cspAlleles)$unrep )

		# for each contributor calculate which alleles are rep, unrep, absent
		for(name in row.names(refData)){
			refAlleles <- refData[name,locus][[1]]
			table.rtf[name,locus] <- summary.helper(refAlleles,cspAlleles)$rtf
			table.latex[name,locus] <- summary.helper(refAlleles,cspAlleles)$latex
			}}	

		# reformat the counts table (c.)
		c.rep  <- data.frame(loci=colnames(cspData), counts=rep.counts, status='replicated')
		c.unrep <- data.frame(loci=colnames(cspData), counts=unrep.counts, status='unreplicated')
		counts <- rbind(c.rep,c.unrep)
		table <- list(rtf=table.rtf,latex=table.latex)		
		summary <- list(table=table,counts=counts)
return(summary)}

#--------------------------------------------------------------------------------------------------------------------
summary.helper <- function(refAlleles,cspAlleles){
	# intricate and irritating operations required to check each allele 
	rep <- unrep <- absent <- NULL
	for(allele in unique(refAlleles)){
		condition <- sum(cspAlleles%in%allele)
		if(condition>1)rep <- c(rep,allele)
		if(condition==1)unrep <- c(unrep,allele)
		if(condition==0)absent <- c(absent,allele)
		}

	# copy objects to avoid cascading changes
	rep.rtf <- rep.latex <- rep
	unrep.rtf <- unrep.latex <- unrep
	absent.rtf <- absent.latex <- absent
	
	# separate by commas, and apply rtf, such that replicated=bold, and unreplicated=italic
	if(!is.null(rep.rtf))rep.rtf <- paste('{\\b ',paste(rep.rtf,collapse=','),'}',sep='')
	if(!is.null(unrep.rtf))unrep.rtf <- paste(unrep.rtf,collapse=',')
	if(!is.null(absent.rtf))absent.rtf <- paste('{\\i ',paste(absent.rtf,collapse=','),'}',sep='')
	rtf <- paste(c(rep.rtf,unrep.rtf,absent.rtf),collapse=',')

	# separate by commas, and apply latex, such that replicated=bold, and unreplicated=italic
	if(!is.null(rep.latex))rep.latex <- paste('{\\bf',paste(rep.latex,collapse=','),'}',sep='')
	if(!is.null(unrep.latex))unrep.latex <- paste('{\\em',paste(unrep.latex,collapse=','),'}',sep='')
	if(!is.null(absent.latex))absent.latex <- paste('{\\fx',paste(absent.latex,collapse=','),'}',sep='')
	latex <- paste(c(rep.latex,unrep.latex,absent.latex),collapse=',')

return(list(rtf=rtf,latex=latex,rep=length(rep),unrep=length(unrep)))}

#--------------------------------------------------------------------------------------------------------------------
overall.likelihood.table.reformatter <- function(prosecutionResults,defenceResults){
	P <- -prosecutionResults$optim$bestval
	D <- -defenceResults$optim$bestval
	table <- cbind(
		round.3(data.frame(Prosecution.log10=P,Defence.log10=D,Ratio.log10=P-D)),
		round.0(data.frame(Ratio=10^(P-D)))
		)
	table  <- t(table); colnames(table) <- 'estimate'
	extra <- data.frame(calculation=row.names(table))
	result <- cbind(extra,table)
return(result)}

#--------------------------------------------------------------------------------------------------------------------
rcontConvert <- function(refIndiv,rcont){
	# Convert rcont to full rcont (including ref individual)
	# 	refIndiv: reference individual specified in args
	#	rcont: rcont parameters from do.call(optim,params)
	if(refIndiv == 1) rcont = c(1, rcont)
      else if(refIndiv > length(rcont)) rcont = c(rcont, 1)
      else rcont = c(rcont[1:refIndiv-1], 1, rcont[refIndiv:length(rcont)])
return(rcont)}

#--------------------------------------------------------------------------------------------------------------------
calc.dropout = function(results, hypothesis){ 
	# Calculates dropout rates for every contributor subject to dropout and for every replicate
	# 	hypothesis: generated by either defence.hypothesis() or  prosecution.hypothesis()
	#	results: results from do.call(optim,params)

	# Number of contributors subject to dropout + number of unknowns + 1 (dropin)
	N <- nrow(hypothesis$dropoutProfs) + hypothesis$nUnknowns + 1
	nrep <- nrow(hypothesis$cspProfile)
	do <- results$optim$bestmem[grep("dropout",names(results$optim$bestmem))]
	rcont <- results$optim$bestmem[grep("rcont",names(results$optim$bestmem))]
	rcont <- rcontConvert(hypothesis$refIndiv,rcont)
	BB <- results$optim$bestmem[grep("power",names(results$optim$bestmem))]
	drout <- matrix(0,N-1,nrep)
	if(N>1) for(x in 1:(N-1)) for(z in 1:nrep) drout[x,z] <- do[z]/(do[z]+rcont[x]^-BB*(1-do[z]))
return(drout)}

#--------------------------------------------------------------------------------------------------------------------
dropDeg <- function(hypothesis,results,genetics){
	# Output tables for dropout and degradation
	# 	hypothesis: generated by either defence.hypothesis() or prosecution.hypothesis() 
	#	results: results from do.call(optim,params)

	dropoutsLogical <- determine.dropout(genetics$refData,genetics$cspData)
	Qdrop <- dropoutsLogical[names(dropoutsLogical)==genetics$nameQ]
	knownDropoutsLogical <- dropoutsLogical[names(dropoutsLogical)!=rownames(hypothesis$queriedProfile)]
	# Number of Known contributors with No Dropout
	Nknd <- length(which(!knownDropoutsLogical)) 

	# create the row names, are arranged into the correct order: K, Q/X, U
	names.Dropout <- names(which(knownDropoutsLogical))
	names.NoDropout <- names(which(!knownDropoutsLogical))
	nameK <- c(names.NoDropout,names.Dropout)	
	Names=c()
	if(length(nameK)>0)for (n in 1:length(nameK)) Names[n]=paste(nameK[n],' (K',n,')',sep='')
	Names[length(c(genetics$nameQ,nameK))] = paste(genetics$nameQ,'(Q)')
	nU <- hypothesis$nUnknowns
	if(hypothesis$hypothesis=="defence"){
		Names[length(c(genetics$nameQ,nameK))] = 'X' 
		nU <- nU -1 # it already contains an extra one for X
		}
	if(nU>0)for(n in 1:nU)Names <- c(Names,paste('U',n,sep=''))

	# column names for the table
	runNames = c();for(rName in 1:genetics$nrep)runNames[rName]=paste('Dropout',paste('(Rep ',rName,')',sep=''))

	# dropout values
	h <- h1 <- round.3(calc.dropout(results, hypothesis))	
	# degradation values
	d <- d1 <- 10^results$optim$bestmem[grep("degradation",names(results$optim$bestmem))]
	# under pros, if Q is not subject to dropout, an extra 0 needs to be added to both dropout and degradation for Q
	if(hypothesis$hypothesis=="prosecution"){
		if(Qdrop==F){
			h = rbind(h[0:nrow(hypothesis$dropoutProfs),,drop=F],0)
			d = c(d[0:nrow(hypothesis$dropoutProfs)],0)
			if(hypothesis$nUnknowns>0){
				h = rbind(h,h1[(nrow(hypothesis$dropoutProfs)+1):length(d1),,drop=F])
				d = c(d,d1[(nrow(hypothesis$dropoutProfs)+1):length(d1)])
				}
			}
		}

	# Dropout is assumed zero for fully represented profiles, therefore suffixed on the dropout table
	if(Nknd>0) for(n in 1:Nknd){
		h = rbind(0,h)
		d = c(0,d)
		}
	Dropout = round.3(data.frame(h,d))
	colnames(Dropout)= c(runNames[1:genetics$nrep],'Degradation (overall)')
	row.names(Dropout) = Names
return(Dropout)}

#--------------------------------------------------------------------------------------------------------------------
overall.dropout.table.reformatter <- function(prosecutionHypothesis,defenceHypothesis,prosecutionResults,defenceResults,genetics){
	P.dropDeg <- dropDeg(prosecutionHypothesis,prosecutionResults,genetics)
	P.extra <- data.frame(hypothesis=rep('Prosecution',nrow(P.dropDeg)),contributor=rownames(P.dropDeg))
	P.combined <- cbind(P.extra,P.dropDeg)
	D.dropDeg <- dropDeg(defenceHypothesis,defenceResults,genetics)
	D.extra <- data.frame(hypothesis=rep('Defence',nrow(D.dropDeg)),contributor=rownames(D.dropDeg))
	D.combined <- cbind(D.extra,D.dropDeg)
	combined <- rbind(P.combined,D.combined)
return(combined)}

#--------------------------------------------------------------------------------------------------------------------
overall.dropin.table.reformatter <- function(prosecutionResults,defenceResults){
	P.dropin <- prosecutionResults$optim$bestmem["dropin"]
	D.dropin <- defenceResults$optim$bestmem["dropin"]
	table <- round.3(data.frame(hypothesis=c('Prosecution','Defence'),dropin=c(P.dropin,D.dropin)))
return(table)}

#--------------------------------------------------------------------------------------------------------------------
optimised.parameter.table.reformatter <- function(hypothesis,result){
	table <- t(rbind(result$optim$bestmem,result$member$upper,result$member$lower))
	extra <- data.frame(parameter=rownames(table))
	combined <- round.3(cbind(extra,table))
	colnames(combined) <- c('parameter','estimate','upper bound','lower bound')
return(combined)}

#--------------------------------------------------------------------------------------------------------------------
chosen.parameter.table.reformatter <- function(prosecutionHypothesis){
	keep <- c(7,8,9,11,12,13,14,15)
	table <- as.data.frame(unlist(prosecutionHypothesis[keep]))
	extra <- data.frame(Parameter= rownames(table))
	combined <- cbind(extra,table)
	colnames(combined) <- c('Parameter','User input')
return(combined)}

#--------------------------------------------------------------------------------------------------------------------
ideal <- function(hypothesis,rr){
	# Calculates idealised likelihood assuming Q is perfect match
	# Parameters:
	# 	hypothesis: generated by either defence.hypothesis() or prosecution.hypothesis(), although rr only makes sense under defence
	#	rr: relatedness arguments from args
	ideal.match <- 1
	for(j in 1:ncol(hypothesis$cspProfile)){
		af = hypothesis$alleleDb[j][[1]]
		kn = hypothesis$queriedProfile[,j][[1]]
		p1 = af[row.names(af)==kn[1],1]
		p2 = af[row.names(af)==kn[2],1]
		ideal.match = ideal.match/(rr[2] + rr[1]*(p1+p2)/2 + (1-sum(rr))*p1*p2*(1+(kn[1]!=kn[2])))
		}
	result <- data.frame(calculation =c('likelihood ratio','Log10 likelihood ratio'),estimate=c(round.0(ideal.match),round.3(log10(ideal.match))))
return(result)}

#--------------------------------------------------------------------------------------------------------------------
# a few odds and sods to simplify the reports
line <- paste(rep('-',95),collapse='-') # creates a straight line
spacer <- function(doc,n=1) for(x in 1:n)addNewLine(doc) # adds blank lines
fs0 <- 26 # font size for header (main)
fs1 <- 20 # font size for header (sub1)
fs2 <- 15 # font size for header (sub2)

#--------------------------------------------------------------------------------------------------------------------
common.report.section <- function(names,genetics){
	# objects common to both the allele report and the final output report are done once here, for consistency, and saves repeating code

	# Create a new Docx. 
	doc <- RTF(names$filename, width=11,height=8.5,omi=c(1,1,1,1))

	addParagraph(doc, line)
	spacer(doc,3)
	addHeader( doc, title=names$title, subtitle=names$subtitle, font.size=fs0 )
	addHeader( doc, hyp.P(genetics), font.size=fs2 )
	addHeader( doc, hyp.D(genetics), font.size=fs2 )
	addParagraph(doc, line)
	addPageBreak(doc, width=11,height=8.5,omi=c(1,1,1,1) )

	addTOC(doc)
	addPageBreak(doc, width=11,height=8.5,omi=c(1,1,1,1) )

	addHeader(doc, "Data provided by forensic scientist", TOC.level=1,font.size=fs1)
	addHeader(doc, "Crime scene profiles (CSP)",TOC.level=2,font.size=fs2)
	addTable(doc, csp.table.reformatter(genetics),col.justify='C', header.col.justify='C',font.size=8)
	spacer(doc,3)

	addHeader(doc, "Reference profiles", TOC.level=2, font.size=fs2 )
	addTable(doc, reference.table.reformatter(genetics), col.justify='C', header.col.justify='C',font.size=8)
	spacer(doc,1)
	addParagraph( doc, "Assessed using the 'certain' allelic designations only." )
	addParagraph(doc, "{\\b replicated alleles}" )
	addParagraph(doc, "unreplicated alleles" )
	addParagraph(doc, "{\\i absent alleles}" )
	addPageBreak( doc, width=11,height=8.5,omi=c(1,1,1,1))

	addHeader(doc, "Summary", TOC.level=1,font.size=fs1)
	addHeader(doc, "Unattributable alleles", TOC.level=2, font.size=fs2)
	addPlot( doc, plot.fun = print, x = unattributable.plot.maker(genetics) , width = 10, height = 3.5)
	addParagraph( doc, "The number of 'certain' alleles that cannot be attributed to a known profile.")
	spacer(doc,3)

	addHeader(doc, "Unusual alleles", TOC.level=2, font.size=fs2 )
	addTable(doc, unusual.alleles(genetics), col.justify='C', header.col.justify='C')
	spacer(doc,3)

	addHeader(doc, "Approximate representation", TOC.level=2, font.size=fs2)
	addTable(doc, estimates.reformatter(genetics), col.justify='C', header.col.justify='C')
	spacer(doc,1)
	addParagraph( doc, "The fraction of an individual's alleles (as a percentage) that have been designated as 'certain' alleles in each replicate. This estimate is not used by likeLTD, and is intended to assist informal assessments of possible known contributors to the CSP. A more formal approach is to do a likeLTD run to compute the likelihood ratio (LR) for that individual contributor.")
	spacer(doc,3)

return(doc)}

#--------------------------------------------------------------------------------------------------------------------
allele.report <- function(admin,file=NULL){

 	# admin: List containing administration data, as packed by pack.admin.input()
	# file: defaults to creating its own sequential file names (to avoid overwriting)

	# create genetics information
	genetics <- pack.genetics.for.allele.report(admin)
 	
	# Latex output
	latex.maker(genetics,(paste(admin$outputPath,"/",file," table.tex",sep="")))

	names <- filename.maker(admin$outputPath,file,type='allele')
	names$subtitle <- admin$caseName

	doc <- common.report.section(names,genetics)

	addHeader(doc, "Suggested parameter values", , TOC.level=2, font.size=fs2)
	addTable(doc, hypothesis.generator(genetics), col.justify='C', header.col.justify='C')
	spacer(doc,1)
	addParagraph( doc, "Recommended values for 'nUnknowns', choose from 0,1 or 2 (likeLTD automatically adds and additional unknown X to the defence hypothesis in place of the queried profile Q).")
	addParagraph( doc, "Recommended values for 'doDropin', choose from T or F.")
	addParagraph( doc, "All the attributable alleles must either come from an unknown or dropin.")
	spacer(doc,3)

	addHeader(doc, "System information", TOC.level=1,font.size=fs1)
	addTable(doc,  system.info(), col.justify='L', header.col.justify='L')

done(doc)}

#--------------------------------------------------------------------------------------------------------------------
output.report <- function(prosecutionHypothesis,defenceHypothesis,prosecutionResults,defenceResults,file=NULL){

 	# prosecutionHypothesis: generated by prosecution.hypothesis()
 	# defenceHypothesis: generated by defence.hypothesis()
	# prosecutionResults: results from do.call(optim, prosecutionParams)
  	# defenceResults: results from do.call(optim, defenceParams)
	# file: defaults to creating its own sequential file names (to avoid overwriting)

	# create genetics information 
	genetics <- pack.genetics.for.output.report(prosecutionHypothesis,defenceHypothesis)
	names <- filename.maker(prosecutionHypothesis$outputPath,file,type='results')
	names$subtitle <- prosecutionHypothesis$caseName

	doc <- common.report.section(names,genetics)

	addHeader(doc, "Likelihoods at each locus", TOC.level=2, font.size=fs2)
	addTable(doc, local.likelihood.table.reformatter(prosecutionHypothesis,defenceHypothesis,prosecutionResults,defenceResults) ,col.justify='C', header.col.justify='C',font.size=8)
	spacer(doc,3)

	addHeader(doc, "Overall Likelihood", TOC.level=2, font.size=fs2)
	addTable(doc, overall.likelihood.table.reformatter(prosecutionResults,defenceResults) ,col.justify='C', header.col.justify='C')
	spacer(doc,3)

	addHeader(doc, "Theoretical maximum LR", TOC.level=2, font.size=fs2)
	addTable(doc, ideal(defenceHypothesis,defenceHypothesis$relatedness), col.justify='C', header.col.justify='C')
	spacer(doc,3)

	addHeader(doc, "Dropout and degradation parameter estimates", TOC.level=2, font.size=fs2)
	addTable(doc, overall.dropout.table.reformatter(prosecutionHypothesis,defenceHypothesis,prosecutionResults,defenceResults,genetics), col.justify='C', header.col.justify='C')
	spacer(doc,3)

	addHeader(doc, "Dropin parameter estimates", TOC.level=2, font.size=fs2)
	addTable(doc, overall.dropin.table.reformatter(prosecutionResults,defenceResults), col.justify='C', header.col.justify='C')
	spacer(doc,3)

	addHeader(doc, "User defined parameters", TOC.level=1, font.size=fs1)
	addTable(doc, chosen.parameter.table.reformatter(prosecutionHypothesis), col.justify='L', header.col.justify='L')
	spacer(doc,3)

	addHeader(doc, "Optimised parameters", TOC.level=1, font.size=fs1)
	addHeader(doc, "Prosecution parameters", TOC.level=2, font.size=fs2)
	addTable(doc, optimised.parameter.table.reformatter(prosecutionHypothesis,prosecutionResults), col.justify='L', header.col.justify='L')
	spacer(doc,3)

	addHeader(doc, "Defence parameters", TOC.level=2, font.size=fs2)
	addTable(doc, optimised.parameter.table.reformatter(defenceHypothesis,defenceResults), col.justify='L', header.col.justify='L')
	spacer(doc,3)

	addHeader(doc, "System information", TOC.level=1,font.size=fs1)
	addTable(doc,  system.info(), col.justify='L', header.col.justify='L')

done(doc)}
#--------------------------------------------------------------------------------------------------------------------



