% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mice.binarize.R
\name{mice.binarize}
\alias{mice.binarize}
\title{Binarize Factor Columns in Data Frames}
\usage{
mice.binarize(data, include_ordered = TRUE, include_observed = FALSE,
  cols = NULL, pred_matrix = (1 - diag(1, ncol(data))))
}
\arguments{
\item{data}{Matrix or data frame that contains factor columns which we want to
convert into an equivalent set of binary columns.}

\item{include_ordered}{Logical variable indicating whether we also want to
transform ordered factors. Default is \code{TRUE}.}

\item{include_observed}{Logical variable indicating whether we also want to
transform factor columns in which all values are observed. Default is
\code{FALSE}.}

\item{cols}{Numerical vector corresponding to the indices or character vector
corresponding to the names of factor columns which we want to transform. By
default, its value is \code{NULL}, indicating that the algorithm
automatically identifies all factor columns that are to be binarized. If
however the user specifies its value, the function exclusively transforms
the specified columns, ignoring the values of the other optional parameters.}

\item{pred_matrix}{A custom predictor matrix relating to input \code{data},
which will get transformed into the format that fits to the binarized output
data frame. The result of this transformation will be stored in the
\code{pred_matrix}-element of the output and should then be used as the
\code{predictorMatrix} parameter in \code{mice()} to ensure that binary
columns relating to the same factor column in the original data do not
predict each other, yielding cleaner imputation models. If not specified,
the default is the massive imputation predictor matrix.}
}
\value{
List containing the following three elements:
\describe{
 \item{data}{ The binarized data frame.}
 \item{par_list}{A list containing the original data frame as well as some
 parameters with further information on the transformation. This list is
 needed to retransform the (possibly imputed) data at later stager via the
 \code{mice.factorize()} function, and should not be edited by the user
 under any circumstance. Next to the original data, the most notable
 element of this list would be "dummy_cols", which itself is a list of the
 column tuples that correspond to the transformed factor column from the
 original data set, and therefore works perfectly as input for
 \code{mice.post.matching()} (cf. examples below).}
 \item{pred_matrix}{Transformed version of input \code{pred_matrix} that
 should be used as the input argument \code{predictorMatrix} of
 \code{mice()}.}
}
}
\description{
This function replaces factor columns in data frames in-place by a set of
binary columns which represent the so-called one-hot encoding of this factor.
More precisely, a column of a factor with \code{n} levels will be transformed
into a set of \code{n} binary columns, each representing exactly one category
of the original factor. Hence, the value \code{1} occurs in a column if and
only if the original factor had the value corresponding to that column. \cr
Further, this function also returns a predictor matrix that fits to the
binarized data frame and, when used as input parameter in \code{mice()},
ensures that binary columns that relate to the same factor do not predict each
other.
}
\examples{

#------------------------------------------------------------------------------
# first set of examples illustrating basic functionality
#------------------------------------------------------------------------------

# binarize all factor columns in boys_data that contain NAs
boys_bin <- mice.binarize(boys_data)


# binarize only column 'gen' in boys_data
boys_bin <- mice.binarize(boys_data, cols = c("gen"))

# read out binarized data
boys_bin$data

\dontrun{
#------------------------------------------------------------------------------
# this example illustrates the combined functionalities of mice.binarize,
# mice.factorize and mice.post.matching on the dataset 'boys' from mice, which
# yields different imputations on the factor columns 'gen', 'phb' and 'reg'
# than mice() would output
#------------------------------------------------------------------------------

# binarize all factor columns in boys_data that contain NAs
boys_bin <- mice.binarize(boys)

# run mice on binarized data, note that we need to use boys_bin$data to grab
# the actual binarized data and that we use the output predictor matrix
# boys_bin$pred_matrix which is recommended for obtaining better imputation
# models
mids_boys <- mice(boys_bin$data, predictorMatrix = boys_bin$pred_matrix)

# it is very likely that mice imputed multiple ones among one set of dummy
# variables, so we need to post-process
post_boys <- mice.post.matching(mids_boys, distmetric = "residual")

# now we can safely retransform to the original data, with non-binarized
# imputations
res_boys <- mice.factorize(post_boys$midsobj, boys_bin$par_list)

# analyze the distribution of imputed variables, e.g. of the column 'gen',
# using the mice version of with()
with(res_boys, table(gen))
}


}
\seealso{
\code{\link[miceExt]{mice.factorize}},
 \code{\link[miceExt]{mice.post.matching}}, \code{\link[mice]{mice}}
}
\author{
Tobias Schumacher, Philipp Gaffert
}
