#' @export convert2sbml
#' @author  Created by: Tomas Radivoyevitch - Modified and Maintained by: Daniel Camilo Osorio <dcosorioh@unal.edu.co>
#  Bioinformatics and Systems Biology Lab      | Universidad Nacional de Colombia
#  Experimental and Computational Biochemistry | Pontificia Universidad Javeriana
#' @title Write a model in a SBML format.
#' @description This function converts a data.frame to a valid SBML file. The Systems Biology Markup Language (SBML) is a representation format, based on XML, for communicating and storing computational models of biological processes.
#' More Info: Encyclopedia of Systems Biology Dubitzky, W., Wolkenhauer, O., Yokota, H., Cho, K.-H. (Eds.) SBML, pp2057-2062 Springer 2013.
#' @details This function takes a data.frame as input and convert it to a valid sbmlR object, then the object is written into the SBML output file.
#' @param data A data.frame with the following mandatory colnames: \itemize{
#' \item \code{"ID":} A list of single character strings containing the reaction abbreviations, Entries in the field abbreviation are used as reaction ids, so they must be unique.
#' \item \code{"REACTION":} A set of stoichiometric reaction with the following format: \code{"H2O[c] + Urea-1-carboxylate[c] <=> 2 CO2[c] + 2 NH3[c]"} Where arrows and plus signs are surrounded by a "space character".
#' It is also expected that stoichiometry coefficients are surrounded by spaces, (nothe the "2" before the CO2[c] or the NH3[c]).
#' It also expects arrows to be in the form "\code{=>}" or "\code{<=>}".
#' Meaning that arrows like "\code{==>}", "\code{<==>}", "\code{-->}" or "\code{->}" will not be parsed and will lead to errors.,
#' \item \code{"GPR":} A set of genes joined by boolean operators as AND or OR, rules may be nested by parenthesis. (optional: column can be empty),
#' \item \code{"LOWER.BOUND":} A list of numeric values containing the lower bounds of the reaction rates.
#' If not set, zero is used for an irreversible reaction and 1000 for a reversible reaction. (optional: column can be empty),
#' \item \code{"UPPER.BOUND":} A list of numeric values containing the upper bounds of the reaction rates.
#' If not set, 1000 is used by default. (optional: column can be empty),
#' \item \code{"OBJECTIVE":} A list of numeric values containing objective values for each reaction (optional: column can be empty).
#' }
#' @param outfile A writable path for the output 'SBML' file to be generated.
#' @param optimizedFor A character string specifying the toolbox for which the SBML file must be optimized; must be one of \code{'sybil'}, \code{'RAVEN'} or \code{'COBRA'}. A \code{'sybil'} optimized SBML file is generated by default.
#' @param boundary A character string specifying the compartment to be used as boundary
#' @return A SBML file.
#' @examples
#' \dontrun{
#' # Loading a CSV file
#' glycolysis <- read.csv2(system.file("extdata", "glycolysisKEGG.csv", package = "minval"))
#'
#' # Data structure
#' head(glycolysis)
#'
#' # Writing SBML file
#' ## Optimized for the 'sybil' package
#' convert2sbml(
#'  data = glycolysis, 
#'  outfile = "glycolysis_sybil.xml", 
#'  optimizedFor = "sybil", 
#'  boundary = "b"
#'  )
#'  
#' ## Optimized for the 'COBRA' toolbox
#' convert2sbml(
#'  data = glycolysis, 
#'  outfile = "glycolysis_cobra.xml", 
#'  optimizedFor = "COBRA", 
#'  boundary = "b"
#'  )
#'  
#' ## Optimized for the 'RAVEN' toolbox
#' convert2sbml(
#'  data = glycolysis, 
#'  outfile = "glycolysis_raven.xml", 
#'  optimizedFor = "RAVEN", 
#'  boundary = "b"
#'  )
#' }
#' @seealso Original 'saveSBML': https://www.bioconductor.org/packages/release/bioc/html/SBMLR.html
#' @keywords Convert SBML Metabolic Reconstruction
#'
convert2sbml<-function(data,outfile,optimizedFor="sybil",boundary="b"){
  optimizedFor <- match.arg(optimizedFor,c("sybil","COBRA","RAVEN"),several.ok = FALSE)
  # Creating SBMLR model
  model <- convert2sbmlR(data,optimizedFor,boundary)
  # Writing model
  # This function is a modified copy of saveSBML function included in SBMLR package.
  # Original 'saveSBML' function was writed by Tomas Radivoyevitch
  # Please see: https://www.bioconductor.org/packages/release/bioc/html/SBMLR.html
  writeSBML <- function(model,outfile) {
    fid <- file(outfile, "w")
    sbml=model[["sbml"]]
    id=model[["id"]]
    notes=model[["notes"]]
    htmlNotes=model[["htmlNotes"]]
    compartments=model[["compartments"]]
    species=model[["species"]]
    globalParameters=model[["globalParameters"]]
    rules=model[["rules"]]
    reactions=model[["reactions"]]
    units=model[["units"]]

    nNotes=length(notes)
    nCompartments=length(compartments)
    nReactions=length(reactions)
    nSpecies=length(species)
    nGlobalParameters=length(globalParameters)
    nRules=length(rules)
    nUnits=length(units)

    cat("<?xml version=\"1.0\" encoding=\"UTF-8\"?>", file=fid, sep="\n")
    cat("<sbml xmlns=\"http://www.sbml.org/sbml/level2\" level=\"2\" version=\"1\">", file=fid, sep="\n")
    cat(sprintf("<model id=\"%s\" name=\"%s\">",id,id), file=fid, sep="\n")
    if(nNotes>0){
      cat("<notes>", file=fid, sep="\n")
      cat(" <body xmlns=\"http://www.w3.org/1999/xhtml\">", file=fid, sep="\n")
      sapply(1:nNotes, function(i){cat(sprintf("   <p> %s  </p>",notes[[i]]), file=fid, sep="\n")})
      cat(" </body>", file=fid, sep="\n")
      cat("</notes>", file=fid, sep="\n")
    }
    if(nCompartments>0){
      cat("<listOfCompartments>", file=fid, sep="\n")
      sapply(1:nCompartments, function(i){cat(sprintf("   <compartment id=\"%s\"  name=\"%s\"/>", compartments[[i]][["id"]],compartments[[i]][["name"]]), file=fid, sep="\n")})
      cat("</listOfCompartments>", file=fid, sep="\n")
    }
    if(nSpecies>0){
      cat("<listOfSpecies>", file=fid, sep="\n")
      sapply(1:nSpecies,function(i){cat(sprintf("   <species id=\"%s\" name=\"%s\" compartment=\"%s\" boundaryCondition=\"%s\"/>",species[[i]][["id"]],species[[i]][["name"]],species[[i]][["compartment"]],ifelse(grepl(boundary,species[[i]][["compartment"]]),"true","false")), file=fid, sep="\n")})
      cat("</listOfSpecies>", file=fid, sep="\n")
    }
    cat("<listOfReactions>", file=fid, sep="\n")
    sapply(1:nReactions, function(i){
      if (is.null(reactions[[i]][["reversible"]])){ print("Internal SBMLR object should have reverse flag set")} else{
        cat(sprintf("  <reaction id=\"%s\"  reversible=\"%s\">",reactions[[i]][["id"]], ifelse(reactions[[i]][["reversible"]],"true","false")), file=fid, sep="\n")
      }
      gpr = reactions[[i]][["notes"]]
      if(!is.na(gpr[["GPR"]])){
        cat(sprintf("    <notes>"),file=fid,sep="\n")
        cat(sprintf("      <html xmlns=\"http://www.w3.org/1999/xhtml\"><p>GENE_ASSOCIATION: %s</p></html>",gpr[["GPR"]]),file=fid,sep="\n")
        cat(sprintf("    </notes>"),file=fid,sep="\n")
      }
      reactants=reactions[[i]][["reactants"]]
      if (!is.null(reactants[[1]])) {
        cat("    <listOfReactants>", file=fid, sep="\n")
        sapply(1:length(reactants[["reactants"]]),function(j){ cat(sprintf("      <speciesReference species=\"%s\" stoichiometry=\"%s\"/>", reactants[["reactants"]][[j]],reactants[["stoichiometry"]][[j]]), file=fid, sep="\n")})
        cat("    </listOfReactants>", file=fid, sep="\n")
      }

      # Just switched the order of these two blocks to fix the errors
      products=reactions[[i]][["products"]]
      if (!is.null(products[[1]])) {
        cat("    <listOfProducts>", file=fid, sep="\n")
        sapply(1:length(products[["products"]]), function(j){ cat(sprintf("      <speciesReference species=\"%s\" stoichiometry=\"%s\"/>", products[["products"]][[j]],products[["stoichiometry"]][[j]]), file=fid, sep="\n")})
        cat("    </listOfProducts>", file=fid, sep="\n")
      }

      cat("  <kineticLaw>", file=fid, sep="\n")
      cat("    <math xmlns=\"http://www.w3.org/1998/Math/MathML\">", file=fid, sep="\n")
      # cat(reactions[[i]]$mathmlLaw)
      ml=XML::saveXML(reactions[[i]]$mathmlLaw,prefix=NULL,file=fid) # annoying warnings were coming from here without file=fid
      cat("    </math>", file=fid, sep="\n")

      parameters=reactions[[i]][["parameters"]]
      nlocalParameters = length(parameters)
      if(nlocalParameters > 0){
        cat("    <listOfParameters>", file=fid, sep="\n")
        sapply(1:nlocalParameters,function(j){ cat(sprintf("      <parameter id=\"%s\" value=\"%g\"/>", names(parameters)[j],parameters[[j]]), file=fid, sep="\n")})
        cat("    </listOfParameters>", file=fid, sep="\n")
        cat("    </kineticLaw>", file=fid, sep="\n")
        cat("  </reaction>", file=fid, sep="\n")}})

    cat("</listOfReactions>", file=fid, sep="\n")
    cat("</model>", file=fid, sep="\n")
    cat("</sbml>", file=fid, sep="\n")
    close(fid)
  }
  writeSBML(model,outfile)
}
