#' Create learner object.
#'
#' For a classification learner the \code{predict.type} can be set
#' to \dQuote{prob} to predict probabilities and the maximum
#' value selects the label. The threshold used to assign the label can later be changed using the
#' \code{\link{setThreshold}} function.
#'
#' @template arg_lrncl
#' @param id [\code{character(1)}]\cr
#'   Id string for object. Used to display object.
#'   Default is \code{cl}.
#' @param predict.type [\code{character(1)}]\cr
#'   Classification: \dQuote{response} (= labels) or \dQuote{prob} (= probabilities and labels by selecting the ones with maximal probability).
#'   Regression: \dQuote{response} (= mean response) or \dQuote{se} (= standard errors and mean response).
#'   Survival: \dQuote{response} (= some sort of orderable risk) or \dQuote{prob} (= time dependent probabilities).
#'   Default is \dQuote{response}.
#' @param ... [any]\cr
#'   Optional named (hyper)parameters.
#'   Alternatively these can be given using the \code{par.vals} argument.
#' @param par.vals [\code{list}]\cr
#'   Optional list of named (hyper)parameters. The arguments in
#'   \code{...} take precedence over values in this list. We strongly
#'   encourage you to use one or the other to pass (hyper)parameters
#'   to the learner but not both.
#' @return [\code{\link{Learner}}].
#' @export
#' @aliases Learner
#' @seealso [\code{\link{resample}}], [\code{\link{predict.WrappedModel}}]
#' @examples
#' makeLearner("classif.rpart")
#' makeLearner("classif.lda", predict.type = "prob")
#' lrn = makeLearner("classif.lda", method = "t", nu = 10)
#' print(lrn$par.vals)
makeLearner = function(cl, id = cl, predict.type = "response", ..., par.vals = list()) {
  assertString(cl)
  constructor = getS3method("makeRLearner", class = cl)
  wl = do.call(constructor, list())

  if (!missing(id)) {
    assertString(id)
    wl$id = id
  }
  assertList(par.vals)
  if (!nzchar(cl))
    stop("Cannot create learner from empty string!")
  if (!inherits(wl, "RLearner"))
    stop("Learner must be a basic RLearner!")
  wl = setHyperPars(learner = wl, ..., par.vals = par.vals)
  wl = setPredictType(learner = wl, predict.type = predict.type)
  return(wl)
}

#' @export
print.Learner = function(x, ...) {
  # FIXME: the "old" printer was a little bit more informative...
  cat(
    "Learner ", x$id, " from package ", collapse(x$package), "\n",
    "Type: ", x$type, "\n",
    "Class: ", class(x)[1L], "\n",
    "Properties: ", collapse(x$properties), "\n",
    "Predict-Type: ", x$predict.type, "\n",
    "Hyperparameters: ", getHyperParsString(x), "\n\n",
    sep =""
  )
}
