% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelplot.R
\name{modelplot}
\alias{modelplot}
\title{Plot model coefficients using points or point-ranges}
\usage{
modelplot(
  models,
  conf_level = 0.95,
  coef_map = NULL,
  coef_omit = NULL,
  coef_rename = NULL,
  statistic_override = NULL,
  facet = FALSE,
  draw = TRUE,
  background = NULL,
  ...
)
}
\arguments{
\item{models}{a single model object or a (potentially named) list of models
to summarize}

\item{conf_level}{confidence level to use for confidence intervals}

\item{coef_map}{named character vector. Values refer to the variable names
that will appear in the table. Names refer to the original term names stored
in the model object, e.g. c("hp:mpg"="hp X mpg") for an interaction term.
Coefficients that are omitted from this vector will be omitted from the
table. The table will be ordered in the same order as this vector.}

\item{coef_omit}{string regular expression. Omits all matching coefficients
from the table (using `grepl`).}

\item{coef_rename}{named character vector. Values refer to the variable names
that will appear in the table. Names refer to the original term names stored
in the model object, e.g. c("hp:mpg"="hp X mpg") for an interaction term.}

\item{statistic_override}{manually override the uncertainy estimates. This
argument accepts three types of input:
\itemize{
  \item a function or list of functions of length(models) which produce variance-covariance matrices with row and column names equal to the names of your coefficient estimates. For example, `R` supplies the `vcov` function, and the `sandwich` package supplies `vcovHC`, `vcovHAC`, etc.
  \item a list of length(models) variance-covariance matrices with row and column names equal to the names of your coefficient estimates.
  \item a list of length(models) vectors with names equal to the names of your coefficient estimates. Numeric vectors are formatted according to `fmt` and placed in brackets, character vectors printed as given.
}}

\item{facet}{TRUE or FALSE. When the 'models' argument includes several
model objects, TRUE draws terms in separate facets, and FALSE draws terms
side-by-side (dodged).}

\item{draw}{TRUE returns a 'ggplot2' object, FALSE returns the data.frame
used to draw the plot.}

\item{background}{A list of 'ggplot2' geoms to add to the background of the
plot.  This is especially useful to display annotations "behind" the
'geom_pointrange' that 'modelplot' draws.}

\item{...}{all other arguments are passed to the `tidy` and `glance` methods
used to extract estimates from the model. For example, this allows users to
set `exponentiate=TRUE` to exponentiate logistic regression coefficients.}
}
\description{
Plot model coefficients using points or point-ranges
}
\examples{
\dontrun{

library(modelsummary)

# single model
mod <- lm(hp ~ vs + drat, mtcars)
modelplot(mod)

# omit terms with string matches or regexes
modelplot(mod, coef_omit = 'Interc')

# rename, reorder and subset with 'coef_map'
cm <- c('vs' = 'V-shape engine',
  'drat' = 'Rear axle ratio')
modelplot(mod, coef_map = cm)

# several models
models <- list()
models[['Small model']] <- lm(hp ~ vs, mtcars)
models[['Medium model']] <- lm(hp ~ vs + factor(cyl), mtcars)
models[['Large model']] <- lm(hp ~ vs + drat + factor(cyl), mtcars)
modelplot(models)

# customize your plots with 'ggplot2' functions
library(ggplot2)

modelplot(models) +
  scale_color_brewer(type = 'qual') +
  theme_classic()

# pass arguments to 'geom_pointrange' through the ... ellipsis
modelplot(mod, color = 'red', size = 1, fatten = .5)

# add geoms to the background, behind geom_pointrange
b <- list(geom_vline(xintercept = 0, color = 'orange'),
  annotate("rect", alpha = .1,
    xmin = -.5, xmax = .5,
    ymin = -Inf, ymax = Inf),
  geom_point(aes(y = term, x = estimate), alpha = .3,
    size = 10, color = 'red', shape = 'square'))
modelplot(mod, background = b)
}

}
