% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/addExtraMarkers.R
\name{addExtraMarkers}
\alias{addExtraMarkers}
\title{Add extra markers}
\usage{
addExtraMarkers(
  mpcrossMapped,
  newMarkers,
  useOnlyExtraImputationPoints = TRUE,
  reorderRadius = 103,
  maxOffset = 50,
  knownChromosome,
  imputationArgs = NULL,
  onlyStatistics = FALSE,
  orderCrossArgs = list(),
  verbose = TRUE,
  reorder = TRUE
)
}
\arguments{
\item{mpcrossMapped}{An object of class \code{mpcrossMapped} (dataset with a map), which must include imputed IBD genotypes and recombination fraction data.}

\item{newMarkers}{An object of class \code{mpcross} containing the new markers to add.}

\item{useOnlyExtraImputationPoints}{Should we only attempt to add the new marker at points at which imputation data has been generated, which are \emph{not} markers? Currently this must be \code{TRUE}. In future \code{FALSE} may be allowed.}

\item{reorderRadius}{The width of the region (in terms of number of markers) in which to attempt to reorder, after the extra markers are added.}

\item{maxOffset}{The maxOffset parameter for the call to \code{\link{estimateMap}}, which is used to re-estimate the map (locally), after the additional markers are added.}

\item{knownChromosome}{The name of a chromosome, if the extra markers are known to go on a specific chromosome}

\item{imputationArgs}{A list containing additional arguments to \code{\link{imputeFounders}}.}

\item{onlyStatistics}{If this argument is \code{TRUE}, then only the chi-squared test statistic values are computed, and the markers are not added.}

\item{orderCrossArgs}{A list containing additional arguments to \code{\link{orderCross}}.}

\item{verbose}{Should extra logging output be generated?}

\item{reorder}{Should local reordering be performed after the extra markers are added?}
}
\value{
If \code{onlyStatistics} was set to \code{TRUE}, an object of class \code{addExtraMarkersStatistics} containing the test statistic values. If \code{onlyStatistics} was set to \code{FALSE}, a list containing the test statistic values in entry \code{statistics} and in entry \code{object}, a new object containing the input object with the new markers added.
}
\description{
Add extra markers to a map, using a QTL-mapping style approach.
}
\details{
This function uses a QTL-mapping style approach to add extra markers to an existing map. The code uses the imputation data at a collection of points, and the marker alleles for the \emph{first} marker of the extra markers. If the imputed genotype at a point is \emph{independent} from the genotype at the new marker, then the new marker probably should \emph{not} be mapped to that point. If the imputed genotype at a point and the marker allele are \emph{strongly dependent}, then the new marker \emph{should} probably be mapped to that point. Dependence and independence are measured using a chi-squared test stastistic for independence. \emph{All the extra markers} are then mapped to the position where the test statistic is largest. It is recommended that only single markers be added at a time, unless you are extremely confident that all the extra markers should be located at the same position.

Currently the set of points at which the new markers are considered for addition is the set of points at which imputation data is available, \emph{which are not markers}. The intention is that this set of points should be an equally spaced grid of points; this reduced the number of tests that are performed, as generally there are far fewer points in the grid, than there are markers. After the new marker is added, local reordering will need to be performed anyway, making any loss in accuracy by using the grid of points largely irrelevant. In future it may be possible to use the set of all marker positions as the set of points at which tests are performed, by setting \code{useOnlyExtraImputationPoints} to \code{FALSE}. 

Once the extra markers have added, local reordering is optionally performed, depending on argument \code{reordering}. The radius of the region on which reordering is performed, in terms of the number of markers, is \code{reorderRadius}. 

Once the optional reordering step has been performed, the map is recomputed locally, to account for the addition of the extra markers. The argument \code{maxOffset} is passed through to \code{estimateMap}. Finally, the imputation data will be recomputed if \code{imputationArgs} is not \code{NULL}; in that case, \code{imputationArgs} should contain a list of arguments to \code{imputeFounders}. It is recommended that the imputation data be recomputed if further markers are to be added. 

In some cases the user will want to apply a threshold to the maximum value of the test statistics, and only add the marker if the test statistics exceed the threshold. In this case the function should be called twice. For the first call, \code{onlyStatistics} should be set to \code{FALSE}. If the resulting test statistics exceed the threshold, then \code{addExtrMarkers} should be called again with \code{onlyStatistics} set to \code{TRUE}.
}
\examples{
data(simulatedFourParentData)
#Create object that includes the correct map
mapped <- new("mpcrossMapped", simulatedFourParentData, map = simulatedFourParentMap)
#Remove marker number 50. Normally the map is discarded, but we specify to keep it. 
removedMiddle <- subset(mapped, markers = (1:101)[-50], keepMap = TRUE)
#Compute imputation data, at all the markers, and an equally spaced grid of points
removedMiddle <- imputeFounders(removedMiddle, errorProb = 0.02, 
	extraPositions = generateGridPositions(1))
#Estimate recombination fractions
removedMiddle <- estimateRF(removedMiddle)
#Get out the extra marker to add
extraMarker <- subset(simulatedFourParentData, markers = 50)
#Add the extra marker, without doing any local reordering. After the marker is added, 
#	recompute the imputation data, using the same arguments as previously. 
\donttest{withExtra <- addExtraMarkers(mpcrossMapped = removedMiddle, newMarkers = extraMarker, 
	reorder = FALSE, imputationArgs = list(errorProb = 0.02, 
	extraPositions = generateGridPositions(1)))$object}
}
